package com.atlassian.confluence.plugins.team;

import com.atlassian.confluence.api.model.web.WebItemView;
import com.atlassian.confluence.core.DataSourceFactory;
import com.atlassian.confluence.core.PluginDataSourceFactory;
import com.atlassian.confluence.languages.LocaleManager;
import com.atlassian.confluence.mail.notification.NotificationManager;
import com.atlassian.confluence.mail.notification.listeners.NotificationTemplate;
import com.atlassian.confluence.mail.template.PreRenderedMailNotificationQueueItem;
import com.atlassian.confluence.plugin.descriptor.mail.NotificationContext;
import com.atlassian.confluence.plugin.descriptor.mail.NotificationRenderManager;
import com.atlassian.confluence.plugins.createcontent.api.events.SpaceBlueprintHomePageCreateEvent;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.user.UserAccessor;
import com.atlassian.confluence.util.i18n.I18NBean;
import com.atlassian.confluence.util.i18n.I18NBeanFactory;
import com.atlassian.core.task.MultiQueueTaskManager;
import com.atlassian.core.task.Task;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.plugin.web.descriptors.WebItemModuleDescriptor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;

@Component
public class SpaceHomePageEventListener {

    private static final String EMAIL_RESOURCE_KEY = "com.atlassian.confluence.plugins.confluence-email-resources";
    private static final String TEAM_SPACE_EMAIL_RESOURCES = "team-space-email-resources";
    private static final String TEAM_SPACE_EMAIL_COMPLETE_KEY = TeamSpaceConstants.TEAM_SPACE_MODULE_KEY + ":" + TEAM_SPACE_EMAIL_RESOURCES;
    private static final String TEAM_SPACE_EMAIL_SOY_TEMPLATE = "Confluence.Templates.Team.Space.Notifications.teamSpaceEmail.soy";

    private static final String TEAM_SPACE_ICON_KEY = "team-space-icon";
    private static final String TEAM_SPACE_TYPE = "team-space";

    private List<DataSource> iconResources = new ArrayList<>();

    private final UserAccessor userAccessor;
    private final NotificationManager notificationManager;
    private final DataSourceFactory dataSourceFactory;
    private final I18NBeanFactory i18NBeanFactory;
    private final LocaleManager localeManager;
    private final MultiQueueTaskManager taskManager;
    private final NotificationRenderManager notificationRenderManager;
    private final EventPublisher eventPublisher;

    @Autowired
    public SpaceHomePageEventListener(@ComponentImport UserAccessor userAccessor,
                                      @ComponentImport NotificationManager notificationManager,
                                      @ComponentImport DataSourceFactory dataSourceFactory,
                                      @ComponentImport I18NBeanFactory i18NBeanFactory,
                                      @ComponentImport LocaleManager localeManager,
                                      @ComponentImport MultiQueueTaskManager taskManager,
                                      @ComponentImport NotificationRenderManager notificationRenderManager,
                                      @ComponentImport EventPublisher eventPublisher) {
        this.userAccessor = userAccessor;
        this.notificationManager = notificationManager;
        this.dataSourceFactory = dataSourceFactory;
        this.i18NBeanFactory = i18NBeanFactory;
        this.localeManager = localeManager;
        this.taskManager = taskManager;
        this.notificationRenderManager = notificationRenderManager;
        this.eventPublisher = eventPublisher;
    }

    @PostConstruct
    public void initialise() {
        this.eventPublisher.register(this);
    }

    @PreDestroy
    public void teardown() {
        this.eventPublisher.unregister(this);
    }

    @EventListener
    public void onSpaceHomePageCreate(SpaceBlueprintHomePageCreateEvent event) {
        if (!TeamSpaceConstants.TEAM_SPACE_COMPLETE_KEY.equals(event.getSpaceBlueprint().getModuleCompleteKey())) {
            return;
        }
        String members = (String) event.getContext().get("members");
        String description = (String) event.getContext().get("description");

        final Space space = event.getSpace();
        String[] userNames = members.split(",");
        List<ConfluenceUser> users = new ArrayList<>();
        for (String username : userNames) {
            final ConfluenceUser user = userAccessor.getUserByName(username);
            if (user != null) {
                notificationManager.addSpaceNotification(user, space);
                users.add(user);
            }
        }

        ConfluenceUser creator = event.getCreator();

        if (NotificationTemplate.ADG.isEnabled("team.space")) {
            sendNotifications(creator, description, space, users);
        }
    }

    private void sendNotifications(ConfluenceUser creator, String description, Space space, List<ConfluenceUser> users) {
        List<ConfluenceUser> allTeamMembers = new ArrayList<>();
        Map<String, DataHandler> allAvatarHandlers = new HashMap<>();
        Map<ConfluenceUser, String> avatarUserMap = new HashMap<>();

        // Add sender Avatar
        DataHandler avatarHandler = new DataHandler((dataSourceFactory.getAvatar(creator)));
        allAvatarHandlers.put(avatarHandler.getName(), avatarHandler);
        avatarUserMap.put(creator, avatarHandler.getName());

        for (ConfluenceUser user : users) {
            avatarHandler = new DataHandler((dataSourceFactory.getAvatar(user)));
            allAvatarHandlers.put(avatarHandler.getName(), avatarHandler);
            avatarUserMap.put(user, avatarHandler.getName());
            allTeamMembers.add(user);
        }

        DataHandler senderAvatarDataHandler = new DataHandler(dataSourceFactory.getAvatar(creator));

        NotificationContext context = new NotificationContext();
        context.put("spaceName", space.getName());
        context.put("spaceUrl", space.getUrlPath());
        context.put("teamSpaceType", TEAM_SPACE_TYPE);
        context.put("sender", creator);
        context.put("contentSummary", description);
        context.put("avatarCid", senderAvatarDataHandler.getName());

        context.put("spaceKey", space.getKey());
        context.put("homePageUrl", space.getHomePage().getUrlPath());
        notificationRenderManager.attachActionIconImages("email.adg.space.action.links", context);

        populateIcons(context);

        for (ConfluenceUser user : users) {
            if (user.equals(creator)) { // don't send notification to the space creator
                continue;
            }

            List<ConfluenceUser> teamMembers = new ArrayList<>(allTeamMembers);
            teamMembers.remove(user); // exclude email recipient from the list
            context.put("userList", teamMembers);

            // Each user might have its own locale,
            // so we update the context for each email
            context.put("actionLinks", getWebItemLinks("email.adg.space.action.links", context, localeManager.getLocale(user)));

            HashMap<String, DataHandler> teamAvatarHandlers = new HashMap<>(allAvatarHandlers);
            teamAvatarHandlers.remove(avatarUserMap.get(user));
            sendTeamSpaceNotification(creator, user, convertToDataSource(teamAvatarHandlers), context);
        }
    }

    private List<WebItemView> getWebItemLinks(String section, NotificationContext context, Locale locale) {
        requireNonNull(locale, "User's locale is not set");
        I18NBean i18n = i18NBeanFactory.getI18NBean(locale);
        List<WebItemModuleDescriptor> webItems = notificationRenderManager.getDisplayableItems(section, context);
        return webItems.stream()
                .map(webItem -> {
                    String url = webItem.getLink().getRenderedUrl(context.getMap());
                    String i18nKey = webItem.getWebLabel().getKey();
                    String label = i18n.getText(i18nKey);
                    return WebItemView
                            .builder()
                            .setModuleKey(webItem.getKey())
                            .setUrl(url)
                            .setUrlWithoutContextPath(url)
                            .setLabel(label)
                            .build();
                })
                .collect(toList());
    }

    private void sendTeamSpaceNotification(ConfluenceUser creator, ConfluenceUser receivingUser, List<DataSource> teamMemberAvatarHandlers, NotificationContext context) {
        I18NBean i18n = i18NBeanFactory.getI18NBean(localeManager.getLocale(receivingUser));
        String subject = i18n.getText("confluence.blueprints.space.team.email.subject", new String[]{creator.getFullName()});
        taskManager.addTask("mail", createNotificationTask(creator, receivingUser, subject, context, teamMemberAvatarHandlers));
    }

    private Task createNotificationTask(ConfluenceUser creator, ConfluenceUser receiver, String subject,
                                        NotificationContext context, List<DataSource> avatarHandlers) {
        PreRenderedMailNotificationQueueItem.Builder builder = PreRenderedMailNotificationQueueItem.with(receiver, TEAM_SPACE_EMAIL_SOY_TEMPLATE, subject)
                .andSender(creator)
                .andTemplateLocation(TEAM_SPACE_EMAIL_COMPLETE_KEY)
                .andContext(context.getMap())
                .andRelatedBodyParts(avatarHandlers)
                .andRelatedBodyParts(iconResources)
                .andRelatedBodyParts(imagesUsedByChromeTemplate());
        return builder.render();
    }

    private void populateIcons(NotificationContext context) {
        iconResources.add(
                dataSourceFactory.createForPlugin(TeamSpaceConstants.TEAM_SPACE_MODULE_KEY)
                        .get()
                        .getResourceFromModuleByName(TEAM_SPACE_EMAIL_RESOURCES, TEAM_SPACE_ICON_KEY)
                        .get()
        ); // team space icon
        iconResources.addAll(context.getTemplateImageDataSources());
    }

    private List<DataSource> convertToDataSource(Map<String, DataHandler> avatarDataHandlers) {
        return avatarDataHandlers.values()
                .stream()
                .map(DataHandler::getDataSource)
                .collect(toList());
    }

    private Iterable<DataSource> imagesUsedByChromeTemplate() {
        return dataSourceFactory.createForPlugin(EMAIL_RESOURCE_KEY)
                .get()
                .getResourcesFromModules("chrome-template", PluginDataSourceFactory.FilterByType.IMAGE::test)
                .get();
    }
}
