package com.atlassian.confluence.plugins.documentation;

import com.atlassian.confluence.labels.Label;
import com.atlassian.confluence.labels.LabelManager;
import com.atlassian.confluence.labels.Labelable;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.pages.PageManager;
import com.atlassian.confluence.plugins.createcontent.api.events.SpaceBlueprintHomePageCreateEvent;
import com.atlassian.confluence.search.IndexManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.List;

@Component
public class CreateSpaceHomePageEventListener {

    private static final String DOCUMENTATION_SPACE_COMPLETE_KEY = "com.atlassian.confluence.plugins.confluence-space-blueprints:documentation-space-blueprint";
    private final LabelManager labelManager;
    private final PageManager pageManager;
    private final IndexManager indexManager;
    private final EventPublisher eventPublisher;

    @Autowired
    public CreateSpaceHomePageEventListener(@ComponentImport LabelManager labelManager,
                                            @ComponentImport PageManager pageManager,
                                            @ComponentImport IndexManager indexManager,
                                            @ComponentImport EventPublisher eventPublisher) {
        this.labelManager = labelManager;
        this.pageManager = pageManager;
        this.indexManager = indexManager;
        this.eventPublisher = eventPublisher;
    }

    @PostConstruct
    public void initialise() {
        this.eventPublisher.register(this);
    }

    @PreDestroy
    public void teardown() {
        this.eventPublisher.unregister(this);
    }

    @EventListener
    public void onSpaceHomePageCreate(SpaceBlueprintHomePageCreateEvent event) {
        if (!DOCUMENTATION_SPACE_COMPLETE_KEY.equals(event.getSpaceBlueprint().getModuleCompleteKey())) {
            return;
        }

        List<Label> labels = new ArrayList<>();
        labels.add(new Label("featured"));
        labels.add(new Label("documentation-space-sample"));

        // add labels to the pages
        Space space = event.getSpace();
        long homePageId = space.getHomePage().getId();
        List<Page> pages = pageManager.getPages(space, false);
        for (Page page : pages) {
            if (page.getId() != homePageId) {
                for (Label label : labels) {
                    // LabelManager#addLabel(ContentEntityObject, Label) was removed in 7.0 , we need the explicit cast for compatibility
                    labelManager.addLabel((Labelable) page, label);
                }
            }
        }

        indexManager.flushQueue(IndexManager.IndexQueueFlushMode.ENTIRE_QUEUE);
    }
}
