package com.atlassian.confluence.plugins.jirareports;

import com.atlassian.applinks.api.ApplicationId;
import com.atlassian.applinks.api.ApplicationLinkService;
import com.atlassian.applinks.api.ReadOnlyApplicationLink;
import com.atlassian.confluence.extra.jira.JiraIssuesManager;
import com.atlassian.confluence.plugins.SoftwareBlueprintsContextProviderHelper;
import com.atlassian.confluence.plugins.common.event.SoftwareBPAnalyticEventUtils;
import com.atlassian.confluence.plugins.createcontent.api.contextproviders.AbstractBlueprintContextProvider;
import com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContext;
import com.google.common.collect.Maps;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URLEncoder;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;

public class ChangeLogAdvanceDialogContextProvider extends AbstractBlueprintContextProvider {

    private static final Logger LOGGER = LoggerFactory.getLogger(ChangeLogAdvanceDialogContextProvider.class);

    private static final String SOY_JIRAISSUES_MACRO_TEMPLATE = "Confluence.Blueprints.JiraReports.Template.jiraissues.soy";
    private static final String DYNAMIC_JIRA_REPORT_TYPE = "dynamic";
    private static final int MAX_RESULT = 1000;

    private SoftwareBlueprintsContextProviderHelper helper;
    private ApplicationLinkService appLinkService;
    private JiraIssuesHelper jiraIssuesHelper;

    public ChangeLogAdvanceDialogContextProvider(SoftwareBlueprintsContextProviderHelper helper, ApplicationLinkService applicationLinkService, JiraIssuesHelper jiraIssuesHelper) {
        this.helper = helper;
        this.appLinkService = applicationLinkService;
        this.jiraIssuesHelper = jiraIssuesHelper;
    }

    @Override
    protected BlueprintContext updateBlueprintContext(BlueprintContext context) {
        context.setTitle((String) context.get("title"));

        SimpleDateFormat simpleDateFormat = new SimpleDateFormat(helper.getDateFormat(), helper.getAuthenticatedUserLocale());
        context.put("date", simpleDateFormat.format(new Date()));
        addJiraMacroToContextMap(context);
        doAnalytic(context);
        return context;
    }

    private void addJiraMacroToContextMap(BlueprintContext contextMap) {
        if (DYNAMIC_JIRA_REPORT_TYPE.equals(contextMap.get("jira-report-type"))) {
            renderDynamicJiraIssues(contextMap);
        } else {
            renderSnapshotJiraIssues(contextMap);
        }
    }

    private void renderSnapshotJiraIssues(BlueprintContext contextMap) {
        String appId = (String) contextMap.get("jira-server-id");
        if (StringUtils.isNotBlank(appId)) {
            try {
                ReadOnlyApplicationLink appLink = appLinkService.getApplicationLink(new ApplicationId(appId));
                String jqlSearch = URLEncoder.encode((String) contextMap.get("jira-query"), "UTF-8");
                JiraIssuesManager.Channel channel = jiraIssuesHelper.getChannel(appLink, jqlSearch, MAX_RESULT);
                contextMap.put("jiraissuesmacro", jiraIssuesHelper.renderJiraIssues(channel));
                contextMap.put("jiraissuescountmacro", jiraIssuesHelper.getTotalIssueNumber(channel) + " "
                        + helper.getText("jirareports.changelog.blueprint.total.issues.title"));
            } catch (Exception e) {
                LOGGER.error("Can not render jira issues", e);
                contextMap.put("jiraissuesmacro", helper.renderTimeout());
            }
        }
    }

    private void renderDynamicJiraIssues(BlueprintContext contextMap) {
        String serverId = (String) contextMap.get("jira-server-id");
        String serverName = (String) contextMap.get("jira-server-name");
        String jqlSearch = (String) contextMap.get("jira-query");
        if (StringUtils.isNotBlank(serverId) && StringUtils.isNotBlank(serverName) && StringUtils.isNotBlank(jqlSearch)) {
            String keyJQL = "jqlQuery";
            String valJQL = jqlSearch;
            //if JQL is key
            String[] jqlItems = jqlSearch.split("=");
            if (jqlItems.length > 1 && jqlItems[0].trim().equals("key")) {
                keyJQL = "key";
                valJQL = jqlItems[1].trim();
            }

            contextMap.put("jiraissuesmacro", getJiraIssuesMacro(keyJQL, valJQL, serverId, serverName, false));
            contextMap.put("jiraissuescountmacro", getJiraIssuesMacro("jqlQuery", jqlSearch, serverId, serverName, true));
        }
    }

    private String getJiraIssuesMacro(String keyJQL, String valJQL, String serverId, String serverName, boolean isCount) {
        HashMap<String, Object> jiraIssuesMacroContext = Maps.newHashMap();
        jiraIssuesMacroContext.put("serverId", serverId);
        jiraIssuesMacroContext.put("server", serverName);
        jiraIssuesMacroContext.put("keyJQL", keyJQL);
        jiraIssuesMacroContext.put("valJQL", valJQL);
        jiraIssuesMacroContext.put("isCount", isCount);
        return helper.renderFromSoy(SoftwareBlueprintsContextProviderHelper.PLUGIN_KEY, SOY_JIRAISSUES_MACRO_TEMPLATE, jiraIssuesMacroContext);
    }

    private void doAnalytic(BlueprintContext contextMap) {
        helper.publishAnalyticEvent(SoftwareBPAnalyticEventUtils.CHANGELOG_CREATE_EVENT_NAME);
        if (DYNAMIC_JIRA_REPORT_TYPE.equals(contextMap.get("jira-report-type"))) {
            helper.publishAnalyticEvent(SoftwareBPAnalyticEventUtils.CHANGELOG_CREATE_DYNAMIC_EVENT_NAME);
        } else {
            helper.publishAnalyticEvent(SoftwareBPAnalyticEventUtils.CHANGELOG_CREATE_STATIC_EVENT_NAME);
        }
    }

}
