define(
    "cp/component/annotation/comment-view",
    [
        "backbone",
        "underscore",
        "jquery",
        "ajs",
        "cp/component/annotation/comment",
        "cp/component/annotation/reply-view",
        "MediaViewer",
        "cp/component/annotation/likes-view",
        "cp/component/utils/editor-view",
        "core/template-store-singleton"
    ],
    function(
        Backbone,
        _,
        $,
        AJS,
        Comment,
        ReplyView,
        FileViewer,
        LikesView,
        EditorView,
        templateStore
    ) {

        "use strict";
        var CommentView = Backbone.View.extend({
            model: Comment,

            tagName: "div",

            className: "cp-comment-view cp-editor",

            events: {
                "click input": "convertToEditor",
                "click #cp-top-level-annotation .cp-annotation-edit": "editComment",
                "click .cp-annotation-resolve": "resolve"
            },

            initialize: function(options) {
                this.listenTo(this.model.replies, "reset add", this.showReplies);
                this.listenTo(this.model, "change", this.render);
                this.listenTo(this.model, "likesReady", this.showLikes);
                this.currentUser = {
                    name: AJS.Meta.get("current-user-fullname"),
                    avatar: AJS.Meta.get("current-user-avatar-uri-reference"),
                    profile: AJS.contextPath() + "/display/~" + AJS.Meta.get("remote-user")
                };
                this._annotationView = options.annotationView;
            },

            renderComment: function() {
                return templateStore.get('Annotation.topLevelAnnotation')({
                    id: this.model.get("id"),
                    author: this.model.get("author"),
                    comment: this.model.get("comment"),
                    canEdit: this.model.get("hasEditPermission"),
                    canResolve: this.model.get("hasResolvePermission"),
                    resolved: this.model.get("resolved"),
                    date: this.model.get("date")
                });
            },

            renderTopLevelAnnotation: function () {
                this.$el.find("#cp-top-level-annotation").replaceWith(this.renderComment());
                this.showLikes();
            },

            render: function() {
                if (!this.model.get("id")) {
                    return this;
                }

                if (this._annotationView.isCommentVisible(this.model)) {
                    this.$el.html(this.renderComment());

                    $(templateStore.get('Annotation.annotationReplyInput')({
                        author: this.currentUser,
                        canReply: this.model.get("hasReplyPermission"),
                        resolved: this.model.get("resolved")
                    })).appendTo(this.$el);

                    this.showReplies();
                    this.showLikes();
                } else {
                    if (this.model.get('resolved')) {
                        this.$el.html(templateStore.get('Annotation.resolved')());
                    }
                }

                return this;
            },

            editComment: function(e) {
                if(!this._annotationView.canOpenNewEditor()) {
                    return;
                }

                var content = this.model.get('editorFormat');
                this.editorView = this._annotationView._editorView = new EditorView({
                    editorSetup: function() {
                        var $loader = $(templateStore.get('Annotation.editorLoader')());
                        this.$el.find("#cp-annotation-main-comment").replaceWith($loader);
                        this.$el.find("#cp-top-level-annotation .cp-annotation-actions").hide();
                    }.bind(this),
                    container: this.$el.find('#cp-top-level-annotation'),
                    form: this.$el.find('form.aui'),
                    saveHandler: _.bind(this.saveEdit, this),
                    cancelHandler: _.bind(this.cancelHandler, this),
                    content: content,
                    postDeactivate: _.bind(this.restoreCommentOnly, this),
                    errorCallback: _.bind(this._handleEditorError, this),
                    restoreCallback: _.bind(this.completedHandler, this),
                    mediaViewer: this._annotationView._mediaViewer
                });
                this.editorView.render();
            },

            saveEdit: function(e) {
                e && e.preventDefault();

                if(this.editorView.getContent() === "") {
                    this._generateError(AJS.I18n.getText('cp.annotations.save.empty'));
                    return;
                }

                this.model.set({editorFormat: this.editorView.getContent()}, {silent: true});
                this.model.save(null, {
                    wait: true,
                    success: this.restoreActions.bind(this),
                    error: this._handleError.bind(this)
                });
                this.editorView && this.editorView.remove();
            },

            cancelHandler: function(e) {
                this.completedHandler(e);
                // HACK: Trigger event that helps JIRA Issue Macro will handle asynchronous when edit comment
                AJS.trigger("ic-jim-async-supported");
            },

            completedHandler: function(e) {
                this.restoreActions();
                e.preventDefault();
            },

            restoreActions: function() {
                this.$el.find("#cp-top-level-annotation .cp-annotation-actions").show();
                this.editorView && this.editorView.remove();
            },

            restoreCommentOnly: function(e) {
                this.$el.find("#cp-top-level-annotation .cp-annotation-actions").show();
                this.renderTopLevelAnnotation();
            },

            showLikes: function() {
                var likes = this.model.likes;
                var likesView = new LikesView({
                    el: this.$el.find(".cp-annotation-comment-like"),
                    id: this.model.get('id'),
                    collection: likes,
                    _annotationView: this._annotationView
                });
                likesView.render();
            },
            
            convertToInputBox: function(e) {
                e && e.preventDefault();
                var $input = $(templateStore.get('Annotation.replyInputBox')());
                this.$el.find("form.aui").replaceWith($input);
                this.editorView && this.editorView.remove();
            },

            restoreAfterEditorDeactivate: function(e) {
                var $input = $(templateStore.get('Annotation.replyInputBox')());
                this.$el.find("form.aui.fadeIn").replaceWith($input);
            },

            convertToEditor: function() {
                if(!this._annotationView.canOpenNewEditor()) {
                    return;
                }

                this.editorView = this._annotationView._editorView = new EditorView({
                    editorSetup: function() {
                        var $loader = $(templateStore.get('Annotation.editorLoader')());
                        this.$el.find("input").replaceWith($loader);
                    }.bind(this),
                    container: this.$el,
                    saveHandler: _.bind(this.saveReply, this),
                    cancelHandler: _.bind(this.convertToInputBox, this),
                    content: "",
                    postDeactivate: _.bind(this.restoreAfterEditorDeactivate, this),
                    errorCallback: _.bind(this._handleEditorError, this),
                    restoreCallback: function() {
                        $('input').blur();
                    },
                    mediaViewer: this._annotationView._mediaViewer
                });
                this.editorView.render();
            },

            resolve: function() {
                if (this.model.get("resolved")) {
                    AJS.trigger('analyticsEvent', {
                        name: 'confluence-spaces.previews.annotation.unresolve',
                        data: {
                            fileType: this._annotationView._mediaViewer.getCurrentFile().get("type"),
                            commentId: this.model.get("id"),
                            replyCount: this.model.replies.length
                        }
                    });
                    this.model.setResolved(false);
                } else {
                    if(!this._annotationView.canOpenNewEditor()) {
                        return;
                    }

                    AJS.trigger('analyticsEvent', {
                        name: 'confluence-spaces.previews.annotation.resolve',
                        data: {
                            fileType: this._annotationView._mediaViewer.getCurrentFile().get("type"),
                            commentId: this.model.get("id"),
                            replyCount: this.model.replies.length
                        }
                    });
                    this.model.setResolved(true);
                    this._annotationView._closeEditor();
                }
            },

            saveReply: function(e) {
                e && e.preventDefault();

                var replies = this.model.replies;

                if(this.editorView.getContent() === "") {
                    this._generateError(AJS.I18n.getText('cp.annotations.save.empty'));
                    return;
                }
                this._annotationView.clearErrorFlags();
                replies.addReply({
                    author: this.currentUser,
                    editorFormat: this.editorView.getContent(),
                    parentId: this.model.get("id")
                }, { error: this._handleError.bind(this) });
                AJS.trigger('analyticsEvent', {
                    name: 'confluence-spaces.previews.annotation.reply',
                    data: {
                        commentId: this.model.get('id')
                    }
                });
                this.convertToInputBox();
            },

            showReplies: function() {
                this.$el.find(".cp-annotation-reply").remove();

                var replies = this.model.replies,
                    that = this;

                _.each(replies.models, function(reply) {
                    var replyView = new ReplyView({
                        model: reply,
                        annotationView: that._annotationView,
                        commentView: that
                    });
                    that.$el.find("#cp-annotation-reply-input").before(replyView.render().el);
                });
            },

            _handleError: function() {
                this._generateError(AJS.I18n.getText('cp.annotations.save.fail'));
            },

            _handleEditorError: function() {
                this._generateError(AJS.I18n.getText('cp.annotations.editor.open.fail'));
            },

            _generateError: function(error) {
                this._annotationView._generateError(error);
            },

            _isResolved: function() {
                return this.model.get("resolved");
            }
        });

        return CommentView;
    });

