/**
 * @tainted: AJS.Confluence.FileTypesUtils
 */
define("cp/component/upload/upload-new-version-dialog-view", [
    "jquery",
    "backbone",
    "underscore",
    "ajs",
    "MediaViewer",
    "core/template-store-singleton"
],
function (
    $,
    Backbone,
    _,
    AJS,
    MediaViewer,
    templateStore
) {

    "use strict";
    var File = MediaViewer.require("core/file");
    var PERCENTAGE_COMPLETED = 1;

    var UploadNewVersionDialog = Backbone.View.extend({

        id: "upload-new-version-dialog",
        tagName: "section",
        className: "aui-layer aui-dialog2 aui-dialog2-small",

        events: {
            "click .close-button": "_closeDialog",
            "click .cancel-button": "_cancelDialog"
        },

        initialize: function (options) {
            this._mediaViewer = options.mediaViewer;
            this.uploader = options.uploader;

            if (this.uploader) {
                this.listenTo(this.uploader, "cp.uploader.filesAdded", this._handleFileAdded);
                this.listenTo(this.uploader, "cp.uploader.uploadProgress", this._handleUploadProgress);
                this.listenTo(this.uploader, "cp.uploader.fileUploaded", this._handleFileUploaded);
                this.listenTo(this.uploader, "cp.uploader.error", this._handleError);
            }

            this.uploadService = options.uploadService;

            this.$el.attr({"role": "dialog", "aria-hidden": "true",
                "data-aui-remove-on-hide": "true", "data-aui-modal": "true",
                "data-fv-allow-focus": "true"
            });
        },

        render: function () {
            this.$el.html(templateStore.get("dialogUploadNewVersion")());
            this._updateElements();

            // CONFDEV-31041: Need to store inline overflow css value for body to restore when close
            this.bodyInlineCSSOverflowValue = $("body")[0].style.overflow;

            this.dialog = AJS.dialog2(this.$el).show();

            this._mediaViewer.getView().lockNavigationKeys();

            return this;
        },

        isVisible: function () {
            return this.$el.is(":visible");
        },

        _updateElements: function() {
            this.$header = this.$(".aui-dialog2-header-main");
            this.$progressBar = this.$(".aui-progress-indicator");
            this.$cancelButton = this.$(".cancel-button");
            this.$closeButton = this.$(".close-button");
            this.$fileNameDiv = this.$(".file-name");
            this.$fileNameText = this.$(".file-name-text");
            this.$icon = this.$(".aui-icon");

            this.$msg = this.$("#upload-new-version-error-msg");
            this.$comment = this.$("#version-comment");
            this.$spinner = this.$(".spinner");
        },

        _toggleCancelButton: function (isShowCancelButton) {
            this.$cancelButton.toggleClass("hidden", !isShowCancelButton);
            this.$closeButton.toggleClass("hidden", isShowCancelButton);
        },

        /**
         * Enable or disable button state
         * @param $button jQuery object
         * @param isEnabled {boolean} true means enable button state, false means disable button state
         * @private
         */
        _toggleEnableButton: function ($button, isEnabled) {
            $button.attr("aria-disabled", !isEnabled);
            $button.prop("disabled", !isEnabled);
        },

        _closeDialog: function () {
            // when close dialog, add version change comment
            var versionComment = this.$comment.val();

            if (versionComment) {
                if (versionComment.length <= 255) {

                    this._toggleEnableButton(this.$closeButton, false);
                    this.$spinner.spin();

                    this.uploadService.addVersionChangeComment(
                        this.$comment.val(),
                        this._mediaViewer.getCurrentFile(),
                        _.bind(this._kill, this),
                        _.bind(this._handleError, this)
                    );

                    AJS.trigger("analyticsEvent", {
                        name: "confluence-spaces.previews.upload.submit-comment"
                    });
                } else {
                    var message = AJS.I18n.getText("cp.validation.comment.error");
                    this._handleLongCommentError(message);
                }
            } else {
                this._kill();
            }
        },

        _cancelDialog: function () {
            this.uploader.cancelUpload();
            this._kill();

            AJS.trigger("analyticsEvent", {
                name: "confluence-spaces.previews.upload.cancel"
            });
        },

        _kill: function () {
            if (this.uploader) {
                this.stopListening(this.uploader);
            }
            this.dialog.hide();
            // CONFDEV-31041: Need to restore inline overflow css value for body
            $("body")[0].style.overflow = this.bodyInlineCSSOverflowValue;
            this._mediaViewer.getView().unlockNavigationKeys();
        },

        _handleFileAdded: function (file) {
            this.$fileNameText.text(file.name);
            this._showIcon(file);
            AJS.progressBars && AJS.progressBars.setIndeterminate(this.$progressBar);

            var currentFile = this._mediaViewer.getCurrentFile();
            var titleCurrentFile = currentFile ? currentFile.get("title") : "";

            AJS.trigger("analyticsEvent", {
                name: "confluence-spaces.previews.upload.start",
                data: {uploadSameName: titleCurrentFile === file.name}
            });
        },

        _handleUploadProgress: function (percentage) {
            if (!percentage) {
                return;
            }

            if (percentage === PERCENTAGE_COMPLETED) {
                this._toggleEnableButton(this.$cancelButton, false);
            }

            AJS.progressBars && AJS.progressBars.update(this.$progressBar, percentage);
        },

        _handleFileUploaded: function (clientFile, response) {

            this.uploadService.promiseFileModel(response).done(_.bind(function (newFile) {
                this._toggleCancelButton(false);
                this.$header.text(AJS.I18n.getText("cp.upload.new.version.successful"));

                this._mediaViewer._fileState.replaceCurrent(new File(newFile));
                this._mediaViewer.showFile(this._mediaViewer._fileState.getCurrent());

                this._mediaViewer._fileState.set("isNewFileUploaded", true);
            }, this));
        },

        _handleError: function (message) {
            this.$msg.empty();

            AJS.messages.warning("#upload-new-version-error-msg", {
                body: templateStore.get("uploadErrorMessage")({message: message}),
                closeable: false
            });

            this.$msg.removeClass("hidden");
            this.$fileNameDiv.hide();
            this.$progressBar.hide();
            this.$comment.val("");
            this.$comment.hide();

            // hide Cancel button, show Done button and make sure Done button is enabled.
            this._toggleCancelButton(false);
            this._toggleEnableButton(this.$closeButton, true);

            this.$spinner.spinStop();

            this.$header.text(AJS.I18n.getText("cp.upload.new.version.failed"));

            AJS.trigger("analyticsEvent", {
                name: "confluence-spaces.previews.upload.failed"
            });
        },

        _handleLongCommentError: function (message) {
            this.$msg.empty();

            AJS.messages.warning("#upload-new-version-error-msg", {
                body: templateStore.get("uploadErrorMessage")({message: message}),
                closeable: true
            });

            this.$msg.removeClass("hidden");
        },

        // Show icon based on MIME type.
        _showIcon: function (file) {
            var mimeType = file.nativeFile && file.nativeFile.type;
            // default one is "aui-iconfont-page-blank".
            var auiIconClassName = AJS.Confluence.FileTypesUtils.getAUIIconFromMime(mimeType);
            this.$icon.addClass(auiIconClassName);
        }
    });

    return UploadNewVersionDialog;
});