define('confluence/page-hierarchy/dialog/copy-dialog', [
    'ajs',
    'confluence/legacy',
    'confluence/page-hierarchy/fetch/fetch-homepage',
    'confluence/page-hierarchy/fetch/fetch-space-permissions',
    'confluence/page-hierarchy/fetch/fetch-children',
    'confluence/page-hierarchy/fetch/fetch-page-info',
    'confluence/page-hierarchy/service/dialog-service',
    'confluence/page-hierarchy/state/copy-state',
    'confluence/page-hierarchy/util/analytics-event'
], function (AJS,
             Confluence,
             fetchHomepage,
             fetchSpacePermissions,
             fetchChildren,
             fetchPageInfo,
             DialogService,
             state,
             analyticsEvent) {
    var NAME = 'copy-page-dialog';
    var COPY_DIALOG_NEXT_BUTTON_SELECTOR = '#copy-dialog-next';
    var COPY_DIALOG_CLOSE_BUTTON_SELECTOR = '#copy-dialog-close';
    var INCLUDE_CHILDREN_SELECTOR = '#include-children';
    var SELECTION_ERROR_SELECTOR = '#selection-error';
    var TEMPLATE_NAME = 'copyDialog';
    var $ = AJS.$;

    var dialog = DialogService.get(NAME, {
        templateName: TEMPLATE_NAME,
        templateParameters: {},
        onShow: _bindValues,
        onInit: _init
    });

    /**
     * Do initial setup actions when the dialog is first loaded. Run once.
     * @returns {undefined}
     * @private
     */
    function _init() {
        _setupAutoCompletes();
        _bindEvents();
        _bindButtons();
    }

    /**
     * Bind events to the control buttons for the dialog
     * @returns {undefined}
     * @private
     */
    function _bindButtons() {
        var $copyPageDialogNext = dialog.$element.find(COPY_DIALOG_NEXT_BUTTON_SELECTOR);
        var $copyDialogClose = dialog.$element.find(COPY_DIALOG_CLOSE_BUTTON_SELECTOR);

        // Next button binding
        $copyPageDialogNext.click(function () {
            if (state.getState().includeChildren) {
                AJS.trigger(state.COPY_HIERARCHY_EVENT);
            }
            else {
                AJS.trigger(state.COPY_PAGE_EVENT);
            }
        });

        // Close button binding
        $copyDialogClose.click(function (e) {
            e.preventDefault();
            dialog.hide();
        });
    }

    /**
     * Set up the autocomplete fields for the space and page selectors.
     * @returns {undefined}
     * @see _bindFields for regular text and checkbox fields
     * @private
     */
    function _setupAutoCompletes() {
        var $destinationSpaceInput = dialog.$element.find("#copy-destination-space");
        var $destinationPageInput = dialog.$element.find("#copy-destination-page");
        var currentSpaceName = AJS.Meta.get("space-name");
        var currentSpaceKey = AJS.Meta.get("space-key");
        // If we get page information for parent page. We should not care about previous version
        // Previous version can only be gotten for current page
        var parentPageId = AJS.Meta.get('parent-page-id') || AJS.Meta.get('page-id');
        var latestPageId = AJS.Meta.get('parent-page-id') || AJS.Meta.get('latest-page-id');

        state.setDestinationSpaceKey(currentSpaceKey);

        _ensureSelections();

        dialog.async(_checkSpacePermissions(currentSpaceKey), fetchPageInfo(parentPageId, latestPageId)).done(function (hasPermission, parentPage) {
            var parentPageId = parentPage.id;
            var parentPageTitle = parentPage.title;
            var parentPageUrl = parentPage._links.webui;

            if (hasPermission) {
                __setDestinationPage(parentPageId, parentPageTitle, parentPageUrl);
            } else {
                $destinationPageInput.prop('disabled', true);
                analyticsEvent.publish(analyticsEvent.COPY_EXISTING_SPACE_NO_PERMISSION_ANALYTICS_EVENT);
            }

            // Defaults
            $destinationPageInput.attr("data-spacekey", currentSpaceKey);

            // Autocompletes
            Confluence.Binder.autocompleteSpace(dialog.$element);
            Confluence.Binder.autocompletePage(dialog.$element);

            $destinationSpaceInput.val(currentSpaceName);
            $destinationSpaceInput.data('title', currentSpaceName);

            $destinationSpaceInput.focus(function () {
                $(this).select();
            });

            $destinationPageInput.focus(function () {
                $(this).select();
            });

            $destinationSpaceInput.keyup(function () {
                // Clear our selections if the title has changed but the user hasn't selected a valid space yet
                if (_checkShouldTriggerReselection($destinationSpaceInput)) {
                    $(SELECTION_ERROR_SELECTOR).empty();
                    $destinationPageInput.val('');
                    state.setDestinationPageId('');
                    state.setDestinationPageTitle('');
                    _ensureSelections();
                }
            });
            $destinationPageInput.keyup(function () {
                // Clear our selections if the title has changed but the user hasn't selected a valid page yet
                if (_checkShouldTriggerReselection($destinationPageInput)) {
                    $(SELECTION_ERROR_SELECTOR).empty();
                    state.setDestinationPageId('');
                    state.setDestinationPageTitle('');
                    _ensureSelections();
                }
            });

            _checkTitle($destinationSpaceInput);
            _checkTitle($destinationPageInput);

            // Space Bindings
            $destinationSpaceInput.on("selected.autocomplete-content", function (e, data) {
                if (data && data.content) {
                    analyticsEvent.publish(analyticsEvent.COPY_NEW_SPACE_ANALYTICS_EVENT);
                    var spaceKey = data.content.key;
                    $destinationSpaceInput.data('title', data.content.title);
                    state.setDestinationSpaceKey(spaceKey);
                    // set this attribute so that the page search will be constrained to this space
                    $destinationPageInput.attr("data-spacekey", spaceKey);
                    $destinationPageInput.prop('disabled', true);

                    dialog.async(_checkSpacePermissions(spaceKey), fetchHomepage(spaceKey)).done(function (hasPermission, homepage) {
                        if (!hasPermission) {
                            analyticsEvent.publish(analyticsEvent.COPY_NEW_SPACE_NO_PERMISSION_ANALYTICS_EVENT);
                            return;
                        }

                        $destinationPageInput.prop('disabled', false);

                        if (!homepage) {
                            AJS.messages.warning("#selection-error", {
                                title: AJS.I18n.getText('copy.page.hierarchy.dialog.destination.homepage.error.title'),
                                body: '<p>' + AJS.I18n.getText('copy.page.hierarchy.dialog.destination.homepage.error.description') + '</p>'
                            });
                        } else {
                            __setDestinationPage(homepage.id, homepage.title, homepage._links.webui);
                        }
                    });
                }
            });

            // Page Bindings
            $destinationPageInput.on("selected.autocomplete-content", function (e, data) {
                if (data && data.content) {
                    analyticsEvent.publish(analyticsEvent.COPY_NEW_PAGE_ANALYTICS_EVENT);
                    $destinationPageInput.data('title', data.content.title);
                    dialog.async(fetchPageInfo(data.content.id)).done(function (page) {
                        __setDestinationPage(page.id, page.title, page._links.webui);
                    });

                }
            });
        });

        /**
         * Set the destination page from the pageId and title and do any required UI work
         * @param {number} pageId Page Id
         * @param {string} pageTitle Page Title
         * @param {string} pageUrl Page Url
         * @returns {undefined}
         * @private
         */
        function __setDestinationPage(pageId, pageTitle, pageUrl) {
            state.setDestinationPageId(pageId);
            state.setDestinationPageTitle(pageTitle);
            state.setDestinationPageUrl(AJS.contextPath() + pageUrl);

            $destinationPageInput.val(pageTitle);
            $destinationPageInput.data('title', pageTitle);
            $destinationPageInput.focus();

            _ensureSelections();
        }
    }

    /**
     * Check if the user has permission to add pages in their selected space. If not, show an error.
     * @param {string} spaceKey Space Key to check
     * @returns {jQuery.Deferred} A jQuery promise with value true if user has permission, false otherwise.
     * @private
     */
    function _checkSpacePermissions(spaceKey) {
        var deferred = $.Deferred();
        dialog.async(fetchSpacePermissions(spaceKey)).done(function (permissions) {
            state.setPermissions(permissions);

            if (!permissions.pages) {
                AJS.messages.warning("#selection-error", {
                    body: '<p>' + AJS.I18n.getText('copy.page.hierarchy.validation.noCreatePagePermission') + '</p>'
                });

                deferred.resolve(false);
            } else {
                deferred.resolve(true);
            }
        });

        return deferred.promise();
    }

    /**
     * Bind a change event on the input to check if it has a proper space/page selected. It checks
     * the data.title of the element to make sure it is properly set.
     * @param {jQuery} $input The input to check
     * @returns {undefined}
     * @private
     */
    function _checkTitle($input) {
        $input.blur(function () {
            if (!$input.data('title')) {
                $input.val('');
            }
        });
    }

    /**
     * Check if the page/space input value is the same as the value selected from the autocomplete.
     * If it is then we shouldn't trigger the user to reselect it. If it's not, then they need to
     * select a valid option from the dropdown
     * @param {jQuery} $input The input element to check.
     * @returns {boolean} true iff the user needs to select a new valid option.
     * @private
     */
    function _checkShouldTriggerReselection($input) {
        if (!$input.data('title') || $input.val() !== $input.data('title')) {
            $input.removeData('title');
            return true;
        }

        return false;
    }

    /**
     * Ensures that the user has chosen a proper destination id
     * @returns {undefined}
     * @private
     */
    function _ensureSelections() {
        var $copyPageDialogNext = dialog.$element.find(COPY_DIALOG_NEXT_BUTTON_SELECTOR);
        var destinationPageId = state.getOptions().destinationPageId;
        var title = destinationPageId ? '' : AJS.I18n.getText('copy.page.hierarchy.dialog.destination.parentPage.error.not.selected');
        $copyPageDialogNext.prop('disabled', !destinationPageId);
        $copyPageDialogNext.prop('title', title);
    }

    /**
     * Binds any existing values to fields
     * @returns {undefined}
     * @private
     */
    function _bindValues() {
        _bindFields(true);
    }

    /**
     * Binds any events to sync field data. Should be called once.
     * @returns {undefined}
     * @private
     */
    function _bindEvents() {
        _bindFields(false);
    }

    /**
     * Either writes values or binds events to fields
     * @param {boolean} writeValues If true, this method writes values for the fields,
     *                              otherwise it binds its events.
     * @returns {undefined}
     * @private
     */
    function _bindFields(writeValues) {
        var $includeChildren = dialog.$element.find(INCLUDE_CHILDREN_SELECTOR);
        var pageId = AJS.Meta.get('page-id');

        if (writeValues) {
            $includeChildren.prop('checked', state.getState().includeChildren);
            _setNextButtonText();
        } else {
            dialog.async(fetchChildren(pageId)).done(function (data) {
                var children = data.results;
                if (children.length) {
                    $includeChildren.prop('disabled', false);
                    $includeChildren.change(function () {
                        state.setIncludeChildren($includeChildren.prop('checked'));
                        _setNextButtonText();
                    });
                } else {
                    $includeChildren.prop('disabled', true);
                    $includeChildren.closest('.field-group').prop('title', AJS.I18n.getText('copy.page.hierarchy.dialog.noChildPages'));
                }
                _ensureSelections();
            });

        }
    }

    /**
     * Set the text of the next button
     * @returns {undefined}
     * @private
     */
    function _setNextButtonText() {
        var $copyPageDialogNext = dialog.$element.find(COPY_DIALOG_NEXT_BUTTON_SELECTOR);
        if (state.getState().includeChildren) {
            $copyPageDialogNext.html(AJS.I18n.getText('copy.page.dialog.next.button'));
        } else {
            $copyPageDialogNext.html(AJS.I18n.getText('copy.page.hierarchy.dialog.submit.button'));
        }
    }

    return dialog;
});