define('confluence/ob/modules/intro/intro-find-spaces-sequence', [
    'jquery',
    'underscore',
    'confluence/ob/common/promise',
    'confluence/ob/modules/intro/intro-sequence-utils',
    'confluence/ob/common/common-core',
    'confluence/ob/common/bacon'
], function ($, _, Promise, IntroSequenceUtils, CommonCore, Bacon) {
    'use strict';

    // TODO start using container parameter in all of the steps

    var spaceClient = {
        getRelevant: function () {return $.get(AJS.contextPath() + '/rest/onboarding/1/space/relevant')},
        search: function (query) {return $.get(AJS.contextPath() + '/rest/onboarding/1/space?query=' + query);},
        followSpaces: function (spaceKeys) {
            return $.ajax({
                url: AJS.contextPath() + '/rest/onboarding/1/space/relevant',
                type: 'PUT',
                data: JSON.stringify(spaceKeys),
                contentType: 'application/json'
            });
        }
    };

    var IntroFindSpacesSequence = function (options) {
        this.options = _.extend({}, options);

        this.getRelevantSpacesPromise = IntroSequenceUtils.hasStep('find-spaces')
                ? spaceClient.getRelevant()
                : $.Deferred().reject('No spaces found. Might be due to all pages being restricted.');
    };

    /**
     * @type SequenceInitFunction
     */
    IntroFindSpacesSequence.prototype.init = function (container, analytics) {
        if (!IntroSequenceUtils.hasStep('find-spaces')) {
            return $.Deferred().reject();
        }

        var that = this;

        return Promise(function (resolve, reject) {
                    // vars
                    var searchInputDebounceWaitMs = that.options.searchInputDebounceWaitMs || 500;
                    var MIN_SPACES_TO_SELECT = 1;
                    var spacesAvailable = AJS.Meta.get('available-spaces-count');
                    var showSearch = !!spacesAvailable && spacesAvailable > 10;
                    var maxSpacesToShow = showSearch ? 6 : 8;
                    var defaultSpaceKeysForAnalytics = [];
                    var followSpacesDeferred = $.Deferred();

                    // jqueries
                    var $contentEl = IntroSequenceUtils.createProgressTracker('find-spaces');
                    var $content = $(Confluence.Templates.OB.findSpaces());

                    var $button = $content.find('.intro-find-spaces-button-continue');
                    var $searchBar = $content.find('#find-spaces-search');
                    var $searchContainer = $content.find('.intro-find-spaces-search');
                    var $searchClearButton = $content.find('.intro-find-spaces-search span.aui-iconfont-close-dialog');
                    var $selectedHeader = $content.find('.selected-header');
                    var $selectedHeaderArrow = $content.find('.selected-header .arrow');
                    var $selectedCount = $content.find('.selected-count');
                    var $selectedContainer = $content.find('.selected-spaces');
                    var $spaceContainer = $content.find('.intro-find-spaces-relevant-spaces');
                    var $submitSpinner = $content.find('.intro-find-spaces-button-spinner');

                    // streams and properties
                    var initialSpacesDataProperty = Bacon.fromPromise(that.getRelevantSpacesPromise).toProperty();

                    var spaceCheckboxToggleStream = $spaceContainer
                            .asEventStream('change', 'input[type="checkbox"]', function () {
                                return {
                                    key: this.name,
                                    selected: $(this).is(':checked')
                                };
                            });

                    var spaceRemovedFromSelectedStream = $selectedContainer
                            .asEventStream('click', '.selected-space', function () {
                                return $(this).attr('data-space-key');
                            })
                            .map(function (spaceKey) {
                                return {
                                    key: spaceKey,
                                    selected: false
                                };
                            });

                    var selectedSpaceKeysProperty = spaceCheckboxToggleStream
                            .merge(spaceRemovedFromSelectedStream)
                            .scan([], function (acc, space) {
                                return space.selected ? acc.concat(space.key) : _.without(acc, space.key);
                            });

                    var selectedCountProperty = selectedSpaceKeysProperty.map(function (spaces) {
                        return spaces.length;
                    });

                    var searchInputStream = $searchBar
                            .asEventStream('input', function () {
                                return $(this).val();
                            });

                    var searchRequestStream = searchInputStream
                            .debounce(searchInputDebounceWaitMs)
                            .filter(isBlank);

                    var searchInputChangedAfterRequestProperty = Bacon
                            .combineWith(_.isEqual, searchInputStream, searchRequestStream)
                            .not();
                    // combine is lazy, needs eager eval
                    searchInputChangedAfterRequestProperty.onValue($.noop);

                    var searchIsEmptyProperty = searchInputStream
                            .map(function (query) { return $.trim(query).length === 0; })
                            .toProperty(true);

                    var searchResponseProperty = searchRequestStream
                            .flatMapLatest(function (query) {
                                return Bacon
                                        .fromPromise(spaceClient.search(query))
                                        .skipWhile(searchInputChangedAfterRequestProperty);
                            });

                    var searchResultsWithSelected = Bacon
                            .combineWith(addSelectedPropertyToSpaceData, searchResponseProperty, selectedSpaceKeysProperty)
                            .sampledBy(searchResponseProperty);

                    var enoughSpacesSelectedProperty = selectedCountProperty
                            .map(function (count) {
                                return count >= MIN_SPACES_TO_SELECT;
                            });
                    var buttonClickedProperty = $button
                            .asEventStream('click', function () { return true; })
                            .toProperty(false);
                    var buttonEnabledProperty = enoughSpacesSelectedProperty
                            .and(buttonClickedProperty.not());

                    var showSelectedContainerStream = $selectedHeader.asEventStream('click', function () {
                        return !$selectedContainer.is(':visible');
                    });

                    var searchDataCacheProperty = searchResponseProperty
                            .scan([], mergeSpaceData);
                    var spaceDataCacheProperty = Bacon
                            .combineWith(mergeSpaceData, initialSpacesDataProperty, searchDataCacheProperty);

                    var selectedSpacesDataProperty = Bacon
                            .combineWith(getSelectedSpacesFromData, spaceDataCacheProperty, selectedSpaceKeysProperty);

                    var submitFormStream = selectedSpaceKeysProperty
                            .sampledBy(buttonClickedProperty)
                            .skip(1);

                    var defaultSpaceDataStream = Bacon
                            .combineWith(addSelectedPropertyToSpaceData, initialSpacesDataProperty, selectedSpaceKeysProperty);

                    var renderDefaultSpacesProperty = searchIsEmptyProperty
                            .skip(1)
                            .flatMap(function (isEmpty) {
                                return isEmpty ? true : Bacon.never();
                            });

                    var renderDefaultDataSpacesStream = defaultSpaceDataStream
                            .sampledBy(renderDefaultSpacesProperty);

                    var searchClearStream = $searchClearButton.asEventStream('click');

                    var searchBarFocusStream = Bacon
                            .mergeAll(spaceCheckboxToggleStream, spaceRemovedFromSelectedStream, searchClearStream)
                            .map(true);

                    // stream handlers
                    buttonEnabledProperty
                            .not()
                            .assign($button, 'prop', 'disabled');

                    enoughSpacesSelectedProperty.onValue(function (isEnoughSpacesSelected) {
                        if (isEnoughSpacesSelected) {
                            $button.text(AJS.I18n.getText('onboarding.sequence.find.spaces.button.finish'));
                        } else {
                            $button.text(AJS.I18n.getText('onboarding.sequence.find.spaces.button.pick'));
                        }
                    });

                    searchClearStream.onValue(function () {
                        $searchBar.val('').trigger('input');
                    });
                    searchBarFocusStream.onValue(function () {
                        $searchBar.focus();
                    });
                    searchIsEmptyProperty.onValue(function (isClear) {
                        if (!isClear) {
                            $spaceContainer.html('').spin();
                        }
                        $searchClearButton.toggle(!isClear);
                    });
                    searchResultsWithSelected.onValue(function (data) {
                        $spaceContainer.spinStop();
                        if (data.length > 0) {
                            renderSpaces($spaceContainer, data);
                        } else {
                            $spaceContainer.text(AJS.I18n.getText('onboarding.sequence.find.spaces.no.results'));
                        }
                    });

                    selectedCountProperty.onValue(function (count) {
                        $selectedCount.text('(' + count + ')');
                    });
                    selectedSpacesDataProperty.onValue(function (selectedSpacesData) {
                        var $selectedSpaces = $(Confluence.Templates.OB.selectedSpaces({
                            spaces: selectedSpacesData
                        }));
                        $selectedContainer.html($selectedSpaces);
                    });

                    showSelectedContainerStream.assign($selectedContainer, 'toggle');
                    showSelectedContainerStream.not().assign($selectedHeaderArrow, 'toggleClass', 'arrow-down');
                    showSelectedContainerStream.assign($selectedHeaderArrow, 'toggleClass', 'arrow-up');

                    spaceRemovedFromSelectedStream.onValue(function (space) {
                        $('div[data-space-key="' + space.key + '"] input').attr('checked', false);
                    });

                    initialSpacesDataProperty
                            .onValue(function (data) {
                                if (!data || data.length === 0) {
                                    reject();
                                } else {
                                    defaultSpaceKeysForAnalytics = _.take(data, maxSpacesToShow).map(function (space) {
                                        return space.key;
                                    });
                                    renderSpaces($spaceContainer, data);
                                }
                            });

                    initialSpacesDataProperty.onError(reject);

                    searchResponseProperty.onError(function () {
                        $spaceContainer.text(AJS.I18n.getText('onboarding.sequence.find.spaces.no.results'));
                    });

                    renderDefaultDataSpacesStream.onValue(function (data) {
                        renderSpaces($spaceContainer, data);
                    });

                    submitFormStream.onValue(function (keys) {
                        $submitSpinner.spin();
                        var numberOfSpacesFollowed = keys.length;
                        var spacesPickedFromSearches = _.difference(keys, defaultSpaceKeysForAnalytics).length;

                        spaceClient.followSpaces(keys)
                                .done(function () {
                                    followSpacesDeferred.resolve({
                                        numberOfSpacesFollowed: numberOfSpacesFollowed,
                                        spacesPickedFromSearches: spacesPickedFromSearches
                                    });
                                })
                                .fail(function (xhr) {
                                    followSpacesDeferred.reject(xhr);
                                })
                                .always(function () {
                                    resolve();
                                });
                    });

                    // functions
                    init();

                    function init() {
                        $contentEl.empty().append($content);
                        $spaceContainer.spin();
                        that.getRelevantSpacesPromise.always(function () {
                            $spaceContainer.spinStop();
                        });

                        $searchContainer
                                .add($selectedHeader)
                                .toggle(showSearch);
                        if (showSearch) {
                            $spaceContainer.css('min-height', '330px'); //fix height so search results don't cause jump
                        }
                        disableEnterOnInputs();
                        addAnalytics(searchResponseProperty, showSearch, showSelectedContainerStream,
                                followSpacesDeferred.promise(), initialSpacesDataProperty);
                    }

                    function disableEnterOnInputs() {
                        var KEYCODE_ENTER = 13;
                        $(document).on("keypress", "input", function (event) {
                            return event.keyCode !== KEYCODE_ENTER;
                        });
                    }

                    function isBlank(str) {
                        return $.trim(str).length > 0;
                    }

                    function renderSpaces($target, spaceData) {
                        var unescapedSpaceData = _.map(spaceData, function (data) {
                            var temp = data;
                            temp.name = decodeEntities(data.name);
                            return temp;
                        });
                        var $defaultResults = $(Confluence.Templates.OB.findSpaceItems({
                            spaces: _.take(unescapedSpaceData, maxSpacesToShow)
                        }));
                        $target.html($defaultResults);
                    }

                    function addSelectedPropertyToSpaceData(spaceData, selectedSpaceKeys) {
                        return _.map(spaceData, function (space) {
                            return _.extend(space, {selected: _.contains(selectedSpaceKeys, space.key)});
                        });
                    }

                    function getSelectedSpacesFromData(spaceData, selectedSpaceKeys) {
                        return _.filter(spaceData, function (space) {
                            return _.contains(selectedSpaceKeys, space.key);
                        });
                    }

                    function mergeSpaceData(data1, data2) {
                        return _.uniq(_.union(data1, data2), false, function (space) { return space.key;});
                    }
                }
        );

        function decodeEntities(encodedString) {
            var textArea = document.createElement('textarea');
            textArea.innerHTML = encodedString;
            return textArea.value;
        }

        function addAnalytics(searchResponseProperty, showSearch, showSelectedContainerStream,
                followSpacesPromise, initialSpacesDataStream) {
            if (showSearch) {
                analytics.pushEvent(IntroFindSpacesSequence.EVENTS.SEARCH_BAR_SHOWN);
            } else {
                analytics.pushEvent(IntroFindSpacesSequence.EVENTS.SEARCH_BAR_HIDDEN);
            }

            searchResponseProperty.onValue(function (val) {
                var searchResultCount = val.length;
                if (searchResultCount < 5) {
                    analytics.pushEvent(IntroFindSpacesSequence.EVENTS.SEARCH_FOUND +
                            '.' + searchResultCount);
                } else {
                    analytics.pushEvent(IntroFindSpacesSequence.EVENTS.SEARCH_FOUND_5_OR_MORE);
                }
            });

            searchResponseProperty.onError(function (xhr) {
                analytics.pushEvent(IntroFindSpacesSequence.EVENTS.ERROR_SEARCH_FAILED, xhr);
            });

            showSelectedContainerStream.onValue(function (show) {
                if (show) {
                    analytics.pushEvent(IntroFindSpacesSequence.EVENTS.SELECTED_OPEN);
                } else {
                    analytics.pushEvent(IntroFindSpacesSequence.EVENTS.SELECTED_CLOSE);
                }
            });

            followSpacesPromise.done(function (data) {
                if (data.numberOfSpacesFollowed < 5) {
                    analytics.pushEvent(IntroFindSpacesSequence.EVENTS.FOLLOWED_SPACES +
                            '.' + data.numberOfSpacesFollowed);
                } else {
                    analytics.pushEvent(IntroFindSpacesSequence.EVENTS.FOLLOWED_SPACES_5_OR_MORE);
                }

                if (showSearch && data.spacesPickedFromSearches < 5) {
                    analytics.pushEvent(IntroFindSpacesSequence.EVENTS.FOLLOWED_SEARCHED_SPACES +
                            '.' + data.spacesPickedFromSearches);
                } else if (showSearch) {
                    analytics.pushEvent(IntroFindSpacesSequence.EVENTS.FOLLOWED_SEARCHED_SPACES_5_OR_MORE);
                }
            }).fail(function (xhr) {
                analytics.pushEvent(IntroFindSpacesSequence.EVENTS.ERROR_SERVER_SAVE_FAILED, xhr);
            });

            initialSpacesDataStream.onValue(function (data) {
                if (!data || data.length === 0) {
                    analytics.pushEvent(IntroFindSpacesSequence.EVENTS.ERROR_LOAD_RELEVANT_SPACES_FAILED_NO_DATA);
                } else {
                    analytics.pushEvent(IntroFindSpacesSequence.EVENTS.RELEVANT_SPACES_LOADED,
                            {spacesLoaded: data.length});
                }
            });

            initialSpacesDataStream.onError(function (xhr) {
                analytics.pushEvent(IntroFindSpacesSequence.EVENTS.ERROR_LOAD_RELEVANT_SPACES_FAILED, xhr);
            });
        }
    };

    IntroFindSpacesSequence.EVENTS = {
        SEARCH_BAR_SHOWN: 'search.shown',
        SEARCH_BAR_HIDDEN: 'search.hidden',
        SEARCH_FOUND: 'search.found',
        SEARCH_FOUND_5_OR_MORE: 'search.found.5.or.more',
        FOLLOWED_SPACES: 'followed.spaces',
        FOLLOWED_SPACES_5_OR_MORE: 'followed.spaces.5.or.more',
        FOLLOWED_SEARCHED_SPACES: 'followed.searched.spaces',
        FOLLOWED_SEARCHED_SPACES_5_OR_MORE: 'followed.searched.spaces.5.or.more',
        RELEVANT_SPACES_LOADED: 'relevant.spaces.loaded',
        SELECTED_OPEN: 'selected.open',
        SELECTED_CLOSE: 'selected.close',
        ERROR_SEARCH_FAILED: 'error.search.failed',
        ERROR_SERVER_SAVE_FAILED: 'error.server.save.failed',
        ERROR_LOAD_RELEVANT_SPACES_FAILED: 'error.load.relevant.spaces.failed',
        ERROR_LOAD_RELEVANT_SPACES_FAILED_NO_DATA: 'error.load.relevant.spaces.failed.no.data'
    };

    return IntroFindSpacesSequence;
});
