package com.atlassian.confluence.plugins.mentions;

import com.atlassian.confluence.content.render.xhtml.XmlEventReaderFactory;
import com.atlassian.confluence.content.render.xhtml.storage.link.user.UserMentionsExtractor;
import com.atlassian.confluence.core.BodyContent;
import com.atlassian.confluence.core.BodyType;
import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.plugins.mentions.api.MentionFinder;

import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLStreamException;
import java.io.StringReader;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.atlassian.confluence.core.BodyType.WIKI;
import static com.atlassian.confluence.core.BodyType.XHTML;
import static com.google.common.collect.Sets.newHashSet;
import static java.util.Collections.emptySet;

public class MentionFinderImpl implements MentionFinder {
    private static final Pattern USER_PROFILE_WIKI_MARKUP_LINK_PATTERN = Pattern.compile("\\[~[^\\\\,]+?\\]");

    private final XmlEventReaderFactory xmlEventReaderFactory;

    private final UserMentionsExtractor mentionExtractor;

    public MentionFinderImpl(XmlEventReaderFactory xmlEventReaderFactory, UserMentionsExtractor mentionExtractor) {
        this.xmlEventReaderFactory = xmlEventReaderFactory;
        this.mentionExtractor = mentionExtractor;
    }

    @Override
    public Set<String> getMentionedUsernames(ContentEntityObject ceo) {
        return getMentionedUsernames(ceo.getBodyContent());
    }

    @Override
    public Set<String> getMentionedUsernames(BodyContent content) {
        return getMentionedUsernames(content.getBody(), content.getBodyType());
    }

    @Override
    public Set<String> getNewMentionedUsernames(BodyContent oldContent, BodyContent newContent) {
        Set<String> usernames = getMentionedUsernames(newContent);
        Iterable<String> oldUsernames = getMentionedUsernames(oldContent);

        // remove usernames in the old content from the set of usernames in the new content so we can only notify newly
        // mentioned users
        for (String oldUsername : oldUsernames) {
            usernames.remove(oldUsername);
        }

        return usernames;
    }

    private Set<String> getMentionedUsernames(String body, BodyType bodyType) {
        if (bodyType == WIKI) {
            return getMentionedUsernamesFromWikiMarkupContent(body);
        } else if (bodyType == XHTML) {
            return getMentionedUsernamesFromXhtmlContent(body);
        } else {
            return emptySet();
        }
    }

    private Set<String> getMentionedUsernamesFromWikiMarkupContent(String content) {
        Matcher matcher = USER_PROFILE_WIKI_MARKUP_LINK_PATTERN.matcher(content);

        Set<String> usernames = new HashSet<>();

        while (matcher.find()) {
            String link = content.substring(matcher.start(), matcher.end());
            // get rid of [~ and ]
            String username = link.substring(2, link.length() - 1);
            usernames.add(username);
        }

        return usernames;
    }

    private Set<String> getMentionedUsernamesFromXhtmlContent(String content) {
        try {
            XMLEventReader reader = xmlEventReaderFactory.createStorageXmlEventReader(new StringReader(content));
            return newHashSet(mentionExtractor.extractUserMentions(reader));
        } catch (XMLStreamException e) {
            throw new RuntimeException(e);
        }
    }
}
