define('confluence/cjc/table-content-view', [
    'jquery',
    'underscore',
    'ajs',
    'backbone',
    'confluence/cjc/common/form-helper',
    'confluence/cjc/issue/model',
    'confluence/cjc/issue/collection'
],
function(
    $,
    _,
    AJS,
    Backbone,
    textHelper,
    IssueModel,
    IssueCollection
) {
    'use strict';

    /**
     * TableModeView is Backbone View for table in page content.
     * The table is current one which users is hightlight text on.
     */
    var TableModeView = Backbone.View.extend({

        initialize: function(options) {
            this.mappingFields = options.mappingFields;

            this.columns = this._extractHeaderFromTable();
            this.issueCollection = new IssueCollection();
        },

        getListColumns: function() {
            return this.columns;
        },

        getIssueRowIndexes: function() {
            return this.issueCollection.map(function(issueModel) {
                return issueModel.get('indexRow');
            });
        },

        _getCellContentByType: function($cellContent, issueField) {
            if (issueField.attributes.name === 'summary') {
                return $.trim(textHelper.removeLineBreaksAndTruncate($cellContent.text()));
            }
            return $.trim($cellContent.text());
        },

        // extract issues from the column
        extractIssuesFromTable: function() {
            this.issueCollection.reset();

            this.$el
                .find('> tbody > tr')
                .each(function(indexRow, elRow) {
                    var issueModel = new IssueModel();
                    issueModel.set('indexRow', indexRow);

                    var shouldSkip = false;

                    this.mappingFields.each(function(issueField) {
                        var indexColumn = issueField.get('index');
                        if (indexColumn === -1) {
                            // if a field is required but it does not map to any table column,
                            // we don't count as create-able issue
                            if (issueField.get('required')) {
                                shouldSkip = true;
                            }

                            return;
                        }

                        issueModel.set('indexColumn', indexColumn);

                        var $cell = $(elRow).find('td:eq(' + indexColumn + ')');
                        var cellContent = this._getCellContentByType($cell, issueField);

                        if (issueField.get('required') && !(cellContent instanceof Error)) {
                            if (!cellContent || !cellContent.length) {
                                shouldSkip = true;
                            }
                        }

                        var issueFieldForEachCell = issueField.clone();
                        issueFieldForEachCell.set('content', cellContent);
                        issueModel.get('fields').add(issueFieldForEachCell);
                    }.bind(this));

                    if (!shouldSkip) {
                        this.issueCollection.add(issueModel);
                    }
                }.bind(this));

            return this.issueCollection;
        },

        /**
         * Extract some information from table header.
         *
         * @returns {Array} - array of table header information, each item has table header text and column index.
         * @private
         */
        _extractHeaderFromTable: function() {
            var listTableIndexAndText = [];

            var tableHeaderCSS = '> thead > tr > th, > tbody > tr > th';
            this.$el
                .find(tableHeaderCSS).filter(':visible')
                .each(function(index, element) {
                    var $e = $(element);

                    var text = $.trim($e.find('> div').text());
                    text = text ? text : $.trim($e.text());

                    if (text) {
                        listTableIndexAndText.push({
                            index: index,
                            text: text
                        });
                    }

                });

            return listTableIndexAndText;
        },

        getColumnNameByIndex: function (index) {
            var columns = this.getListColumns();
            var matches = _.filter(columns, function(column) { return column.index === index});
            var columnName = matches.length >= 1 ? matches[0].text : '';

            return columnName;
        },

        /**
         * Verify if selected text in table is sorted table or not.
         */
        isInSortedTable: function() {
            //CONFDEV-32336: Due to conflunce table has change css class, add new for showing waring message correctly
            var selectorCSSs = [
                '> thead th.tablesorter-headerSortUp',
                '> thead th.tablesorter-headerSortDown',
                '> thead th.tablesorter-headerAscDesc',
                '> thead th.tablesorter-headerAsc',
                '> thead th.tablesorter-headerDesc',
            ];

            var isInSortedTable = _.some(selectorCSSs, function(selector) {
                return this.$el.find(selector).length > 0;
            }, this);

            return isInSortedTable;
        }
    });

    return TableModeView;
});
