define('confluence/cjc/create-issue-form/create-issue-form-dialog-view', [
    'jquery',
    'ajs',
    'underscore',
    'backbone',
    'confluence/cjc/common/form-helper',
    'confluence/cjc/error/error-type',
    'confluence/cjc/issue-field/init',
    'confluence/cjc/create-issue-form/create-single-issue-view',
    'confluence/cjc/create-issue-form/create-multiples-issues-view',
    'confluence/cjc/error/error-message-view',
    'confluence/cjc/create-issue-form/extra-required-fields-view',
    'confluence/cjc/create-issue-form/jira-epic-content-view',
    'confluence/cjc/create-issue-form/project-issue-type-select-view',
    'confluence/cjc/create-issue-form/server-select-view',
    'confluence/cjc/common/select2-helper',
    'confluence/cjc/analytics',
    'confluence/cjc/common/services',
    'confluence/cjc/common/jira-issues-helper'
],
function(
    $,
    AJS,
    _,
    Backbone,
    formHelper,
    ErrorType,
    issueFiledsInit,
    CreateSingleIssueView,
    CreateMultipleIssuesView,
    ErrorMessageView,
    ExtraRequiredFieldsView,
    JiraEpicContentView,
    ProjectIssueTypeSelectView,
    ServerSelectView,
    select2Helper,
    analytics,
    services,
    jiraIssueHelper
) {
    'use strict';

    var confluenceStorageManager;
    if (window.Confluence.storageManager) {
        confluenceStorageManager = window.Confluence.storageManager;
    } else {
        confluenceStorageManager = require('confluence/confluence-storage-manage');
    }

    var CreateIssueFormView = Backbone.View.extend({
        template: window.Confluence.CreateJiraContent,

        events: {
            'click .js-text-suggestion': '_onClickSwithModeLink',
            'click .create-issue-cancel': '_onClickCancelLink',
            'submit form#jira-content-create-issue-form': '_onSubmitForm'
        },

        initialize: function(/* options */) {
            // Creating Single Issue mode is default
            this.currentMode = CreateIssueFormView.TEXT_MODE;

            // jQuery DOM element of form
            this.ui = {
                $createIssueFormContent: null
            };

            // List table header and index
            this.listColumnInTable = [];

            // AUI Inline Dialog instance
            this.auiInlineDialog = null;

            // an native Selection Range object of browser
            this.selectionObject = null;

            // supported fields map
            this.mappingFieldCollection = null;

            // Status of description field
            this.currentDescriptionFieldStatus = null;

            // Declare an instance of Confluence.storageManager
            this.storageManager = confluenceStorageManager('jira-content');

            this.storedDialogValues = null;

            // an instance of `CreateSingleIssueView`.
            this.createSingleIssueView = null;

            // an instance of `CreateMultipleIssuesView`.
            this.createMultipleIssuesView = null;

            // an instance of `ErrorMessageView`.
            this.errorMessageView = null;

            // an instance of `ExtraRequiredFieldsView`.
            this.extraRequiredFieldsView = null;

            // an instance of `JiraEpicContentView`.
            this.jiraEpicContentView = null;

            // an instance of `ProjectIssueTypeSelectView`.
            this.projectIssueTypeSelectView = null;

            // an instance of `ServerSelectView`.
            this.serverSelectView = null;
        },

        render: function($form, selObject) {
            // re-set current `this.$el`
            this.$el = $form;
            this.el = this.$el[0];
            this.delegateEvents();

            // keep the text selection at the time the form is shown
            this.selectionObject = selObject;

            this.mappingFieldCollection = issueFiledsInit.init();

            this._initSingleAndMultipleCreateIssueFormView();

            this._renderFormContent();
            this._updateChangedMode();
        },

        /**
         * Render form
         * @private
         */
        _renderFormContent: function() {
            var tableContentView = this.createMultipleIssuesView && this.createMultipleIssuesView.tableContentView
                                        ? this.createMultipleIssuesView.tableContentView
                                        : null;

            this.listColumnInTable = tableContentView && tableContentView.getListColumns();

            var truncSummaryText = formHelper.trunc(this.selectionObject.text);
            var isCutLongText = formHelper.isCutLongText(truncSummaryText);

            var markup = this.template.createIssueDialog({
                summary: truncSummaryText,
                isCutLongText: isCutLongText,
                comment: this.selectionObject.text,
                mappingFields: this.mappingFieldCollection.toJSON(),
                tableComumnIndexAndText: this.listColumnInTable ? this.listColumnInTable : []
            });

            this.$el.html(markup);
            this.ui.$createIssueFormContent = this.$('#issue-content, #jira-epic-content');

            // show description field as first time
            this.currentDescriptionFieldStatus = window.Confluence.CreateJiraContent.JiraIssue.FieldStatus.VISIBLE;
            this.storedDialogValues = this._loadFormValues();

            this._initSubViews();

            // make sure we don't show text-suggestion when we highlight in a table which has one row and highlight in sentence.
            if (this.createMultipleIssuesView) {
                var columnIssues = this.createMultipleIssuesView.getIssueCollection();
                if (!columnIssues || columnIssues.length <= 1) {
                    this.$('.js-text-suggestion').remove();
                }
            } else {
                this.$('.js-text-suggestion').remove();
            }

            formHelper.preventEscKeyResetFormInIE(this.$el);
            this.trigger('rendered');
        },

        _initSubViews: function() {
            this._initErrorMessageView();
            this._initServerSelectVIew();

            this.extraRequiredFieldsView = new ExtraRequiredFieldsView({
                el: this.$('#jira-required-fields-panel')[0],
                formObjectView: this
            });

            this.jiraEpicContentView = new JiraEpicContentView({
                el: this.$('#jira-epic-content'),
                formObjectView: this
            });

            this._initProjectIssueTypeSelectView();
            
            Backbone.on('conf.jira.content.projects.loaded', function(projects) {
                var serverId = this.serverSelectView.getCurrentServer().id;
                this.jiraEpicContentView.detectEpicIssueOnPage(serverId);
            }.bind(this));
        },

        _initSingleAndMultipleCreateIssueFormView: function() {
            // in case creating issue from table, this is the containing table.
            var $tableElement = $(this.selectionObject.containingElement).closest('table');
            if ($tableElement.length > 0) {
                this.createMultipleIssuesView = new CreateMultipleIssuesView({
                    el: this.el,
                    formObjectView: this,
                    tableElement: $tableElement[0]
                });
            }

            // create view instance for create single issue
            this.createSingleIssueView = new CreateSingleIssueView({
                el: this.el,
                formObjectView: this
            });
        },

        _initErrorMessageView: function() {
            this.errorMessageView = new ErrorMessageView({
                el: this.$('.messages')[0]
            });

            this.errorMessageView.on('change', function() {
                this._toggleEnableSubmitButton(!this.errorMessageView.hasCritical());
            }.bind(this));
        },

        _initServerSelectVIew: function() {
            this.serverSelectView = new ServerSelectView({
                el: this.$('#jira-required-fields-panel')[0],
                formObjectView: this
            });

            this.serverSelectView.on('server.loaded.success', function() {
                this.jiraEpicContentView.reset();
                this.projectIssueTypeSelectView.loadProjects();
            }.bind(this));

            this.serverSelectView.on('server.select.changed', function() {
                this.jiraEpicContentView.reset();

                // In case user select other project which not match with our value, storedDialogValues is set to False, Then they re-select a valid one we have to reset storedDialogValues
                this.storedDialogValues = this._loadFormValues();
                this.projectIssueTypeSelectView.updateAfterServerChanged();
            }.bind(this));

            this.serverSelectView.on('authoriszed.success', function() {
                this.errorMessageView.removeMessage(ErrorType.ERROR_JIRA_SERVER_ID);
                this.jiraEpicContentView.reset();
                this.projectIssueTypeSelectView.loadProjects();
            }.bind(this));

            this.serverSelectView.on('fill.server.data.to.select', function($serverSelect) {
                // If storedDialogValues is defined, bind it into Select
                if (this.storedDialogValues) {
                    var isSetSuccess = select2Helper.setSelectFieldValue($serverSelect, this.storedDialogValues.serverId);
                    if (!isSetSuccess) {
                        this.storedDialogValues = false;
                    }
                }
            }.bind(this));

            this.serverSelectView.render();
        },

        _initProjectIssueTypeSelectView: function() {
            this.projectIssueTypeSelectView = new ProjectIssueTypeSelectView({
                el: this.$('#project-issue-type-select'),
                formObjectView: this,
                getStoredDialogValues: function() {
                    return this.storedDialogValues;
                }.bind(this),
                setStoredDialogValues: function(value) {
                    this.storedDialogValues = value;
                }.bind(this)
            });

            this.projectIssueTypeSelectView.on('project.changed', function(projectValue) {
                // Check if selected Project is match with
                if (this.storedDialogValues &&
                    projectValue !== this.storedDialogValues.projectId) {
                    this.storedDialogValues = false;
                }

                this.createMultipleIssuesView && this.createMultipleIssuesView.fillPreviewIssuesFromTable();

                this.checkCreateIssueFields();
            }.bind(this));

            this.projectIssueTypeSelectView.on('projects.loaded.error', function(err) {
                if (err.authenticationUri) {
                    var currentServer = this.serverSelectView.getCurrentServer();
                    currentServer.authUrl = err.authenticationUri;
                    this.serverSelectView.jiraServerStatus(401);
                } else if (err.status) {
                    this.serverSelectView.jiraServerStatus(err.status);
                } else {
                    this.serverSelectView.showJiraServerWarning(err.message);
                }
            }.bind(this));

            this.projectIssueTypeSelectView.render();
        },

        _updateChangedMode: function() {
            // clean up some kinds of message when switching mode.
            this.errorMessageView.removeMessage(ErrorType.ERROR_INVALID_SUBMIT_DATA_ID);
            this.errorMessageView.removeMessage(ErrorType.ERROR_CAN_NOT_DETECT_TABLE_HEADER);

            if (this.currentMode === CreateIssueFormView.TEXT_MODE) {
                this.createSingleIssueView.refresh();
            } else if (this.currentMode === CreateIssueFormView.TABLE_MODE) {
                this.createMultipleIssuesView.refresh();
            }

            this.refreshDialog();
        },

        resetAllErrorMessage: function() {
            this.errorMessageView.reset();
            // show form content
            this.ui.$createIssueFormContent.removeClass('hidden');

            this.refreshDialog();
        },

        setDialogObject: function(createIssueDialog) {
            this.auiInlineDialog = createIssueDialog;
        },

        /**
         * Handle cancel the dialog
         * @param e
         * @private
         */
        _onClickCancelLink: function(e) {
            e.preventDefault();

            // this variable is used as a flag to let jiraFormHelper.bindHideEventToDialog() function know if
            // the dialog is closed by clicking on Cancel Button
            // when Cancel is clicked, it will assign as TRUE, we don't need to reset this value since this dom will be reset each time it show
            this.auiInlineDialog.isCancelButtonClicked = true;
            this._destroyDialog();
        },

        /**
         * Clean up everything when closing the dialog.
         */
        _destroyDialog: function() {
            this.auiInlineDialog.remove();

            this.createSingleIssueView.remove();
            this.createMultipleIssuesView && this.createMultipleIssuesView.remove();
            this.errorMessageView.remove();

            this.remove();
        },

        bindHideEventToDialog: function() {
            this.serverSelectView.closeAllSelect2Dropdown();
            this.projectIssueTypeSelectView.closeAllSelect2Dropdown();

            // Save current Value for next showing if the dialog is hidden by nature, not by clicking on Cancel button.
            if ( this.auiInlineDialog.isCancelButtonClicked !== true) {
                this._saveFormValues();
            }
        },

        /**
         * Storing input value of creating issue and display it for next time open dialog
         **/
        _saveFormValues: function() {
            var values = this._getAllUserInputParams();
            if (values.serverId && values.projectId && values.issueTypeId) {
                this.storageManager.setItem(CreateIssueFormView.FORM_VALUE_STORAGE_KEY, JSON.stringify(values));
            }
        },

        /**
         * Return form's value as Object
         * FALSE if values is null or invalid
         * StoredValue Object if it's valid
         *
         * @param String key
         */
        _loadFormValues: function(key) {
            // Json string may not valid, so it would be good to wrap it inside a try/catch
            try {
                var selectedValue = JSON.parse(this.storageManager.getItem(key || CreateIssueFormView.FORM_VALUE_STORAGE_KEY));

                if (selectedValue && selectedValue.serverId && selectedValue.projectId && selectedValue.issueTypeId) {
                    return selectedValue;
                }

                return false;
            } catch (err) {
                AJS.debug(err);
                return false;
            }
        },

        /**
         * Handle submit form - clicking Create button of the dialog
         * @param e
         * @private
         */
        _onSubmitForm: function(/* e */) {
            if (this.currentMode === CreateIssueFormView.TEXT_MODE) {
                this.createSingleIssueView.submit();
            } else {
                this.createMultipleIssuesView.submit();
            }
            return false;
        },

        /**
         * Handle swiching between creating single and multiple issues form
         */
        _onClickSwithModeLink: function(e) {
            e.preventDefault();

            var $this = $(e.target);
            var isInTextMode = $this.hasClass('js-go-to-create-multiple-issues');
            if (isInTextMode) {
                this.currentMode = CreateIssueFormView.TABLE_MODE;
            } else {
                this.currentMode = CreateIssueFormView.TEXT_MODE;
            }

            this._updateChangedMode();
        },

        /**
         * Return true if current mode is Creating Single Issue mode
         * @returns {boolean}
         */
        isInTextMode: function() {
            return this.currentMode === CreateIssueFormView.TEXT_MODE;
        },

        /**
         * Return true if the dialog is showing in a Confluence sorted table
         * @returns {null|*}
         */
        isInSortedTable: function() {
            var tableContentView = this.createMultipleIssuesView && this.createMultipleIssuesView.tableContentView;
            return tableContentView && tableContentView.isInSortedTable();
        },

        changeIssuesTypeIconForPreviewPanel: function() {
            this.projectIssueTypeSelectView.changeIssuesTypeIconForPreviewPanel();
        },

        _toggleFormWaiting: function(isWaiting, isNoTextMessage) {
            // if the form does not any critcal error, we can enable create button
            if (this.errorMessageView.hasCritical()) {
                this._toggleEnableSubmitButton(false);
            } else {
                this._toggleEnableSubmitButton(!isWaiting);
            }

            this.$('.buttons-group').toggleClass('loading', isWaiting);
            this.$('.buttons-group').toggleClass('no-text', isNoTextMessage);
        },

        checkCreateIssueFields: function() {
            this.resetAllErrorMessage();

            this.extraRequiredFieldsView.reset();

            if (this.projectIssueTypeSelectView.$projectSelect.val() !== select2Helper.DEFAULT_OPTION_VALUE) {
                var onGetCreateIssueMetaSuccess = function(createMeta) {
                    this.trigger('confluence-jira-content.form-updated-fieldmeta', createMeta);

                    this.currentDescriptionFieldStatus = createMeta.descriptionFieldStatus;

                    if (!_.isEmpty(createMeta.requiredFields)) {
                        this.extraRequiredFieldsView.render(createMeta.requiredFields);
                    }

                    this.jiraEpicContentView.handleEpicContent(createMeta.epicField);
                    this._toggleFormWaiting(false, false);
                };

                var onCreateIssueMetaError = function(/* errorMessage */) {
                    // TODO: render error message.
                    this._toggleFormWaiting(false, false);
                };

                this._toggleFormWaiting(true, false);

                window.Confluence.CreateJiraContent.JiraIssue.getCreateIssueMeta(
                    this._getAllUserInputParams(),
                    onGetCreateIssueMetaSuccess.bind(this),
                    onCreateIssueMetaError.bind(this)
                );

                this.refreshDialog();
            } else {
                this.jiraEpicContentView.handleEpicContent();
            }
        },

        _toggleEnableSubmitButton: function(enable) {
            if (enable) {
                this.$('.create-issue-submit').removeAttr('disabled');
            } else {
                this.$('.create-issue-submit').attr('disabled', 'disabled');
            }
        },

        _disableForm: function() {
            this._toggleFormWaiting(true, true);

            // hide cancel button
            this.$('.create-issue-cancel').addClass('hidden');

            // disable all form inputs.
            this.$('input,select,textarea,submit').disable();
        },

        _convertFormToJSON: function(issues) {
            var self = this;
            var issueFormObject = {
                fields: {
                    project: {
                        id: this.projectIssueTypeSelectView.getProjectId()
                    },
                    issuetype: {
                        id: this.projectIssueTypeSelectView.getIssueTypeId()
                    }
                }
            };

            issueFormObject = this.extraRequiredFieldsView.fillRequiredFieldsToObject(issueFormObject);

            var createIssuesObj = {
                issues: []
            };

            var numOfTruncatedIssues = 0;
            var numOfEditedDescriptions = 0;

            $.each(issues, function(index, issue) {
                if (formHelper.isCutLongText(issue.summary)) {
                    numOfTruncatedIssues++;
                } else if (!issue.description) {
                    // count number of edited descriptions, but don't count case where truncated summary is place in description
                    numOfEditedDescriptions++;
                }

                var issueObject = _.clone(issueFormObject);
                issueObject.fields = _.extend({}, issueObject.fields, issue);

                if (self.currentDescriptionFieldStatus == window.Confluence.CreateJiraContent.JiraIssue.FieldStatus.REQUIRED &&
                        !issue.description) {
                    issueObject.fields.description = issue.summary;
                }
                // try remove issue description when description field is hidden
                if (self.currentDescriptionFieldStatus == window.Confluence.CreateJiraContent.JiraIssue.FieldStatus.HIDDEN) {
                    delete issueObject.fields.description;
                }

                createIssuesObj.issues.push(issueObject);
            });

            if (numOfTruncatedIssues > 0) {
                analytics.sendAnalyticsForTruncatedSummary(numOfTruncatedIssues);
            }

            if (numOfEditedDescriptions > 0) {
                analytics.sendAnalyticsForDescriptionEdited();
            }

            return JSON.stringify(createIssuesObj);
        },

        _buildErrorMessagesWhenCreateFromTable: function(errorIssues, issueRowIndexes) {
            // get error in each error issue when create
            var errorMessages = [];
            var thiz = this;
            if (errorIssues) {
                $.each(errorIssues, function(index, errorIssue) {
                    var errorMessage = AJS.I18n.getText('createjiracontent.dialog.create.issues.result.error',
                            (issueRowIndexes[errorIssue.failedElementNumber] + 1),
                            thiz._getErrorMessages(errorIssue));
                    errorMessages.push(errorMessage);
                });
            }
            return errorMessages;
        },

        /**
         * @param errorIssues list of issues that can't be created by server
         * @param issueRowIndexes list of issue indexes that are expected to be created by server
         * @returns a new issueRowIndexes in which all error row indexes are removed
         */
        _removeIndexOfErrorRows: function(errorIssues, issueRowIndexes) {
            if (!errorIssues) {
                return issueRowIndexes;
            }

            _.each(errorIssues, function(errorIssue) {
                issueRowIndexes[errorIssue.failedElementNumber] = -1;
            });
            return _.filter(issueRowIndexes, function(index) {
                return index >= 0;
            });
        },

        _getAllUserInputParams: function() {
            return {
                serverId: this.serverSelectView.getServerId(),
                projectId: this.projectIssueTypeSelectView.getProjectId(),
                issueTypeId: this.projectIssueTypeSelectView.getIssueTypeId(),
                defaultRequiredFields: CreateIssueFormView.DEFAULT_REQUIRED_FIELDS
            };
        },

        createIssues: function(issues) {
            // check if epic checkbox is checked or not.
            if (this.jiraEpicContentView.isEpicChecked()) {
                analytics.sendAnalyticsForCreatedIssueWithEpic();
                jiraIssueHelper.getCreateIssueMeta(
                    this._getAllUserInputParams(),
                    function(createMeta) {
                        // insert epic link in each issue
                        if (createMeta.epicField) {
                            var epicValue = this.jiraEpicContentView.getEpicCheckBoxValue();
                            $.each(issues, function(index, issue) {
                                issue[createMeta.epicField] = epicValue;
                            });
                        }
                        this._executeCreateJiraIssue(issues);
                    }.bind(this)
                );
            } else {
                this._executeCreateJiraIssue(issues);
            }
        },

        _getErrorMessages: function(errorIssue) {
            var errorFieldValues = _.values(errorIssue.elementErrors.errors);
            var errorFieldKeys = _.keys(errorIssue.elementErrors.errors);

            var getFieldParamObj = this._getAllUserInputParams();
            for (var i = 0; i < errorFieldKeys.length; i++) {
                var fieldName = window.Confluence.CreateJiraContent.JiraIssue.getFieldName(errorFieldKeys[i], getFieldParamObj);
                if (fieldName && errorFieldValues[i].indexOf(fieldName) === -1) {
                    errorFieldValues[i] = fieldName + ': ' + errorFieldValues[i];
                }
            }

            // get and display errors from both errorMessages and errors from response
            var errors = _.union(_.values(errorIssue.elementErrors.errorMessages), errorFieldValues);
            errors = errors.join(', ');
            // some message end with '.' so need replace '.,' by ',' only
            errors = errors.replace('.,', ',');
            return errors;
        },

        _executeCreateJiraIssue: function(issues) {
            this._disableForm();

            var serverId = this.serverSelectView.getServerId();
            var data = this._convertFormToJSON(issues);

            services.createIssues(serverId, data)
                .done(this._onCreateJiraIssueSuccess.bind(this))
                .fail(this._onCreateJiraIssueFail.bind(this));
        },

        /**
         * Handle creating jira issue(s) successfully.
         * @param data
         * @private
         */
        _onCreateJiraIssueSuccess: function(data) {
            var tableColumnIndex = this.mappingFieldCollection.getIssueFieldByName('summary').get('index')
            if (typeof tableColumnIndex !== 'undefined' && !this.isInTextMode()) {
                // created multiple issues
                this._updatePageAfterCreatedIssues(data, tableColumnIndex);
            } else if (!data.errors) {
                // created single issue
                this._updatePageAfterCreatedSingleIssue(data.issues[0]);
            } else {
                // error
                analytics.sendAnalyticsForCreateError('rest');
                this._refreshPageAfterInsertIssues([], false, [this._getErrorMessages(data.errors[0])]);
            }
        },

        /**
         * There is error in ajax call creating Jira issue.
         * @param xhr
         * @private
         */
        _onCreateJiraIssueFail: function(xhr) {
            var errorMessages = [];
            var responseErrorIssues = [];

            if (xhr.status === 500) {
                try {
                    responseErrorIssues = JSON.parse(xhr.responseText);

                    if (responseErrorIssues.errors) {
                        errorMessages.push(responseErrorIssues.errors[0].message);
                    } else if (responseErrorIssues instanceof Array) {
                        // in case we have multiple error issue, just display error in first issue.
                        $.each(responseErrorIssues[0].errors, function(field, errorMsg) {
                            errorMessages.push(errorMsg);
                        });
                    } else {
                        errorMessages.push(responseErrorIssues);
                    }
                } catch (e) {
                    responseErrorIssues = [];
                    errorMessages.push('Parse error:' + e.message);
                }
            } else {
                errorMessages.push(AJS.I18n.getText('createjiracontent.dialog.create.issue.error.server'));
            }

            this._refreshPageAfterInsertIssues([], false, errorMessages);
            analytics.sendAnalyticsForCreateError('xhr');
        },

        /**
         * insert created issue single mode
         * @param issue
         */
        _updatePageAfterCreatedSingleIssue: function(issue) {
            if (this.isInSortedTable() && this.selectionObject.searchText.numMatches !== 1) {
                this._refreshPageAfterInsertIssues([ issue ], false, []);
                return;
            }
            var insertionBean = window.Confluence.HighlightAction.createInsertionBean(
                                    [{
                                        xmlInsertion: this._toIssueMacroXml(issue)
                                    }],
                                    this.selectionObject.searchText
                                );

            analytics.sendAnalyticsForCreatingIssue('text');

            window.Confluence.HighlightAction.insertContentAtSelectionEnd(insertionBean)
                .done(function(data) {
                    if (!data) {
                        analytics.sendAnalyticsForJIMInsertFailed('text', 'algorithm');
                    }
                    this._refreshPageAfterInsertIssues([issue], data, []);
                }.bind(this))
                .fail(function(response) {
                    analytics.sendAnalyticsForJIMInsertFailed('text', 'server');
                    this._refreshPageAfterInsertIssues([issue], false, [], response.statusText);
                }.bind(this));
        },

        _toIssueMacroXml: function(issue) {
            var currentServer = this.serverSelectView.getCurrentServer();
            var jiraMacro = window.Confluence.CreateJiraContent.issueMacroXml({
                showSummary: false,
                serverName: currentServer.name,
                serverId: currentServer.id,
                issueKey: issue.issue.key
            });
            return jiraMacro;
        },

        /**
         * insert created issue from creating issue in table mode
         * @param paramObj: {createdIssueRowIndexes, createdIssues, errorMessages, formObjectView, tableColumnIndex}
         */
        _insertCreatedIssuesInPage: function(paramObj) {
            var self = this;

            if (this.isInSortedTable()) {
                this._refreshPageAfterInsertIssues(paramObj.createdIssues, false, paramObj.errorMessages);
                return;
            }

            var cellXmlInsertions = [];
            // Prepare jira macro xml content for insert into table and created
            // issues messages
            $.each(paramObj.createdIssues, function(index, createdIssue) {
                var cellXmlInsertion = {};
                cellXmlInsertion.rowIndex = paramObj.createdIssueRowIndexes[index];
                cellXmlInsertion.xmlInsertion = self._toIssueMacroXml(createdIssue);
                cellXmlInsertions.push(cellXmlInsertion);
            });

            var tableInsertionBean = window.Confluence.HighlightAction.createTableInsertionBean(
                                            cellXmlInsertions,
                                            paramObj.tableColumnIndex,
                                            this.selectionObject.searchText
                                        );

            analytics.sendAnalyticsForCreatingIssue('table', paramObj.createdIssues.length);

            window.Confluence.HighlightAction.insertContentsInTableColumnCells(tableInsertionBean)
                    .done(function(data) {
                        if (!data) {
                            analytics.sendAnalyticsForJIMInsertFailed('table', 'algorithm');
                        }
                        this._refreshPageAfterInsertIssues(paramObj.createdIssues, data, paramObj.errorMessages);
                    }.bind(this))
                    .fail(function(response) {
                        analytics.sendAnalyticsForJIMInsertFailed('table', 'server');
                        this._refreshPageAfterInsertIssues(paramObj.createdIssues, false, paramObj.errorMessages, response.statusText);
                    }.bind(this));
        },

        _refreshPageAfterInsertIssues: function(createdIssues, addedToPage, errorMessages, statusText) {
            var issuesURL = this.serverSelectView.getCurrentServer().url;
            if (createdIssues.length === 1) {
                issuesURL += '/browse/' + createdIssues[0].issue.key;
            } else {
                var jql = 'key in (';
                jql += $.map(createdIssues, function(issue) {
                    return issue.issue.key;
                }).join(',');
                jql += ')';
                issuesURL += '/issues/?jql=' + encodeURIComponent(jql);
            }
            var currentPath = window.location.href.split('#')[0];
            var oldJIRAIssueCreatedIdx = currentPath.indexOf('JIRAIssuesCreated');
            if (oldJIRAIssueCreatedIdx > 0) {
                currentPath = currentPath.substring(0, oldJIRAIssueCreatedIdx - 1);
            }
            var reloadURL = currentPath
                    + (currentPath.indexOf('?') > 0 ? '&' : '?')
                    + 'JIRAIssuesCreated=true'
                    + '&numOfIssues=' + createdIssues.length
                    + '&issuesURL=' + encodeURIComponent(issuesURL)
                    + '&addedToPage=' + addedToPage;
            if (createdIssues.length > 0) {
                reloadURL += '&issueId=' + encodeURIComponent(createdIssues[0].issue.key);
            }

            if (statusText) {
                reloadURL += '&statusText=' + statusText;
            }
            $.each(errorMessages, function(index, message) {
                reloadURL += '&errorMessages=' + message;
            });

            this._toggleFormWaiting(false, true);
            this.auiInlineDialog.hide();
            window.location.replace(reloadURL);
        },

        _updatePageAfterCreatedIssues: function(data, tableColumnIndex) {
            var tableContentView = this.createMultipleIssuesView.tableContentView;
            var issueRowIndexes = tableContentView.getIssueRowIndexes();

            var createdIssueRowIndexes = [];
            var errorMessages = this._buildErrorMessagesWhenCreateFromTable(data.errors, issueRowIndexes);
            issueRowIndexes = this._removeIndexOfErrorRows(data.errors, issueRowIndexes);

            $.each(data.issues, function(index, issue) {
                createdIssueRowIndexes.push(issueRowIndexes[issue.elementNumber]);
            });
            if (data.issues.length) {
                this._insertCreatedIssuesInPage({
                    createdIssues: data.issues,
                    createdIssueRowIndexes: createdIssueRowIndexes,
                    tableColumnIndex: tableColumnIndex,
                    errorMessages: errorMessages
                });
            } else if (data.errors && data.errors.length) {
                this._refreshPageAfterInsertIssues([], false, errorMessages);
            }
        },

        refreshDialog: function() {
            if (this.auiInlineDialog) {
                this.auiInlineDialog.refresh();
            }
        },
    },
    // declaring constants
    {
        TABLE_MODE: 'table', // Creating Multiple Issues mode
        TEXT_MODE: 'text', // Creating Single Issue mode
        // Declare name of Storege Key as a constant
        FORM_VALUE_STORAGE_KEY: 'savedFormValues',
        DEFAULT_REQUIRED_FIELDS: ['project', 'issuetype', 'summary', 'reporter']
    });

    return CreateIssueFormView;
});
