define('confluence-inline-tasks/task-report-bp',
[
    'jquery',
    'ajs',
    'confluence/legacy',
    'confluence-inline-tasks/space-page-picker-shim'
],
/**
 * Defines macro browser overrides for the Tasks Report macro.
 */
function (
    $,
    AJS,
    Confluence,
    SpacePagePicker
) {
    'use strict';
    // aui/event
    var AuiEvent = {
        bind: AJS.bind,
        trigger: AJS.trigger
    };

    var wizardObj;

    function selectItem($item) {
        $item.addClass('selected').siblings().removeClass('selected');
        AuiEvent.trigger('selected.task-report-types', $item);
    }

    // Select report type dialog
    function initSelectReportTypeDialog(e, state) {
        AuiEvent.trigger('analyticsEvent', {name: 'confluence-spaces.tasks.report_blueprint.clicked'});

        var $selectionContainer = $('#task-report-types');
        var $templates = $selectionContainer.find('.template');
        $templates.first().addClass('selected');
        $templates
            .click(function() {
                selectItem($(this));
            })
            .dblclick(function() {
                $('.create-dialog-create-button:visible').click();
            });
        $selectionContainer
            .attr('tabindex', 0)
            .keydown(function(event) {
                if (event.keyCode === 38 || event.keyCode === 40) { // up or down
                    event.preventDefault(); // prevents page behind the dialog from scrolling in FF

                    var $selected = $selectionContainer.find('.selected');
                    var $newSelected;
                    if (event.keyCode === 38) { // up
                        $newSelected = $selected.prev().length ? $selected.prev() : $templates.last();
                    } else if (event.keyCode === 40) { // down
                        $newSelected = $selected.next().length ? $selected.next() : $templates.first();
                    }
                    selectItem($newSelected);
                }
            });

        $selectionContainer.focus();

        AuiEvent.bind('selected.task-report-types', function(ev, $item) {
            if ($($item).is('#custom-task-report')) {
                // aui/i18n
                setSubmitButtonLabel(AJS.I18n.getText('create.content.plugin.create-dialog.submit-button.label'));
            } else {
                // aui/i18n
                setSubmitButtonLabel(AJS.I18n.getText('create.content.plugin.create-dialog.next-button.label'));
            }
        });

        // set default submit button label
        setSubmitButtonLabel('Next');
    }

    function submitSelectReportTypeDialog(e, state) {
        var selectedTypeId = state.$container.find('.selected').attr('id');
        if (selectedTypeId === 'team-task-report') {
            state.nextPageId = 'teamTaskReportId';
            AuiEvent.trigger('analyticsEvent', {name: 'confluence-spaces.tasks.report_team.clicked'})
        } else if (selectedTypeId === 'location-task-report') {
            state.nextPageId = 'locationTaskReportId';
            AuiEvent.trigger('analyticsEvent', {name: 'confluence-spaces.tasks.report_project.clicked'})
        } else if (selectedTypeId === 'custom-task-report') {
            wizardObj.getSubmissionRestPath = function() {
                AuiEvent.trigger('analyticsEvent', {name: 'confluence-spaces.tasks.report_custom.clicked'});
                return '/rest/create-dialog/1.0/content-blueprint/create-draft';
            };
            setContentTemplate(state, 'custom-task-report-template');
        } else {
            return false;
        }
    }

    // Team task report dialog
    function initReportByTeamDialog(e, state) {
        handlingCheckEventForCBIncludeCompletedTasks();
    }

    function submitReportByTeamDialog(e, state) {
        var $form = $('#report-by-team-form');
        resetErrorMsgs($form);

        var validForm = true;
        validForm = validateRequiredField($('#task-report-team-picker'), AJS.I18n.getText('task.report.blueprint.wizard.team.error.required')) && validForm;
        validForm = validateTitle(state.wizardData.spaceKey) && validForm ;

        validForm ? setContentTemplate(state, 'team-task-report-template')
                  : focusOnTheFirstErrorField($form);
        AJS.trigger('analyticsEvent', {name: 'confluence-spaces.tasks.report_team.created'});
        return validForm;
    }

    // Location task report dialog
    function initReportByLocationDialog(e, state) {
        // build location picker
        buildLocationPicker($('#task-report-location-picker'), state.wizardData.spaceKey);

        // build label picker
        buildLabelPicker($('#task-report-label-picker'));

        handlingCheckEventForCBIncludeCompletedTasks();
    }

    function submitReportByLocationDialog(e, state) {
        var $form = $('#report-by-location-form');
        resetErrorMsgs($form);

        var validForm = true;
        var locationPicker = $('#task-report-location-picker');
        validForm = validateRequiredField(locationPicker, AJS.I18n.getText('task.report.blueprint.wizard.location.error.required')) && validForm;
        validForm = validateTitle(state.wizardData.spaceKey) && validForm;

        if (validForm) {
            // LocationTaskReportContextProvider expects the location as 'locationsSpace' and 'locationsPage', so add
            // these to the wizardData.
            var spageMap = SpacePagePicker.getValueAsMap(locationPicker);
            if (spageMap.space.length) {
                state.wizardData.locationsSpace = spageMap.space.join(',');
            }
            if (spageMap.page.length) {
                state.wizardData.locationsPage = spageMap.page.join(',');
            }
            setContentTemplate(state, 'location-task-report-template');
        }
        else {
            focusOnTheFirstErrorField($form);
        }
        AuiEvent.trigger('analyticsEvent', {name: 'confluence-spaces.tasks.report_project.created'});
        return validForm;
    }

    /**
     * Remove existing visible error message
     * @param $form
     */
    function resetErrorMsgs($form) {
        $form.find('.error').addClass('hidden').empty();
    }

    /**
     * Validate title, checking if it is null or duplicated
     * Show error message if title is not valid
     */
    function validateTitle(spaceKey) {
        var $titleField = $('#task-report-page-title');
        var validTitle = Confluence.Blueprint.validateTitle($titleField, spaceKey);
        if (!validTitle) {
            $titleField.siblings('.error').removeClass('hidden');
        }
        return validTitle;
    }

    /**
     * Checking if the field is empty
     * Show error message if field value is not valid
     */
    function validateRequiredField($field, errorMsg) {
        var fieldValue = $.trim($field.val());
        if (!fieldValue && errorMsg) {
            updateError($field, errorMsg);
        }
        return !!fieldValue;
    }

    /**
     * Focus in the first error field
     */
    function focusOnTheFirstErrorField($form) {
        var $firstVisibleError = $form.find('div.error:not(.hidden)').first();
        var $firstErrorInput = $firstVisibleError.siblings('input').eq(0);
        if ($firstErrorInput.hasClass('select2-offscreen')) {
            $firstErrorInput.select2('focus');
        } else {
            $firstErrorInput.focus();
        }
    }

    /**
     * Set next dialog base on selected report type
     */
    function setContentTemplate(state, templateKey) {
        state.pageData.contentTemplateKey = templateKey;
    }

    /**
     * Display error message
     */
    function updateError($field, error) {
        var $errorField = $field.siblings('.error');
        $errorField.html(error);
        $errorField.removeClass('hidden');
        return error;
    }

    /**
     * Build label picker
     */
    function buildLabelPicker($labelsField) {
        $labelsField.auiSelect2(Confluence.UI.Components.LabelPicker.build());
    }

    /**
     * Build space and page picker
     */
    function buildLocationPicker($locationField, defaultSpace) {
        $locationField.auiSelect2(SpacePagePicker.build({
            spaceKeys: [defaultSpace],
            orgElement: $locationField
        }));
    }

    /**
     * Set label of next button
     */
    function setSubmitButtonLabel(text) {
        $('.create-dialog-create-button:visible').text(text);
    }

    // CONFDEV-23049: since the assignee input field is collapsed, the following checkbox jumps up so that the click event can not be performed on checkbox.
    function handlingCheckEventForCBIncludeCompletedTasks() {
        var pendingClickOnCB;
        var checkBox = $('#task-report-completed-cb');
        var clickTarget = $('label[for=task-report-completed-cb]').add(checkBox);

        clickTarget.mousedown(function() {
            pendingClickOnCB = true;
        });

        $('.dialog-wizard-page-main').on('mouseup', function(e) {
            if (!clickTarget.is(e.target) && pendingClickOnCB) {
                checkBox.click();
            }
            pendingClickOnCB = false;
        });
    }

    return {
        wizarder: function (wizard) {
            wizard.on('post-render.selectTaskReportId', initSelectReportTypeDialog);
            wizard.on('submit.selectTaskReportId', submitSelectReportTypeDialog);

            //for task report by team member
            wizard.on('post-render.teamTaskReportId', initReportByTeamDialog);
            wizard.on('submit.teamTaskReportId', submitReportByTeamDialog);

            //for task report by creator
            wizard.on('post-render.locationTaskReportId', initReportByLocationDialog);
            wizard.on('submit.locationTaskReportId', submitReportByLocationDialog);

            // This is rank, but refactoring this out during the current AMDify task is out of scope. Just be warned
            // that it's here.
            wizardObj = wizard;
        }
    };
});