define('confluence/ic/view/create-comment',
    [
        'underscore',
        'jquery',
        'backbone',
        'ajs',
        'confluence/ic/model/comment',
        'confluence/ic/model/reply',
        'confluence/ic/util/utils',
        'confluence/ic/editor/editor'
    ],
    function(
        _,
        $,
        Backbone,
        AJS,
        Comment,
        Reply,
        Utils,
        Editor
    ) {

    "use strict";

    var CreateCommentView = Backbone.View.extend({
        tagName: "div",

        className: "ic-create-comment-view",

        events: {
            "click .ic-action-save" : "saveComment",
            "click .ic-action-discard" : "discardComment",
            "keyup .ic-textarea" : "onInput",
            "focus .ic-textarea" : "onFocus",
            "click #show-page-comment" : "showPageComment"
        },

        template: Confluence.Templates.IC.createComment,

        initialize: function(options) {
            this.selection = options.selection;
            this.type = options.type; // toplevel, reply or edit
            this.onFinished = options.onFinished || function() {};
            this.commentText = options.commentText || "";
            this.stopFocus = options.stopFocus;
            this.serializedHighlights = options.serializedHighlights;

            // only one edit/reply at a time, if creating an edit view or reply view, destroy all other ongoing edit/reply
            if (options.type === 'edit' || options.type === 'reply') {
                this.template = Confluence.Templates.IC.commentForm;
                Backbone.trigger("ic:discard-edit");
                this.listenTo(Backbone, "ic:discard-edit", this.discardComment);
            }
        },

        finish: function() {
            this.onFinished();
            // HACK!!! for render JIM asynchronous, this for saving top comment
            AJS.trigger("ic-jim-async-supported");
        },

        render: function() {
            var author = this._getAuthorObject();
            var templateData = {
                type: this.type,
                text: this.commentText,
                showMenu: false
            };
            var data = _.extend({}, author, templateData, { darkFeatures: Utils.getDarkFeatures()});
            this.$el.html(this.template(data));

            if (Utils.getDarkFeatures().RICH_TEXT_EDITOR) {
                this.renderEditor();
            } else {
                // focus() the textarea asynchronously. If focus is called before the create form is positioned, it will
                // cause the scroll to jump to the top of the page.
                if (!this.stopFocus) {
                    setTimeout(_.bind(this.focusEditor, this), 1);
                }
            }

            Utils.addSidebarHeadingTooltip(this);

            return this;
        },

        onInput: function(e) {
            var hasText = this.$(e.target).val().trim() !== "";
            this._saveDisable(!hasText);
        },

        onFocus: function() {
            if (this.type === 'edit') {
                var $textarea = this.$('.textarea');
                Utils.moveCaretToEnd($textarea[0]);
                this._saveDisable($textarea.text() === "");
            }
        },

        saveComment: function(e) {
            e.preventDefault();
            if (!this._isValid()) {
                return;
            }

            this._disableForm();

            if (this.type === "reply") {
                this._addReply();
            } else if (this.type === "edit") {
                this._saveEdit();
            } else {
                this._addComment();
            }
        },

        /**
         * If creating a comment, the author is the current user, if editing a comment, the author is the comment creator
         * @returns {{authorUserName: *, authorAvatarUrl: *, authorDisplayName: *}}
         * @private
         */
        _getAuthorObject: function() {
            var authorUserName;
            var authorAvatarUrl;
            var authorDisplayName;
            if (this.type === 'edit') {
                authorUserName = this.model.get("authorUserName");
                authorAvatarUrl = this.model.get("authorAvatarUrl");
                authorDisplayName = this.model.get("authorDisplayName");
            } else {
                authorUserName = AJS.Meta.get('remote-user');
                authorAvatarUrl = Utils.getAuthorAvatarUrl();
                authorDisplayName = Utils.getAuthorDisplayName();
            }
            return {
                authorUserName: authorUserName,
                authorAvatarUrl: authorAvatarUrl,
                authorDisplayName: authorDisplayName
            };
        },

        _saveEdit: function() {
            var text = this._getContent();
            var options = {
                wait: true,
                beforeSend: _.bind(this.beforeSendAjax, this),
                success: _.bind(this._onSaveSuccess, this),
                error: _.bind(this._showEditError, this)
            };
            this.model.save({ body: text }, options);
        },

        _showEditError: function(model, response) {
            //CONFDEV-32184
            if (this.showCaptchaError(response)) return;

            var errorMsg;
            if (response.status === 401) {
                errorMsg = AJS.I18n.getText('inline.comments.edit.permission')
            } else {
                errorMsg = AJS.I18n.getText('inline.comments.server.edit.error')
            }
            this._triggerError(errorMsg);
        },

        _triggerError: function(message, options) {
            Backbone.trigger("ic:error", message, options);
            this._disableForm(false); // Re-enable form
        },

        _addReply: function() {
            var reply = new Reply({
                body: this._getContent(),
                commentId: this.model.get("id"),
                hasDeletePermission: !!AJS.Meta.get('remote-user') || this.model.get("hasDeletePermission")
            });

            reply.save(null, {
                beforeSend: _.bind(this.beforeSendAjax, this),
                success: _.bind(this._onSaveSuccess, this),
                error: _.bind(function(model, response) {
                    //CONFDEV-32184
                    if (!this.showCaptchaError(response)) {
                        this._triggerError(AJS.I18n.getText('inline.comments.sidebar.error.save.reply'));
                    }
                }, this)
            });
        },

        _addComment: function() {
            var comment = new Comment({
                originalSelection: this.selection.searchText.selectedText.replace(/\n/g,''),
                body: this._getContent(),
                matchIndex: this.selection.searchText.index,
                numMatches: this.selection.searchText.numMatches,
                serializedHighlights: this.serializedHighlights
            });
            var errorOptions = {
                callback: this.bindPageReloadLink
            };
            comment.save(null, {
                beforeSend: _.bind(this.beforeSendAjax, this),
                success: _.bind(this._onSaveSuccess, this),
                error: _.bind(function(model, response) {
                    //CONFDEV-32184
                    if (this.showCaptchaError(response)) return;

                    if (response.status === 409) {
                        this._triggerError(AJS.I18n.getText('inline.comments.sidebar.error.stale', '<a href="#" class="ic-reload-page">', '</a>'), errorOptions);
                    } else if (response.status === 412) {
                        AJS.trigger('analytics', {name: 'confluence.comment.inline.cannot.create'});
                        this._triggerError(AJS.I18n.getText('inline.comments.sidebar.error.create.comment'));
                    } else {
                        this._triggerError(AJS.I18n.getText('inline.comments.sidebar.error.save.comment'));
                    }
                }, this)
            });
        },

        _isValid: function() {
            if ($.trim(this._getContent()) === '<p><br /></p>') {
                Backbone.trigger("ic:error", AJS.I18n.getText('inline.comments.sidebar.comment.empty'));
                return false;
            }
            return true;
        },

        _onSaveSuccess: function(response) {
            switch (this.type) {
                case 'topLevel':
                    //when create inline comment on macro, it will create a page comment and return only commentId attr
                    response.set('hasDeletePermission', true);
                    Backbone.trigger("ic:persist", response);
                    Backbone.trigger("ic:view", response, undefined, {ignoreConfirmDialog: true}); // pass ignoreConfirmDialog option will avoid showing confirm message
                    break;
                case 'reply':
                    this.model.replies.push(response);
                    Backbone.trigger("ic:reply:persist", response);
                    this.finish();
                    break;
                case 'edit':
                    Backbone.trigger("ic:edit");
                    this.finish();
                    break;
            }
        },

        _setEditorIsNotDirty: function() {
            if (AJS.Rte.getEditor()) {
                AJS.Rte.getEditor().isNotDirty = 1;
            }
        },

        _disableForm: function(isDisabled) {
            if (isDisabled === undefined) {
                isDisabled = true; // disable if no argument passed
            }
            this._saveDisable(isDisabled);

            if (!Utils.getDarkFeatures().RICH_TEXT_EDITOR) {
                this.$(".ic-action-discard,.ic-textarea").prop("disabled", isDisabled).attr("aria-disabled", isDisabled);
            }
        },

        _saveDisable: function(isDisabled) {
            if (Utils.getDarkFeatures().RICH_TEXT_EDITOR) {
                Editor.setEditorBusy(isDisabled);
                $("#rte-button-cancel")[0].disabled = isDisabled; // Cancel button
            } else {
                this.$(".ic-action-save").prop("disabled", isDisabled).attr("aria-disabled", isDisabled);
            }
        },

        renderEditor: function() {
            var $container = this.$('.ic-body');
            var content = this.commentText;
            var quickComment = AJS.Confluence.QuickEdit.QuickComment;

            return Editor.init({
                container: $container,
                form: $container.find('form.aui'),
                saveHandler: this.type === 'pageComment' ? quickComment.createSaveHandler(_.bind(this._successCreatePageComment, this), quickComment.saveCommentErrorHandler)
                    : _.bind(this.saveComment, this),
                cancelHandler: _.bind(this.discardComment, this),
                fetchContent: function() {
                    var responseDeferred = new $.Deferred();
                    responseDeferred.resolve({editorContent: content});
                    return responseDeferred;
                },
                postDeactivate: _.bind(function() {
                    // In the case that editor and error message is showing, then active top level comment on page,
                    // we shoud also clear error
                    Backbone.trigger("ic:clearError");
                    this.finish();
                }, this),
                hideCancelButton: this.type === 'topLevel'
            })
                .done(_.bind(this.renderEditorCompleted, this))
                .fail(_.bind(this.renderEditorFail, this));
        },

        showPageComment: function(e) {
            var commentId = $(e.target).data('comment-id');
            var $commentElement = $('#comment-' +commentId);
            $commentElement.addClass("focused");
            $commentElement.scrollToElement();
        },

        _successCreatePageComment: function(comment) {
            Confluence.CommentDisplayManager.addComment(Utils.getCurrentUserInfo(), comment, false, false, true);
            AJS.trigger("page.commentAddedOrUpdated", {commentId: comment.id});
            this._setEditorIsNotDirty();
            if (Utils.getDarkFeatures().RICH_TEXT_EDITOR) {
                /*
                 must set to true b/c after editor is removed, a callback postDeactivate will be called, which hide the sidebar
                 this cb was passed from sidebar.js whilst initializing CreateCommentView Object
                 */
                this.keepSidebar = true;
                //CONFDEV-33184: must remove editor in a correct way before removing it from DOM
                Editor.remove();
            }
            this.$el.html(Confluence.Templates.IC.pageComment({commentId: comment.id}));
            Utils.addSidebarHeadingTooltip(this);
        },

        renderEditorCompleted: function() {
            Backbone.trigger("ic:clearError");
            if (this.type === 'topLevel' || this.type ==='pageComment') {
                // Cancel button should be hidden when create new comment
                this.$('#rte-button-cancel').hide();

                if (this.type ==='pageComment') {
                    Backbone.trigger("ic:error", AJS.I18n.getText('page.comments.sidebar.warning'));
                    this.focusEditor();
                }
            }
            //CONFDEV-32184:init captchaManager
            this.captchaManager = new AJS.Confluence.QuickEditCaptchaManager(AJS.Confluence.EditorLoader.getEditorForm());
        },

        renderEditorFail: function() {
            //options for error message
            var errorOptions = {
                closeable: true,
                callback: this.bindPageReloadLink
            };
            switch (this.type) {
                case 'pageComment':
                    this._setEditorIsNotDirty();
                    break;
                case 'topLevel':
                    this.$('.loading-container').hide();
                    this.$('.ic-create-comment').hide();
                    errorOptions.closeable = false;
                    break;
                case 'edit':
                case 'reply':
                    this.finish();
                    break;
            }

            Backbone.trigger("ic:error", AJS.I18n.getText('inline.comments.editor.timeout', '<a href="#" class="ic-reload-page">', '</a>'), errorOptions);
            Backbone.trigger("ic:editor:load:fail");
        },

        /**
         * Bind event to a link to reload page
         */
        bindPageReloadLink: function () {
            //delegate event in order not to worry the message is appended after binding event
            $('.ic-error')
                .off("click", ".ic-reload-page")
                .on("click", ".ic-reload-page", function () {
                    window.location.reload();
                    return false;
                });
        },

        discardComment: function(e) {
            e && e.preventDefault();
            Backbone.trigger("ic:clearError");
            this.finish();
        },

        _getContent: function() {
            if (Utils.getDarkFeatures().RICH_TEXT_EDITOR) {
                return Editor.getContent();
            } else {
                return this.$('textarea').val();
            }
        },

        focusEditor: function() {
            if (Utils.getDarkFeatures().RICH_TEXT_EDITOR) {
                if(this.type === 'pageComment') {
                    var $body = $(AJS.Rte.getEditor() && AJS.Rte.getEditor().getBody());
                    var focusNode = $body.find("div")[0];
                    AJS.Rte.getEditor().selection.select(focusNode, 1);
                    AJS.Rte.getEditor().selection.collapse(1);
                    focusNode.focus();
                } else {
                    AJS.Rte.getEditor() && AJS.Rte.getEditor().focus();
                }

            } else {
                this.$('.ic-textarea:visible').focus();
            }
        },

        destroy: function() {
            if (Utils.getDarkFeatures().RICH_TEXT_EDITOR) {
                Editor.remove();
            }
            this.remove();
        },

        //CONFDEV-32184
        beforeSendAjax: function(xhr) {
            var captcha = this.captchaManager.getCaptchaData();
            if (captcha && captcha.id) {
                xhr.setRequestHeader("X-Atlassian-Captcha-Id", captcha.id);
                xhr.setRequestHeader("X-Atlassian-Captcha-Response", captcha.response);
            }
        },

        //CONFDEV-32184
        /**
         * If it is an error b/c of captcha, we refresh captcha and display error
         * @param response the ajax response
         * @returns {boolean}
         */
        showCaptchaError: function(response) {
            if (response.status === 404 && response.responseText && response.responseText.search(/captcha/i) != -1) {
                this.captchaManager.refreshCaptcha();
                this._triggerError(AJS.I18n.getText("captcha.response.failed"));
                return true;
            }
            return false;
        }
    });

    return CreateCommentView;
});
