require([
    'confluence/hipchat/space-to-room/internal/service',
    'confluence/hipchat/space-to-room/internal/admin-space-picker',
    'underscore'
],
/**
 * Internal ui integration services for space to room configuration.
 *
 * @tainted Confluence.Templates.HipChat.SpaceToRoom.Config (soy template)
 */
function (
    //AJS, Don't inject AJS because in Conf 5.5.6 it is minimised. See HC-9447.
    serviceFactory,
    adminSpacePicker,
    _
) {
    'use strict';

    AJS.toInit(function($) {
        var space = null;
        var form = $('#confluence-space-to-room-mapping-form');
        var notificationTypes = form.data('notification-types');

        if (AJS.Meta.get('space-key')) {
            space = {
                key: AJS.Meta.get('space-key'),
                name: AJS.Meta.get('space-name')
            }
        }

        var confirmDialog;
        var service = serviceFactory(space);
        var configService = service.configService;
        var roomServicePromise = service.roomServicePromise;
        var pendingSpaceRoomNotificationConfigs = (function() {
            var newRoomMap = {};

            return {
                addSpaceRoomNotificationConfig: function(space, room) {
                    if (space && room) {
                        var spaceRoomKey = space.key + '--' +  room.roomId;
                        newRoomMap[spaceRoomKey] = {space: space, room : room};
                    }
                },
                removeRoomBySpaceKeyAndRoomId: function(spaceKey, roomId) {
                    var spaceRoomKey = spaceKey + '--' +  roomId;
                    var removed = !!newRoomMap[spaceRoomKey];
                    delete newRoomMap[spaceRoomKey];
                    return removed;
                }
            };
        })();

        roomServicePromise.fail(function (xhr) {
            AJS.messages.error("#hipchat-space-to-room-messages", {
                title: AJS.I18n.getText('hipchat2-space-configuration.select.room.picker.error')
            });
        });

        function roomQuery(query) {
            roomServicePromise.done(function(roomService) {
                var rooms = roomService.getSelectableRooms();
                var trimmedSearchText = query.term.trim();
                var searchStrLower = trimmedSearchText.toLocaleLowerCase();
                var filteredRooms = _.filter(rooms, function(item) {
                    item.text = item.roomName;
                    return item.roomName.toLocaleLowerCase().indexOf(searchStrLower) >= 0;
                });
                if(HipChat.UserLink.context.userLinked) {
                    var results = [];
                    var matchCount = filteredRooms.length;
                    var exactMatchingRoom = roomService.findRoomByNameIgnoreCase(searchStrLower);
                    if (matchCount >= 1) {
                        results.push({
                            text: AJS.I18n.getText('hipchat2-space-configuration.select.room.matching.title'),
                            children: filteredRooms
                        });
                    }
                    if (HipChat.UserLink.context.userHasManageRoomsScope && !exactMatchingRoom && trimmedSearchText.length > 0) {
                        if (!roomService.findRoomByNameIgnoreCase(trimmedSearchText)) {
                            results.push({
                                text: AJS.I18n.getText('hipchat2-space-configuration.select.room.create.title'),
                                children: [{
                                    id: 0,
                                    text: trimmedSearchText,
                                    roomName: trimmedSearchText,
                                    isCreateRoomItem: true,
                                    isPrivate: false
                                }]
                            });
                        }
                    }
                    query.callback({results: results});
                } else {
                    query.callback({
                        results: [{
                            text: AJS.I18n.getText('hipchat2-space-configuration.select.room.public.title'),
                            children: filteredRooms
                        }]
                    });
                }
            });
        }

        var roomRenderer = function(room) {
            if (room.isCreateRoomItem) {
                return Confluence.Templates.HipChat.SpaceToRoom.Config.renderNewRoomItem({
                    name: room.roomName,
                    privateRoom: room.isPrivate
                });
            }
            if(!room.id) {
                return room.text;
            }
            return Confluence.Templates.HipChat.SpaceToRoom.Config.renderRoomItem({
                name: room.roomName,
                privateRoom: room.isPrivate
            });
        };

        var enableAddRoomButton = function() {
            $('#hipchat-space-to-room-add').removeAttr('aria-disabled').removeAttr('disabled')
        };

        var updateAddMappingButtonEnablement = function (roomService) {
            var addRoomButton = $('#hipchat-space-to-room-add');
            var selectedRoom = getSelectedRoom();
            if (getSelectedSpace && selectedRoom && roomService.allowSelectionByName(selectedRoom.roomName)) {
                addRoomButton.removeAttr('aria-disabled').removeAttr('disabled')
            } else {
                addRoomButton.attr({
                    'aria-disabled': 'true',
                    'disabled': ''
                });
            }
        };

        var roomSelect = $('#hipchat-space-to-room-add-select').auiSelect2({
            placeholder: AJS.I18n.getText('hipchat2-space-configuration.select.room'),
            containerCssClass: 'select2-hipchat-room-select',
            dropdownCssClass: 'select2-hipchat-room-dropdown',
            width: '200px',
            query: roomQuery,
            allowClear: true,
            formatResult: roomRenderer
        });
        var selectedRoom = null;

        var $roomSearchInput = $(".select2-hipchat-room-select .select2-input");
        if ($roomSearchInput.length && "placeholder" in $roomSearchInput[0]) {
            $roomSearchInput.attr("placeholder", AJS.I18n.getText("hipchat2-space-configuration.select.room.picker.empty") + " ...");
        }


        roomSelect.on('change', function() {
            if (getSelectedSpace()) {
                enableAddRoomButton();
            }
        });

        roomSelect.on('select2-selected', function(event) {
            selectedRoom = event.choice;
        });

        if(!HipChat.UserLink.context.userLinked) {
            roomSelect.one('select2-opening', function () {
                var dropdown = AJS.$('.select2-hipchat-room-dropdown');
                dropdown.append(Confluence.Templates.HipChat.SpaceToRoom.Config.privateRoomFooter());
                // Select 2 seems to prevent propogation of our links, so need to add a listener higher up the DOM. :/
                var userLink = dropdown.find('.hipchat-user-link');
                userLink.on('click', function() {
                    $("#select2-drop-mask").click();  // close select2 .close() doesn't exist :/
                    HipChat.UserLink.linkUser();
                });
                var footer = dropdown.find('.hipchat-private-room-footer');
                footer.on('hover', function() {
                    // UX nicety - remove hover selection in select2 list
                    dropdown.find('.select2-highlighted').removeClass('select2-highlighted');
                });
            });
        }

        roomServicePromise.done(function(roomService) {
            roomService.onChange(function(rooms) {
                if (!selectedRoom || !roomService.allowSelectionByName(selectedRoom.roomName)) {
                    roomSelect.auiSelect2("val", "");
                }
                updateAddMappingButtonEnablement(roomService);
            })
        });

        var getSelectedRoom = function () {
            return selectedRoom;
        };

        var spaceSelect = $('#hipchat-space-select').auiSelect2(adminSpacePicker.build({
            select2Options: {
                "width":"250px",
                containerCssClass: "select2-hipchat-space-select"
            }
        }));

        var $searchInput = $(".select2-hipchat-space-select .select2-input");
        if ($searchInput.length && "placeholder" in $searchInput[0]) {
            $searchInput.attr("placeholder", AJS.I18n.getText("confluence.plugins.hipchat.admin.space.picker.empty") + " ...");
        }

        spaceSelect.on('change', function() {
            if (getSelectedRoom()) {
                enableAddRoomButton();
            }
        });

        /**
         * Get the currently selected space
         * @returns {{key:string, name:string}}
         */
        function getSelectedSpace() {
            if (space) {
                return space;
            }

            var selectedSpaceData = spaceSelect.auiSelect2("data");
            if (selectedSpaceData) {
                return {
                    key: selectedSpaceData.id,
                    name: selectedSpaceData.text
                };
            }

            return null;
        }

        function editSpaceRoomNotificationConfigLinkHandler(e) {
            e.preventDefault();
            // Only edit one section at a time
            editSpaceRoomNotificationConfig($(this).closest('.hipchat-room-config'));
        }

        function editSpaceRoomNotificationConfig(roomConfigElem) {
            // Only edit one section at a time
            $('.hipchat-room-config').removeClass('edit-active');
            roomConfigElem.addClass('edit-active');
        }

        function closeSpaceRoomNotificationConfig(e) {
            e.preventDefault();
            $(this).closest('.hipchat-room-config').removeClass('edit-active');
        }

        function deleteSpaceRoomNotificationConfig(e) {
            e.preventDefault();
            var config = $(this).closest('.hipchat-room-config');
            var removeData = {
                config: config,
                roomId: config.attr('data-room-id'),
                roomName: config.attr('data-room-name'),
                roomPrivate: config.attr('data-room-private') === "true",
                spaceKey: config.attr('data-space-key')
            };

            function completeDeleteRoomMapping(removeData) {
                var config = removeData.config;
                config.find('input').attr('disabled', 'disabled');
                config.spin("large");
                configService.removeRoomMapping(removeData.spaceKey, removeData.roomId).done(function () {
                    roomServicePromise.done(function(roomService) {
                        roomService.handleMappingToRoomRemoved({ roomId: removeData.roomId, roomName: removeData.roomName, roomPrivate: removeData.roomPrivate });
                    });
                    config.spinStop();
                    config.remove();
                }).fail(function (jqXHR, textStatus, errorThrown) {
                    config.find('input').removeAttr('disabled');
                    config.spinStop();
                    displayUpdateConfigurationError(jqXHR.status, jqXHR.statusText);
                });
            }

            if (config.find(':checked').length) {
                // Mappings still exist confirm with user
                if (!confirmDialog) {
                    // TODO CONFDEV-28519 ADG 2 - new confirmation dialog (and remove related CSS)
                    confirmDialog = new AJS.Dialog(400, 200, "hipchat-remove-mapping-dialog");
                    confirmDialog.addHeader(AJS.I18n.getText("hipchat2-space-configuration.remove.dialog.header"), "remove-warning");
                    confirmDialog.addPanel("Message Panel", "<div class='remove-mapping-message'></div>");
                    confirmDialog.addButton(AJS.I18n.getText("hipchat2-space-configuration.remove.dialog.confirm"), function() {
                        completeDeleteRoomMapping(confirmDialog.removeData);
                        confirmDialog.hide();
                    });
                    confirmDialog.addCancel(AJS.I18n.getText("hipchat2-space-configuration.remove.dialog.cancel"), function() {
                        confirmDialog.hide();
                    });
                }
                // ensure current removeData is available for dialog click handler
                confirmDialog.removeData = removeData;

                $('#hipchat-remove-mapping-dialog .remove-mapping-message').text(AJS.I18n.getText("hipchat2-space-configuration.remove.dialog.message", removeData.roomName));
                confirmDialog.show();
            } else {
                // Just remove
                completeDeleteRoomMapping(removeData);
            }
        }

        // TODO CONFDEV-28519 ADG 2 - new error style?
        function displayUpdateConfigurationError(statusCode, statusText) {
            displayError(AJS.I18n.getText("hipchat2-space-configuration.update.error"), statusCode, statusText);
        }

        function displayError(title, statusCode, statusText) {
            var messageContainer = $('#hipchat-space-to-room-messages');
            messageContainer.empty();
            AJS.messages.error("#hipchat-space-to-room-messages", {
                title: title,
                body: "<p>" + AJS.escapeHtml(AJS.I18n.getText("hipchat2-space-configuration.update.error.reason", statusCode, statusText)) + "</p>"
            });
        }

        function addRoomMappingHandler(e) {
            e.preventDefault();
            var selectedRoom = getSelectedRoom();
            if (selectedRoom) {
                roomServicePromise.done(function(roomService) {
                    var addRoomButton = $('#hipchat-space-to-room-add');
                    addRoomButton.attr('disabled','disabled');
                    var spinner = $("#hipchat-space-to-room-add-mapping-spinner").spin();
                    roomService.getOrCreateRoom(selectedRoom)
                        .done(function (room) {
                            addRoomButton.removeAttr('disabled');
                            spinner.spinStop();
                            addRoomMapping(room);
                        })
                        .fail(function (newRoomName, statusCode, statusText) {
                            addRoomButton.removeAttr('disabled');
                            spinner.spinStop();
                            displayError(AJS.I18n.getText("hipchat2-space-configuration.select.room.create.error.title", newRoomName), statusCode, statusText);
                        });
                });
            }
        }

        function addRoomMapping(room) {
            var space = getSelectedSpace();

            if (space && room) {
                var spaceRoomNotificationConfig = getSpaceRoomNotificationConfig(space, room);
                if (spaceRoomNotificationConfig) {
                    editSpaceRoomNotificationConfig(spaceRoomNotificationConfig);
                } else {
                    addSpaceRoomNotificationConfig(space, room);
                }
            }

            $(".hipchat-integration-steps").trigger('mapping-added.integration-steps');
        }

        function addSpaceRoomNotificationConfig(space, room) {
            if (space && room) {
                roomServicePromise.done(function(roomService) {
                    roomService.handleMappingToRoomAdded(room);
                });
                var compareSortName = room.roomName.toLocaleLowerCase();
                var renderParams = $.extend({
                    spaceName: space.name,
                    spaceKey: space.key,
                    roomState: 'Visible',
                    notificationTypes: notificationTypes
                }, room);
                var roomConfigMarkup = Confluence.Templates.HipChat.SpaceToRoom.Config.renderRoomMapping(renderParams);
                var insertBefore = $('.hipchat-room-config').filter(function(index, el) {
                    var roomState = $(el).attr('data-room-state');

                    if (roomState != 'Visible') {
                        return true;
                    }

                    var currentName = $(el).attr('data-room-name').toLocaleLowerCase();
                    return currentName.localeCompare(compareSortName) > 0;
                }).first();

                pendingSpaceRoomNotificationConfigs.addSpaceRoomNotificationConfig(space, room);

                if(insertBefore.length) {
                    insertBefore.before(roomConfigMarkup);
                } else {
                    $(".hipchat-room-mapping-list").append(roomConfigMarkup);
                }

                /*
                 * To keep this simple, I'm doing a "check" per default notification. So potentially this means n REST calls for
                 * each new room mapping (where n is the number of notifications active by default). In reality this is just one
                 * right now, but could be more.
                 *
                 * We could optimise to do a batch check, but I didn't think was necessary or worth the effort to build a new
                 * REST resource.
                 */
                _.each(notificationTypes, function(notificationType, index) {
                    if(notificationType.activeByDefault) {
                        mapSpaceRoomNotificationType(space.key, room.roomId, notificationType.key, true);
                    }
                });

                editSpaceRoomNotificationConfig(getSpaceRoomNotificationConfig(space, room));
            }
        }

        function getSpaceRoomNotificationConfig(space, room) {
            var spaceRoomNotificationConfig = $('.hipchat-room-config[data-room-id="' + room.roomId + '"][data-space-key="' + space.key + '"]');
            if (spaceRoomNotificationConfig && spaceRoomNotificationConfig.length) {
                return spaceRoomNotificationConfig;
            } else {
                return null;
            }
        }

        function toggleSpaceRoomNotificationType(e) {
            var check = $(this);
            var notifName = check.attr('data-notification-name');
            var config = check.closest('.hipchat-room-config');
            var roomId = config.attr('data-room-id');
            var enable = check.is(':checked');
            var spaceKey = config.attr('data-space-key');
            mapSpaceRoomNotificationType(spaceKey, roomId, notifName, enable);
        }

        function toggleSpaceRoomNotifyClient() {
            var check = $(this);
            var config = check.closest('.hipchat-room-config');
            var roomId = config.attr('data-room-id');
            var enable = check.is(':checked');
            var spaceKey = config.attr('data-space-key');
            setSpaceRoomNotifyClient(spaceKey, roomId, enable);
        }

        function mapSpaceRoomNotificationType(spaceKey, roomId, notifName, enable) {
            var initialLink = pendingSpaceRoomNotificationConfigs.removeRoomBySpaceKeyAndRoomId(spaceKey, roomId);
            var promise = configService.toggleNotification(spaceKey, roomId, notifName, enable, initialLink);

            promise.always(function() {
                AJS.debug('update done for ', spaceKey, roomId, notifName, enable);
            }).fail(function( jqXHR, textStatus, errorThrown ) {
                AJS.debug('select notification failed', jqXHR.statusCode(), textStatus, jqXHR);
                displayUpdateConfigurationError(jqXHR.status, jqXHR.statusText);
            });
        }

        function setSpaceRoomNotifyClient(spaceKey, roomId, enable) {
            var promise = configService.toggleNotifyClient(spaceKey, roomId, enable);

            promise.always(function() {
                AJS.debug('notify client update done for ', spaceKey, roomId, enable);
            }).fail(function( jqXHR, textStatus, errorThrown ) {
                AJS.debug('notify client update failed', jqXHR.statusCode(), textStatus, jqXHR);
                displayUpdateConfigurationError(jqXHR.status, jqXHR.statusText);
            });
        }

        $('#hipchat-space-to-room-add').click(addRoomMappingHandler);

        $('#hipchat-space-to-room-configuration')
                .delegate('.edit-notification', 'click', editSpaceRoomNotificationConfigLinkHandler)
                .delegate('.close-edit-notification', 'click', closeSpaceRoomNotificationConfig)
                .delegate('.trash-room-mapping', 'click', deleteSpaceRoomNotificationConfig)
                .delegate('.notification-type', 'click', toggleSpaceRoomNotificationType)
                .delegate('.notify-client', 'click', toggleSpaceRoomNotifyClient);
    });

});
