/*
 * Object for registering actions in the highlighted text property panel
 */
Confluence.HighlightAction = (function($) {
    // dictionary of all event handlers registered
    var handlers = {};
    var $mainContent;

    // Provide Three default behavior, this will be assign to shouldDisplay method
    // which will be call to validate when will the button of plugin will be shown/hidden
    var WORKING_AREA = {
        MAINCONTENT_AND_COMMENT: function(selectionRange) {
            return Confluence.HighlightAction.RangeHelper.isSelectionInsideContent($('.wiki-content'), selectionRange);
        },
        MAINCONTENT_ONLY: function(selectionRange) {
            // Since the main content is static we do not need to perform the jquery selection again. This is unlike comments which
            // can be loaded via AJAX
            $mainContent = $mainContent || $('.wiki-content').first();
            return Confluence.HighlightAction.RangeHelper.isSelectionInsideContent($mainContent, selectionRange);
        },
        COMMENT_ONLY: function(selectionRange) {
            return Confluence.HighlightAction.RangeHelper.isSelectionInsideContent($('.comment-content'), selectionRange);
        }
    };

    /*
     * PUBLIC Registers a callback for a specific plugin key
     *
     * @param key plugin specific key to register a callback with (eg. com.atlassian.confluence.plugins.confluence-highlight-actions:quote-comment)
     * @param config option object {onClick} register with the associated key
     */
    function registerButtonHandler(key, option) {
        var defaultOption = {
            onClick: function() {},
            // Return FALSE to prevent this plugin's button show on Action Panel
            // if plugin doesn't provide shouldDisplay method, as default it will able to work on Main Content and Comment Area
            shouldDisplay: WORKING_AREA.MAINCONTENT_AND_COMMENT
        };
        handlers[key] = $.extend(defaultOption, option);
    }

    /*
     * PUBLIC Retrieves handler associated with specific plugin key
     *
     * @param key the key for the handler
     * @return the callback associated with the key
     */
    function getButtonHandler(key) {
        var callback = handlers[key];
        if (!callback) {
            callback = function() {
                AJS.logError("The button with key " + key + " doesn't have a registered handler");
            };
        }
        return callback;
    }

    /*
     * PUBLIC Inserts an XML fragment at the end of the current selection
     *
     * @param searchTextObject metadata about the text selection and its position in the page
     * @param insertion the XML fragment to insert in the text
     * @return ajax function
     */
    function insertContentAtSelectionEnd(insertionBean) {
        var restUrl = Confluence.getContextPath() + "/rest/highlighting/1.0/insert-storage-fragment";
        return $.ajax({
            type : "POST",
            contentType : "application/json",
            url : restUrl,
            data : JSON.stringify(insertionBean)
        });
    }

    /*
     * PUBLIC Inserts XML fragments at the end of the table's cells
     *
     * @param tableInsertionBean metadata about:
     *                  the text selection: pageId, numMatches, index, selectedText
     *                  tableColumnIndex: the column index need update in the table
     *                  cellXmlInsertionBeans: list of xml fragment with rowIndex using for update cells
     * @return ajax function
     */
    function insertContentsInTableColumnCells(tableInsertionBean) {
        var restUrl = Confluence.getContextPath() + "/rest/highlighting/1.0/insert-storage-column-table";
        return $.ajax({
            type : "POST",
            contentType : "application/json",
            url : restUrl,
            data : JSON.stringify(tableInsertionBean)
        });
    }

    /*
     * PUBLIC Create tableInsertionBean object use for call
     * insert XML fragments the end of the table's cells
     *
     * @param cellXmlInsertions list of xml fragment with rowIndex using for update cells
     * @param tableColumnIndex the column index need update in the table
     * @param searchText the text selection object received from select text with:
     *                                      pageId, numMatches, index, selectedText
     */
    function createTableInsertionBean(cellXmlInsertions, tableColumnIndex, searchText) {
        var tableInsertionBean = createBaseBean(searchText);
        tableInsertionBean.tableColumnIndex = tableColumnIndex;
        tableInsertionBean.cellModifications = cellXmlInsertions;
        return tableInsertionBean;
    };

    /*
     * DEPRECATED use createXMLModificationBean
     *
     * PUBLIC Create insertionBean object use for call
     * insert XML fragments the end of the table's cells
     *
     * @param xmlModification xml fragment
     * @param searchText the text selection object received from select text with:
     *                                      pageId, numMatches, index, selectedText
     */
    function createInsertionBean(cellXmlInsertions, searchText) {
        var insertionBean = createBaseBean(searchText);
        insertionBean.xmlModification = cellXmlInsertions[0].xmlInsertion;
        return insertionBean;
    }

    /*
     * PUBLIC Create insertionBean object use for call
     * insert XML fragments the end of the table's cells
     *
     * @param xmlModification xml fragment
     * @param searchText the text selection object received from select text with:
     *                                      pageId, numMatches, index, selectedText
     */
    function createXMLModificationBean(xml, searchText) {
        var insertionBean = createBaseBean(searchText);
        insertionBean.xmlModification = xml;
        return insertionBean;
    }

    /*
     * PUBLIC Removes the current mouse highlight from the page
     */
    function clearTextSelection() {
        if (window.getSelection) {
            window.getSelection().empty && window.getSelection().empty(); // Chrome
            window.getSelection().removeAllRanges && window.getSelection().removeAllRanges(); //FF
        } else {
            window.document.selection && window.document.selection.empty(); // IE
        }
    }

    function createBaseBean(searchText) {
        var insertionBean = {};
        insertionBean.pageId = searchText.pageId;
        insertionBean.selectedText = searchText.selectedText;
        insertionBean.index = searchText.index;
        insertionBean.numMatches = searchText.numMatches;
        insertionBean.lastFetchTime = getLastFetchTime();
        return insertionBean;
    }

    /*
    * Get last fetch time of current page.
    * */
    function getLastFetchTime() {
        return $("meta[name='confluence-request-time']").attr("content");
    }

    return {
        registerButtonHandler: registerButtonHandler,
        getButtonHandler: getButtonHandler,
        insertContentAtSelectionEnd: insertContentAtSelectionEnd,
        insertContentsInTableColumnCells: insertContentsInTableColumnCells,
        createTableInsertionBean: createTableInsertionBean,
        createInsertionBean: createInsertionBean,
        createXMLModificationBean: createXMLModificationBean,
        clearTextSelection: clearTextSelection,
        WORKING_AREA: WORKING_AREA
    };
})(AJS.$);

/**
 * Provide a useful method to interact with Range
 */
Confluence.HighlightAction.RangeHelper = (function($) {
    /**
     * Return an Object which provide all available variable for plugins
     * @param selectionRange
     * @returns {{area: *, text: *, html: *, containingElement: *, range: *}}
     */
    function getRangeOption(selectionRange) {
        return {
            area: getSelectionRects(selectionRange),
            text: getSelectionText(selectionRange),
            html: getSelectionHTML(selectionRange),
            containingElement: getContainingElement(selectionRange),
            range: selectionRange
        };
    }

    /*
     * Returns an object representing a rectangle. the rectangle coordinates are as follows
     * left: start of the text highlight
     * right: either the end of the highlight of the first line or the end of the highlight of the last line, whichever is greater
     * top: top of highlighted text
     * bottom: bottom of highlighted text
     *
     * @param selectionRange range object representing the selected text
     * @return rect object or false if error occurs
     */
    function getSelectionRects(selectionRange) {
        // if using documentation theme, get scrolltop of documentation theme content which is fixed size
        // certain browsers give the scrollTop on html element, some give it on the body element. this works in both.
        var scrollTop = Confluence.DocThemeUtils.getMainContentScrollTop();
        var scrollLeft = Confluence.DocThemeUtils.getMainContentScrollLeft();

        var clientRects = selectionRange.getClientRects();
        // IE fails to provide client rects when highlighting a paragraph containing only an image or triple clicking to
        // highlight any paragraph containing content other than plain text. In this case, it is safe to assume parent
        // element dimensions will represent the highlighted area
        if (!clientRects.length && selectionRange.parentElement()) {
            var $parentElement = $(selectionRange.parentElement());
            var parentOffset = $parentElement.offset();
            clientRects = [{
                top: parentOffset.top - ((Confluence.DocThemeUtils.isDocTheme()) ? 0 : scrollTop),
                left: parentOffset.left - ((Confluence.DocThemeUtils.isDocTheme()) ? 0 : scrollLeft),
                bottom: parentOffset.top + $parentElement.height(),
                right: parentOffset.left + $parentElement.width()
            }];
        }
        var rects = getFirstAndLastSelectionRect(selectionRange, clientRects);

        /*
         * Calculates Create Issue dialog target area
         */
        var getOverlap = function(firstRect, lastRect) {
            var overlap = {};
            overlap.top = firstRect.top;
            overlap.left = firstRect.left + scrollLeft;
            overlap.bottom = lastRect.bottom;

            if (firstRect.left >= lastRect.right) {
                overlap.right = firstRect.right;
            } else {
                overlap.right = lastRect.right;
            }
            overlap.right = overlap.right + scrollLeft;
            // adjust top for doc theme
            overlap.top = overlap.top + scrollTop;
            overlap.bottom = overlap.bottom + scrollTop;
            // set width and height
            overlap.width = overlap.right - overlap.left;
            overlap.height = overlap.bottom - overlap.top;

            return overlap;
        };

        /*
         * Calculates the action panel target area
         */
        var getHighlightStart = function(rect) {
            var highlight = {};
            highlight.width = rect.right - rect.left;
            highlight.height = rect.bottom - rect.top;
            highlight.left = rect.left + scrollLeft;
            highlight.right = rect.right + scrollLeft;
            highlight.top = rect.top + scrollTop;
            highlight.bottom = rect.bottom + scrollTop;
            return highlight;
        }

        /*
         * Adjusts coordinates if using documentation theme to be relative to documentation theme scrolling content
         * container, otherwise coordinates are relative to the viewport
         */
        var adjustRectForDocTheme = function(rect) {
            // if using documentation theme, calculate position relative to docThemeContainer not window
            if (Confluence.DocThemeUtils.isDocTheme()) {
                var docContainerOffset = Confluence.DocThemeUtils.getDocThemeContentElement().offset();
                rect.left = rect.left - docContainerOffset.left;
                rect.right = rect.right - docContainerOffset.left;
                rect.top = rect.top - docContainerOffset.top;
                rect.bottom = rect.bottom - docContainerOffset.top;
            }
            return rect;
        };


        var averageRect = adjustRectForDocTheme(getOverlap(rects.first, rects.last));
        var firstRect = adjustRectForDocTheme(getHighlightStart(rects.first));

        // Some Debugging output. Turn on by typing Confluence.HighlightAction.debug = true in console after page loads
        if (Confluence.HighlightAction.debug) {
            var $highlight_debug = $("<div>").attr("id", "highlight-actions-debug-helper");
            Confluence.DocThemeUtils.appendAbsolutePositionedElement($highlight_debug).css($.extend({position: 'absolute', outline: '1px solid red'}, averageRect));
        }

        return {
            first: firstRect,
            average: averageRect
        };
    }

    /*
     * Return the text contained within the range object
     *
     * @param W3C text range object
     * @return highlighted text stripped of DOM elements
     */
    function getSelectionText(selectionRange) {
        /*
         The selection should always return the textContent as we will be comparing the selectionText to the text content
         of the page (for example, findOccurences uses text content).
         */
        var selectionText = (selectionRange.text != undefined) ? selectionRange.text : selectionRange.cloneContents().textContent;
        /*
         CONF-36789: on IE, selectionRange.toString() always convert all &nbsp; to normal space
         on other browsers, it does not. So we must convert manually to made consistency across all browsers
         */
        return convertSpaces(selectionText);
    }

    /**
     * Return the HTML string contained within the range object
     * @param selectionRange: W3C text range object
     */
    function getSelectionHTML(selectionRange) {
        return (selectionRange.cloneContents) ?
            $('<div>').append(selectionRange.cloneContents()).html() :
            selectionRange.htmlText; // IE8 uses TextRange object with htmlText property
    }

    /*
     * Find deepest common ancestor container of a selection ranges boundary-points
     *
     * @param selectionRange current text selection on the page
     * @return deepest DOM element encompassing the entire selection
     */
    function getContainingElement(selectionRange) {
        if (selectionRange.commonAncestorContainer) {
            var selectRangeElement = selectionRange.commonAncestorContainer;
            if(selectRangeElement.nodeType == 3) {//Is TextNode
                return selectRangeElement.parentNode;
            }
            return selectRangeElement;
        } else if (selectionRange.parentElement) {// IE <= 8
            return selectionRange.parentElement();
        }
    }

    /*
     * Given a selectionRange object and clientRects representing the highlighted text, return the first and last
     * clientRect containing text accounting for browser incompatibilities
     *
     * @param selectionRange selected text range object
     * @param clientRects browser clientRects representing the highlighted text
     * @return an object containing the first and last clientRect of the highlighted text
     */
    function getFirstAndLastSelectionRect(selectionRange, clientRects) {
        // In Chrome triple clicking text will cause all the text in an element to be selected. When calling
        // getClientRects after a triple click, the array will contain rects for each line of highlighted text
        // in the highlighted element and also a clientRects for the adjacent sibling element even though no text
        // is highlighted inside it. If the endOffset is zero, we ignore this final clientRect because no text
        // is highlighted inside of it.
        //
        // Also, in Chrome, Safari and Firefox, get client rects return a clientRects representing each inline element
        // including rectangles for all nested elements. IE8 return many less rects, maybe just one for the ancestor
        // element containing the highlight or a clientRects for each line of highlight without individual rects for each
        // inline element
        var rects = {};
        rects.first = clientRects[0];
        rects.last = clientRects[clientRects.length - 1];
        if (selectionRange.endOffset !== 'undefined') {
            //In IE, if we highlight on resolved inline comment, we will length of clientRects is 1
            if (selectionRange.endOffset == 0 && clientRects.length > 1) {
                rects.last = clientRects[clientRects.length - 2];
            }
        }
        return rects;
    }

    /*
     * Return valid W3C range object for the current selection the content, otherwise return false
     *
     * @param content the content element within which a selection must be contained
     * @return W3C range object for the current selection on the page or false if not defined
     */
    function getUserSelectionRange() {
        // no selection made webkit
        if (window.getSelection && window.getSelection().isCollapsed) {
            return false;
        }

        // no selection made IE
        if (document.selection && (document.selection.type == "None" || document.selection.createRange().htmlText == "")) {
            return false;
        }

        var selectionRange;
        if (window.getSelection) {
            // Firefox support multi range, we should get the last range to support Selenium test
            var range = window.getSelection();
            selectionRange = range.getRangeAt(range.rangeCount - 1);
        } else if (document.selection) {
            selectionRange = document.selection.createRange();
        }

        // don't show the highlight panel if the selection is all whitespaces
        if (/^\s*$/.test(getSelectionText(selectionRange))) {
            var html = getSelectionHTML(selectionRange);
            if (!html) {
                return false;
            }
            // we support to quote image, need to check before return false
            var hasImage = html.toLowerCase().indexOf('<img ') != -1;
            // case not show return false
            if(!hasImage) {
                return false;
            }
        }

        // verify that selection is inside wiki-content
        if (!isSelectionInsideContent($('.wiki-content'), selectionRange)) {
            return false;
        }
        return selectionRange;
    }

    /**
     * Check if selectionRange is valid inside the Content or not
     * @param $content
     * @param selectionRange
     * @returns True if selectionRange is Content or child of Content.
     */
    function isSelectionInsideContent($content, selectionRange) {
        var selectionContainer = getContainingElement(selectionRange);
        var isContent = function() {
            var isValid = false;
            $.each($content, function(index, element) {
                // may be $element is container
                // or if $element is contained inside $content
                if (element === selectionContainer || $.contains(element, selectionContainer)) {
                    isValid = true;
                    return false; // return false to cancel the loop
                }
            });
            return isValid;
        };

        return isContent();
    }

    /*
     * Generates javascript object containing the context of the selected text to help locate in storage format
     *
     * @param $root jQuery element, container of content which can be selected
     * @param selected range object representing the current selection
     * @return object containing metadata about the location of the selected text
     */
    function computeSearchTextObject($root, selected) {
        var fromStart = getSelectionText(extendRangeToStart($root, selected));
        var selectedText = $.trim(getSelectionText(selected));
        var occurrences = findOccurrences(selectedText, $root);

        /*
        CONF-36789: b/c we trim selectedText, fromStart may contains spaces at the end
        we must remove these spaces to make sure fromStart.length is correct
         */
        fromStart = fromStart.replace(/\s*$/, "");

        return {
            pageId: AJS.Meta.get("page-id"),
            selectedText: selectedText,
            index: $.inArray(fromStart.length - selectedText.length, occurrences),
            numMatches: occurrences.length
        };
    }

    /*
     * Return the text content of an element and all descendants
     *
     * @param $root jQuery element whose text content we're interested in
     * @return text content of the passed element
     */
    function getTextContent($root) {
        if (document.createRange) {
            return $root.text();
        } else {
            // IE8 $root.text() doesn't count line ending character (\n). IE8 collapses all whitespace (including)
            // line break into a single space so range.text returns a more accurate text
            range = document.body.createTextRange();
            range.moveToElementText($root.get(0));
            return range.text;
        }
    }

    /*
     * Creates a range containing all text up to the selection text
     *
     * @param $root jQuery element, container of content which can be selected
     * @param selected range object representing the current selection
     * @return range object encompassing all text inside $root up to selected
     */
    function extendRangeToStart($root, selected) {
        var range;
        if (document.createRange) {
            range = document.createRange();
            range.setStart($root.get(0), 0);
            // use originalEndOffset to work around IE issue, since endOffset property of selection object
            // is modified internally by the IE if there's some DOM changes.
            range.setEnd(selected.endContainer, selected.originalEndOffset);
        } else { // IE8
            range = document.body.createTextRange();
            range.moveToElementText($root.get(0));
            range.setEndPoint("EndToEnd", selected);
        }
        return range;
    }

    /*
     * Finds all occurrences of a substring inside a string
     *
     * @param src the string to search
     * @param sub the substring to find
     * @return array of indexes where the substring occurs
     */
    function findOccurrences(selectedText, $root) {

        var pageContent = getTextContent($root);
        pageContent = updatePageContent(selectedText, $root.clone(), pageContent);

        //CONF-36789: convert &nbsp; to normal space, so compare selectedText with pageContent correctly
        pageContent = convertSpaces(pageContent);

        var start = 0;
        var found = -1;
        var indexes = [];
        while ((found = pageContent.indexOf(selectedText, start)) > -1) {
            indexes.push(found);
            start = found + 1;
        }
        return indexes;
    }

    //Replace all texts in macro same with highlight text by space
    function updatePageContent(selectedText, $root, pageContent) {

        var $macros = $root.find('.user-mention, a[href^="/"]');

        $root.find('.conf-macro[data-hasbody="false"], .jira-issue, .jira-issues').each(function() {
            if ($(this).text().indexOf(selectedText) > -1) {
                $macros = $macros.add(this);
            }
        });

        if($macros.length > 0) {
            // string of spaces the same length as the original selectedText
            var replacedText = selectedText.replace(/\S/g, " ");

            // create regular expression from selected text where all special characters are escaped with "\"
            var re = new RegExp(selectedText.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&'), 'g');

            $macros.each(function() {
                // replace occurrences of selectedText within the macro with string of spaces (replacedText), so we don't
                // find these occurrences of selectedText
                var originalText = $(this).text();
                $(this).text(originalText.replace(re, replacedText));
            });

            return getTextContent($root);
        }

        return pageContent;
    }

    /**
     * Convert all non-break spaces <code>u00a0</code> to normal spaces <code>u0020</code>
     * @param {string} str string to be converted
     * @returns {string} result string
     */
    function convertSpaces(str) {
        return str.replace(/\u00a0/g,"\u0020");
    }

    return {
        getRangeOption: getRangeOption,
        getUserSelectionRange: getUserSelectionRange,
        getSelectionRects: getSelectionRects,
        getSelectionText: getSelectionText,
        getSelectionHTML: getSelectionHTML,
        getContainingElement: getContainingElement,
        getFirstAndLastSelectionRect: getFirstAndLastSelectionRect,
        isSelectionInsideContent: isSelectionInsideContent,
        computeSearchTextObject: computeSearchTextObject
    };
})(AJS.$);

AJS.toInit(function($) {
    // Usable area for the plugin. #main-content was another candidate, but does not appear on all pages
    // We only support on main content
    var $wikiContent = $('.wiki-content').first();

    var NodeTypes = {
        ELEMENT_NODE: 1,
        TEXT_NODE: 3
    };

    var NodeNames = {
        IMAGE: 'IMG'
    };

    // pushpin object to mark locations where property panel should open
    var $actionDialogTarget = $("<div>").attr("id", "action-dialog-target");

    // inline dialog containing highlight action buttons
    var actionPanel;
    var actionPanelClass = "selection-action-panel";

    var selectionRange;

    var curRects;

    function init() {
        // endpoint for retrieving available highlight actions
        var restUrl = Confluence.getContextPath() + "/rest/highlighting/1.0/panel-items";
        var pageId = AJS.Meta.get("page-id");
        if(pageId != undefined) {
            restUrl = restUrl + "?pageId=" + pageId;
        }

        var actionsPromise = $.get(restUrl, function(data) {
            if (data.length) {
                createPropertyPanel(data);
            }
        });

        bindHandlers(actionsPromise);
    }

    /*
     * Renders the property panel
     *
     * @param data JSON representation of registered highlight actions
     */
    function createPropertyPanel(data) {
        var dialogEventHandlers = getPanelEventHandlers();
        // width of each button in the property panel
        // CONFDEV-19613: On new version, they add one more pixel as margin-right, increase this variable one more pixel to fix icon wrapped problem, we need to get this dynamic in the future
        var ICON_BUTTON_WIDTH = 29;
        var buttonClicked = false;
        var panelContentWidth = data.length * ICON_BUTTON_WIDTH;

        var panelHTML = Confluence.HighlightPanel.Templates.panelContent({webItems: data});
        var panelExists = false;
        var generateDialogContent = function(content, trigger, showPopup) {
            if (!panelExists) {
                content.append(panelHTML);
                //apply chunky tooltips for buttons
                content.find(".aui-button").tooltip({gravity: "s"});
                setPanelUnselectable(content.parent());
                // popup property panel button handler, gathers information about the selection and triggers
                // the registered callback of the button preseed
                content.find("button").click(function(event) {
                    var key = $(this).attr("data-key");
                    var pluginOption = Confluence.HighlightAction.getButtonHandler(key);
                    buttonClicked = true;
                    actionPanel.hide();
                    var argument = Confluence.HighlightAction.RangeHelper.getRangeOption(selectionRange);
                    if ($.trim(argument.text) !== "") {
                        argument.searchText = Confluence.HighlightAction.RangeHelper.computeSearchTextObject($wikiContent, selectionRange);
                    }
                    pluginOption.onClick(argument);
                });
            }
            showPopup();

            panelExists = true;

            return false;
        };

        // This function will be call before an icon button is show up on Action Dialog,
        // it will involve shouldDisplay method of each plugin. if this function return false, icon button will be hidden.
        var onBeforeShow = function(popup) {
            var shouldShowPopup = false;
            popup.find("button").each(function(index){
                var $button = $(this);
                var key = $button.attr("data-key");
                var pluginOption = Confluence.HighlightAction.getButtonHandler(key);
                var visible = pluginOption.shouldDisplay(selectionRange);
                $button.css('display', visible ? 'inline-block' : 'none');

                // this buttonVisible variable to determine if there is one button is visible
                shouldShowPopup = shouldShowPopup || visible;
            });
            // what happen if there isn't any button is visible? let hide the dialog
            if (!shouldShowPopup) {
                actionPanel.hide();
            } else {
                // Make sure that the width of dialog is always perfect after show/hide icons
                popup.find('.contents').width('auto');
            }
        };

        var initCallback = function() {
            // Send analytics when highlight popup appears
            Confluence.HighlightAction.Analytics.sendAnalyticsForOpeningHighlightOptionPanel();
            // Some plugins is limit working scope, check to hide before them are shown
            onBeforeShow(this.popup);
            dialogEventHandlers.bindHideEvents();
            $actionDialogTarget.show();
        };

        var hideCallback = function() {
            dialogEventHandlers.unbindHideEvents();
            $actionDialogTarget.hide();
        };

        var dialogOptions = {
            centerOnHighlight: true,
            onTop: true,
            fadeTime: 0,
            width: panelContentWidth,
            persistent: true,
            // Keep the property panel open unless text is deselected or a panel button is clicked
            // return value of true closes the dialog, return value of false stops the dialog from closing
            initCallback: initCallback,
            hideCallback: hideCallback
        };

        actionPanel = Confluence.ScrollingInlineDialog($actionDialogTarget, actionPanelClass, generateDialogContent, dialogOptions);
    }

    function setPanelUnselectable($panelContent) {
        // makes panel unselectable on IE8
        // prevent dragging event on Chrome
        $panelContent.children().attr('unselectable', 'on').on('selectstart', false);
    }

    /*
     * Bind mouse handlers for highlight actions
     */
    function bindHandlers(actionsPromise) {
        var panelTimeoutId;
        var NO_DELAY = 0;
        var DELAY_1_SECOND = 1000;

        $(document).on('mouseup', function (e) {
            // Only fire the callback once we have fetched the highlight panels actions using ajax
            actionsPromise.done(function(data) {
                //Panel won't exist if user can't perform any actions on the page
                if (!(data && data.length > 0)) {
                    return;
                }

                var $target = $(e.target);
                // We need to ignore mouseup events that occur in the inline dialogs
                if ($target.closest('.aui-inline-dialog').length !== 0) {
                    return;
                }
                // The following code is wrapped in a setTimeout of 0 because when a user clicks to dismiss highlighted text
                // in Chrome, the mouseup event will fire before the selection is cleared, causing the property panel to
                // remain visible even though the selection is dismissed. The setTimeout gives the browser the needed time
                // to clear selected text before executing the mouseup handler
                setTimeout(function () {
                    clearTimeout(panelTimeoutId);

                    var panelDisplayDelay = DELAY_1_SECOND;
                    if ($(actionPanel[0]).is(":visible")) {
                        panelDisplayDelay = NO_DELAY;
                    }
                    panelTimeoutId = setTimeout(function () { // actions dialog appearing for first time, delay then show
                        displayPropertyPanel();
                    }, panelDisplayDelay);
                }, NO_DELAY);
            });
        });

        actionsPromise.done(function() {
            // hide the property panel when invoking quick edit
            AJS.bind('quickedit.success', function () {
                actionPanel.hide();
            });
        });
    }

    function displayPropertyPanel() {
        selectionRange = Confluence.HighlightAction.RangeHelper.getUserSelectionRange();

        var isSelectionContainsText = function (_selectionRange) {
            return $.trim(_selectionRange.toString()) !== '';
        };

        selectionRange.originalEndOffset = selectionRange.endOffset;

        if (!selectionRange || !isSelectionContainsText(selectionRange)) {
            actionPanel.hide();
            return;
        }
        var selectionRects = Confluence.HighlightAction.RangeHelper.getSelectionRects(selectionRange);
        if (!selectionRects) {
            return;
        }

        var isInNewPosition = positionDialogTarget(selectionRects);
        if (isInNewPosition || !$(actionPanel[0]).is(":visible")) {
            $(actionPanel[0]).hide();
            actionPanel.show();
        }

    }

    /*
     * Provides event handlers that close the dialog on an external click if no text is selected on the page
     */
    function getPanelEventHandlers() {
        var bindHideEvents = function() {
            bindHideOnExternalClick();
            bindHideOnEscPressed();
        };

        var unbindHideEvents = function() {
            unbindHideOnExternalClick();
            unbindHideOnEscPressed();
        };

        // Be defensive and make sure that we haven't already bound the event
        var hasBoundOnExternalClick = false;
        var externalClickNamespace = actionPanelClass + ".inline-dialog-check";

        /**
         * Catch click events on the body to see if the click target occurs outside of this popup
         * If it does, the popup will be hidden
         */
        var bindHideOnExternalClick = function () {
            if (!hasBoundOnExternalClick) {
                $("body").bind("click." + externalClickNamespace, function(e) {
                    var $target = $(e.target);
                    // hide the popup if the target of the event is not in the dialog
                    if ($target.closest('#inline-dialog-' + actionPanelClass + ' .contents').length === 0) {
                        if (!selectionRange) {
                            actionPanel.hide();
                        }
                    }
                });
                hasBoundOnExternalClick = true;
            }
        };

        var unbindHideOnExternalClick = function () {
            if (hasBoundOnExternalClick) {
                $("body").unbind("click." + externalClickNamespace);
            }
            hasBoundOnExternalClick = false;
        };

        var onKeydown = function(e) {
            if (e.keyCode === 27) {
                actionPanel.hide();
            }
        };

        var bindHideOnEscPressed = function() {
            $(document).on("keydown", onKeydown);
        };

        var unbindHideOnEscPressed = function() {
            $(document).off("keydown", onKeydown);
        };

        return {
            bindHideEvents: bindHideEvents,
            unbindHideEvents: unbindHideEvents
        }
    }

    /*
     * Positions the dialog target relative to the selection
     *
     * @param the rects defining the selection area
     * @return boolean value indicating whether the selection position has changed
     */
    function positionDialogTarget(selectionRects) {
        Confluence.DocThemeUtils.appendAbsolutePositionedElement($actionDialogTarget);
        var posChanged = false;
        if (!curRects || selectionRects.first.top != curRects.first.top || selectionRects.first.height != curRects.first.height ||
            selectionRects.first.left != curRects.first.left || selectionRects.first.width != curRects.first.width) {
            $actionDialogTarget.css({
                top: selectionRects.first.top,
                height: selectionRects.first.height,
                left: selectionRects.first.left,
                width: selectionRects.first.width
            });
            curRects = selectionRects;
            posChanged = true;
        }
        return posChanged;
    }

    init();
});
