package com.atlassian.confluence.plugins.createcontent.scheduled;

import bucket.user.propertyset.BucketPropertySetItem;
import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.confluence.core.ContentEntityManager;
import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.core.ContentPropertyManager;
import com.atlassian.confluence.plugins.createcontent.api.services.ContentBlueprintSanitiserManager;
import com.atlassian.confluence.plugins.createcontent.rest.entities.CreateBlueprintPageRestEntity;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintPageEntity;
import com.atlassian.confluence.search.service.ContentTypeEnum;
import com.atlassian.hibernate.PluginHibernateSessionFactory;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import com.atlassian.scheduler.JobRunner;
import com.atlassian.scheduler.JobRunnerRequest;
import com.atlassian.scheduler.JobRunnerResponse;
import net.sf.hibernate.HibernateException;
import net.sf.hibernate.Query;
import net.sf.hibernate.Session;
import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import javax.annotation.Nullable;
import java.io.IOException;
import java.util.List;
import java.util.Optional;

import static com.atlassian.confluence.plugins.createcontent.impl.DefaultRequestStorage.DRAFT_CREATE_REQUEST;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * Scheduled job to clean up {@link CreateBlueprintPageEntity}'s stored in OS_PROPERTYENTRY.
 * These entities provide additional context for pages being created from blueprints, but they can contain sensitive personal information.
 * Done as part of GDPR compliance.
 * <p>
 * This task runs in batches to clean up old entries and stores another entity with the same ID and different key to indicate that the entry in question
 * has been 'cleaned'.
 * <p>
 * It runs indefinitely as an old import may contain more records to clean.
 *
 * @since 10.0.0
 */
@Component("createBlueprintPageEntityCleanupJob")
public class CreateBlueprintDraftPageEntityCleanupJob implements JobRunner {

    private static final Logger log = getLogger(CreateBlueprintDraftPageEntityCleanupJob.class);

    private static final String JOB_BATCH_SIZE_KEY = CreateBlueprintDraftPageEntityCleanupJob.class.getSimpleName() + "-batchSize";
    public static final String CLEANED_RECORD_KEY = "create.blueprint.page.draft.cleaned";

    /**
     * This is a String that is used to 'mark' the version of this cleanup task. Doing this because in the future
     * we may find more private identifiable information (PII) present in these blueprintentities that needs to be cleaned
     * up and so this cleanup job requires updating. Changing this string (preferably keeping it as a sequentially increasing number)
     * will mean entries previously marked as 'processed' will be visible for cleanup again.
     * I'm sorry that it's a String, but the ContentPropertyManager only supports String and Text properties, not numbers.
     */
    @VisibleForTesting
    static final String LATEST_CLEANED_VERSION = "3";

    private static final int DEFAULT_BATCH_SIZE = 1000;

    private final String getEntitiesToCleanupQueryString = "select item " +
            "from BucketPropertySetItem item " +
            "where item.key = :entityKey and not exists ( " +
            "select 1 " +
            "from BucketPropertySetItem item2 " +
            "where item.entityId = item2.entityId and item2.key = :cleanedRecordKey and item2.stringVal = :cleanVal" +
            ")";

    private final TransactionTemplate transactionTemplate;
    private final PluginHibernateSessionFactory pluginHibernateSessionFactory;
    private final ContentBlueprintSanitiserManager sanitiserManager;
    private final ContentPropertyManager contentPropertyManager;
    private final ContentEntityManager contentEntityManager;

    @Autowired
    public CreateBlueprintDraftPageEntityCleanupJob(
            @ComponentImport PluginHibernateSessionFactory pluginHibernateSessionFactory,
            @ComponentImport TransactionTemplate transactionTemplate,
            ContentBlueprintSanitiserManager sanitiserManager,
            @ComponentImport ContentPropertyManager contentPropertyManager,
            @ComponentImport @Qualifier("contentEntityManager") ContentEntityManager contentEntityManager) {
        this.transactionTemplate = transactionTemplate;
        this.pluginHibernateSessionFactory = pluginHibernateSessionFactory;
        this.sanitiserManager = sanitiserManager;
        this.contentPropertyManager = contentPropertyManager;
        this.contentEntityManager = contentEntityManager;
    }

    @Nullable
    @Override
    public JobRunnerResponse runJob(JobRunnerRequest jobRunnerRequest) {
        return transactionTemplate.execute(() -> {
            try {
                Session session = pluginHibernateSessionFactory.getSession();
                List<BucketPropertySetItem> propertySetItems = getEntitiesToCleanup(session);
                if (propertySetItems.size() > 0) {
                    int totalCleaned = performCleanup(session, propertySetItems);
                    String summaryMessage = String.format("Cleaned up %d entries in %d ms", totalCleaned, System.currentTimeMillis() - jobRunnerRequest.getStartTime().getTime());
                    return JobRunnerResponse.success(summaryMessage);
                }
                return JobRunnerResponse.success("No entries were found to cleanup");
            } catch (HibernateException e) {
                return JobRunnerResponse.failed(e);
            }
        });
    }

    private List<BucketPropertySetItem> getEntitiesToCleanup(Session session) throws HibernateException {
        int batchSize = getCurrentBatchSize().orElse(DEFAULT_BATCH_SIZE);
        Query getEntitiesToCleanupQuery = session.createQuery(getEntitiesToCleanupQueryString)
                .setParameter("entityKey", DRAFT_CREATE_REQUEST)
                .setParameter("cleanedRecordKey", CLEANED_RECORD_KEY)
                .setParameter("cleanVal", LATEST_CLEANED_VERSION)
                .setMaxResults(batchSize);

        @SuppressWarnings("unchecked")
        List<BucketPropertySetItem> propertySetItems = (List<BucketPropertySetItem>) getEntitiesToCleanupQuery.list();

        return propertySetItems;
    }

    @VisibleForTesting
    int performCleanup(Session session, List<BucketPropertySetItem> entriesToClean) throws HibernateException {
        int cleanedCount = 0;
        ObjectMapper objectMapper = new ObjectMapper();
        for (BucketPropertySetItem propertySetItem : entriesToClean) {
            ContentEntityObject ceo = contentEntityManager.getById(propertySetItem.getEntityId());
            if (ceo != null) {
                if (ceo.isDraft() || ContentTypeEnum.DRAFT.equals(ceo.getTypeEnum())) {
                    String entityString = propertySetItem.getTextVal();
                    try {
                        CreateBlueprintPageEntity entity = objectMapper.readValue(entityString, CreateBlueprintPageRestEntity.class);
                        propertySetItem.setTextVal(objectMapper.writeValueAsString(sanitiserManager.sanitise(entity)));
                        session.save(propertySetItem);
                        contentPropertyManager.setStringProperty(ceo, CLEANED_RECORD_KEY, LATEST_CLEANED_VERSION);
                        cleanedCount++;
                    } catch (IOException e) {
                        log.error("Could not process the CreateBlueprintPageEntity for cleanup: ", e);
                    }
                } // skip (non-draft) entries to be cleaned up by PropertyEntryGardeningJob
            } else {
                // remove propertySetItems that aren't associated with a CEO anymore
                session.delete(propertySetItem);
                cleanedCount++;
            }
        }
        return cleanedCount;
    }

    @VisibleForTesting
    Optional<Integer> getCurrentBatchSize() {
        return Optional.ofNullable(System.getProperty(JOB_BATCH_SIZE_KEY))
                .map(Integer::parseInt);
    }
}
