package com.atlassian.confluence.plugins.createcontent.rest;

import com.atlassian.confluence.api.model.accessmode.AccessMode;
import com.atlassian.confluence.api.service.accessmode.AccessModeService;
import com.atlassian.confluence.api.service.exceptions.ReadOnlyException;
import com.atlassian.confluence.languages.LocaleManager;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.ResourceErrorType;
import com.atlassian.confluence.plugins.createcontent.exceptions.ResourceException;
import com.atlassian.confluence.plugins.createcontent.rest.entities.CreateDialogWebItemEntity;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.user.PersonalInformationManager;
import com.atlassian.confluence.util.i18n.DocumentationBean;
import com.atlassian.confluence.util.i18n.DocumentationBeanFactory;
import com.atlassian.confluence.util.i18n.I18NBean;
import com.atlassian.confluence.util.i18n.I18NBeanFactory;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.plugins.rest.common.security.AnonymousAllowed;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.core.Response;
import java.util.List;

@Path("/space-blueprint/dialog")
public class SpaceBlueprintDialogResource extends AbstractRestResource {
    private final PermissionManager permissionManager;
    private final I18NBeanFactory i18NBeanFactory;
    private final LocaleManager localeManager;
    private final DocumentationBeanFactory documentationBeanFactory;
    private final BlueprintWebItemService webItemService;
    private final PersonalInformationManager personalInformationManager;
    private final AccessModeService accessModeService;

    public SpaceBlueprintDialogResource(
            final @ComponentImport PermissionManager permissionManager,
            final @ComponentImport SpaceManager spaceManager,
            final @ComponentImport I18NBeanFactory i18NBeanFactory, LocaleManager localeManager,
            final @ComponentImport DocumentationBeanFactory documentationBeanFactory,
            final BlueprintWebItemService webItemService,
            final @ComponentImport PersonalInformationManager personalInformationManager,
            final @ComponentImport AccessModeService accessModeService) {
        super(permissionManager, spaceManager, accessModeService);

        this.permissionManager = permissionManager;
        this.i18NBeanFactory = i18NBeanFactory;
        this.localeManager = localeManager;
        this.documentationBeanFactory = documentationBeanFactory;
        this.webItemService = webItemService;
        this.personalInformationManager = personalInformationManager;
        this.accessModeService = accessModeService;
    }

    @GET
    @Path("web-items")
    @AnonymousAllowed
    public List<CreateDialogWebItemEntity> getWebItems() {
        if (AccessMode.READ_ONLY == accessModeService.getAccessMode()) {
            throw new ReadOnlyException("Read only mode is enabled.");
        }
        final ConfluenceUser remoteUser = getUser();
        boolean canCreateSpaces = userCanCreateSpace(remoteUser);
        boolean canCreatePersonalSpaces = userCanCreatePersonalSpace(remoteUser);
        if (!canCreateSpaces && !canCreatePersonalSpaces) {
            // This will only occur if the user hand-crafts a request, the user's permissions have changed since opening
            // the Create dialog, or if the User's session has expired and the remote user is null.
            throw new ResourceException("You are not permitted to create spaces or personal spaces.", Response.Status.FORBIDDEN,
                    remoteUser == null ? ResourceErrorType.PERMISSION_ANONYMOUS_CREATE_SPACE : ResourceErrorType.PERMISSION_USER_CREATE_SPACE,
                    remoteUser == null ? null : remoteUser.getName());
        }

        final I18NBean i18NBean = i18NBeanFactory.getI18NBean(localeManager.getLocale(remoteUser));
        final DocumentationBean documentationBean = documentationBeanFactory.getDocumentationBean();

        return !canCreateSpaces
                ? webItemService.getCreatePersonalSpaceWebItems(i18NBean, documentationBean, remoteUser)
                : webItemService.getCreateSpaceWebItems(i18NBean, documentationBean, remoteUser);
    }

    private boolean userCanCreateSpace(ConfluenceUser user) {
        return permissionManager.hasCreatePermission(user, PermissionManager.TARGET_APPLICATION, Space.class);
    }

    private boolean userCanCreatePersonalSpace(ConfluenceUser user) {
        return permissionManager.hasCreatePermission(user, personalInformationManager.getOrCreatePersonalInformation(user), Space.class);
    }
}
