package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.pages.PageManager;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.SpaceBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.BlueprintIllegalArgumentException;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.ResourceErrorType;
import com.atlassian.confluence.plugins.createcontent.exceptions.ResourceException;
import com.atlassian.confluence.plugins.createcontent.rest.entities.CreatePersonalSpaceRestEntity;
import com.atlassian.confluence.plugins.createcontent.services.BlueprintResolver;
import com.atlassian.confluence.plugins.createcontent.services.RequestResolver;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintPageEntity;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintPageRequest;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintSpaceEntity;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintSpaceRequest;
import com.atlassian.confluence.plugins.createcontent.services.model.CreatePersonalSpaceRequest;
import com.atlassian.confluence.security.Permission;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.user.PersonalInformation;
import com.atlassian.confluence.user.PersonalInformationManager;
import com.atlassian.confluence.user.UserAccessor;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.sal.api.user.UserKey;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.ws.rs.core.Response;
import java.util.UUID;

import static com.atlassian.confluence.plugins.createcontent.impl.ModuleCompleteKeyUtils.getModuleCompleteKeyFromRelative;
import static com.atlassian.confluence.security.Permission.ADMINISTER;
import static com.atlassian.confluence.security.PermissionManager.TARGET_APPLICATION;
import static com.atlassian.confluence.spaces.Space.PERSONAL_SPACEKEY_IDENTIFIER;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

@Component
public class DefaultRequestResolver implements RequestResolver {
    private final BlueprintResolver blueprintResolver;
    private final SpaceManager spaceManager;
    private final PageManager pageManager;
    private final ContentBlueprintManager contentBlueprintManager;
    private final PermissionManager permissionManager;
    private final SpaceBlueprintManager spaceBlueprintManager;
    private final UserAccessor userAccessor;
    private final PersonalInformationManager personalInformationManager;

    @Autowired
    public DefaultRequestResolver(
            final BlueprintResolver blueprintResolver,
            final @ComponentImport SpaceManager spaceManager,
            final @ComponentImport PageManager pageManager,
            final ContentBlueprintManager contentBlueprintManager,
            final @ComponentImport PermissionManager permissionManager,
            final SpaceBlueprintManager spaceBlueprintManager,
            final @ComponentImport UserAccessor userAccessor,
            final @ComponentImport PersonalInformationManager personalInformationManager) {
        this.blueprintResolver = blueprintResolver;
        this.spaceManager = spaceManager;
        this.pageManager = pageManager;
        this.contentBlueprintManager = contentBlueprintManager;
        this.permissionManager = permissionManager;
        this.spaceBlueprintManager = spaceBlueprintManager;
        this.userAccessor = userAccessor;
        this.personalInformationManager = personalInformationManager;
    }

    @Override
    public CreateBlueprintPageRequest resolve(CreateBlueprintPageEntity entity, ConfluenceUser creator)
            throws BlueprintIllegalArgumentException {
        String spaceKey = entity.getSpaceKey();
        Long parentPageId = entity.getParentPageId();
        // The space will be retrieved by the spaceKey first and then by the spaceId (if the spaceKey is not present)
        // because the entity can be provided as a POST request body (e.g. as implemented in ContentBlueprintResource)
        // and most of the time the spaceKey is included in the request body instead of the spaceId
        Space space = StringUtils.isNotBlank(spaceKey) ? spaceManager.getSpace(spaceKey) : spaceManager.getSpace(entity.getSpaceId());
        Page parentPage = pageManager.getPage(parentPageId);

        ContentBlueprint blueprint;
        if (isNotBlank(entity.getContentBlueprintId())) {
            blueprint = contentBlueprintManager.getById(UUID.fromString(entity.getContentBlueprintId()));
        } else {
            // Coming in from somewhere external that doesn't have the UUID.
            ModuleCompleteKey moduleCompleteKey = new ModuleCompleteKey(entity.getModuleCompleteKey());
            blueprint = contentBlueprintManager.getPluginBackedContentBlueprint(moduleCompleteKey, spaceKey);
        }

        ContentTemplateRef contentTemplateRef = getContentTemplateRef(blueprint, entity);

        validateBlueprintPageEntity(blueprint, creator, space, parentPage);

        return new CreateBlueprintPageRequest(space, entity.getTitle(), entity.getViewPermissionsUsers(), parentPage,
                entity.getContext(), contentTemplateRef, creator, blueprint);
    }

    @Override
    public CreateBlueprintSpaceRequest resolve(@Nonnull CreateBlueprintSpaceEntity entity, @Nullable ConfluenceUser creator) {
        validateBlueprintSpaceEntity(entity, creator);

        UUID spaceBlueprintId = UUID.fromString(entity.getSpaceBlueprintId());
        SpaceBlueprint blueprint = spaceBlueprintManager.getById(spaceBlueprintId);

        return new CreateBlueprintSpaceRequest(blueprint, entity);
    }

    @Override
    public CreatePersonalSpaceRequest resolve(CreatePersonalSpaceRestEntity entity, ConfluenceUser creator) {
        ConfluenceUser spaceUser;
        String spaceUserKey = entity.getSpaceUserKey();
        if (StringUtils.isNotBlank(spaceUserKey)) {
            spaceUser = userAccessor.getUserByKey(new UserKey(spaceUserKey));
        } else {
            // HACK - until front-end can pass the remote user-key with the REST call we have to guess the user...
            spaceUser = creator;
        }

        validatePersonalSpaceEntity(spaceUser, creator);

        return new CreatePersonalSpaceRequest(spaceUser, entity.isSpacePermission());
    }

    private void validatePersonalSpaceEntity(ConfluenceUser spaceUser, ConfluenceUser creator) {
        if (creator == null)
            throw new ResourceException("Anonymous users cannot create personal spaces", Response.Status.BAD_REQUEST, ResourceErrorType.PERMISSION_ANONYMOUS_CREATE_PERSONAL_SPACE);

        if (spaceUser == null)
            throw new ResourceException("Cannot create personal space for unknown users", Response.Status.BAD_REQUEST, ResourceErrorType.PERMISSION_UNKNOWN_USER_CREATE_PERSONAL_SPACE);

        final String spaceUserName = spaceUser.getName();
        if (!creator.getKey().equals(spaceUser.getKey()) && !permissionManager.hasPermission(creator, ADMINISTER, TARGET_APPLICATION)) {
            // Creating a Personal space for someone else, but not an admin, eh?
            throw new ResourceException("No permission to create a personal space for user " +
                    spaceUserName, Response.Status.BAD_REQUEST, ResourceErrorType.PERMISSION_USER_CREATE_PERSONAL_SPACE, spaceUserName);
        }

        PersonalInformation pi = personalInformationManager.getOrCreatePersonalInformation(spaceUser);
        if (!permissionManager.hasCreatePermission(creator, pi, Space.class)) {
            throw new ResourceException("No permission to create personal spaces.", Response.Status.BAD_REQUEST, ResourceErrorType.PERMISSION_USER_CREATE_PERSONAL_SPACE, spaceUserName);
        }

        // FIXME: This should be using the getPersonalSpace() methods (maybe even the new ConfluenceUser ones)
        if (spaceManager.getSpace(PERSONAL_SPACEKEY_IDENTIFIER + spaceUserName) != null) {
            String cause = "A space already exists with key " + PERSONAL_SPACEKEY_IDENTIFIER + spaceUserName;
            throw new ResourceException(cause, Response.Status.BAD_REQUEST, ResourceErrorType.DUPLICATED_PERSONAL_SPACE, spaceUserName);
        }
    }

    private void validateBlueprintSpaceEntity(@Nullable CreateBlueprintSpaceEntity entity, @Nullable ConfluenceUser creator) {
        // FIXME - this stuff lifted from the SpacesSoapService and should be in a common layer.
        if (entity == null) {
            throw new ResourceException("Invalid space entity", Response.Status.BAD_REQUEST, ResourceErrorType.INVALID_ENTITY);
        }
        final String spaceKey = entity.getSpaceKey();
        if (!Space.isValidGlobalSpaceKey(spaceKey)) {
            throw new ResourceException("Invalid space key: " + spaceKey, Response.Status.BAD_REQUEST, ResourceErrorType.INVALID_SPACE_KEY, spaceKey);
        }
        if (!permissionManager.hasCreatePermission(creator, TARGET_APPLICATION, Space.class)) {
            throw new ResourceException("No permission to create spaces.", Response.Status.BAD_REQUEST,
                    creator == null ? ResourceErrorType.PERMISSION_ANONYMOUS_CREATE_SPACE : ResourceErrorType.PERMISSION_USER_CREATE_SPACE,
                    creator == null ? null : creator.getName());
        }
        if (spaceManager.getSpace(spaceKey) != null) {
            throw new ResourceException("A space already exists with key " + spaceKey, Response.Status.BAD_REQUEST, ResourceErrorType.DUPLICATED_SPACE, spaceKey);
        }

        if (entity.getSpaceBlueprintId() == null) {
            throw new ResourceException("Cannot create space from Blueprint with no id.", Response.Status.BAD_REQUEST, ResourceErrorType.INVALID_BLUEPRINT);
        }
    }

    private void validateBlueprintPageEntity(ContentBlueprint blueprint, ConfluenceUser creator, Space space,
                                             Page parentPage) throws BlueprintIllegalArgumentException {
        if (space == null) {
            throw new BlueprintIllegalArgumentException("Invalid space.", ResourceErrorType.INVALID_SPACE);
        }

        if (blueprint == null) {
            throw new BlueprintIllegalArgumentException("Invalid blueprint module key specified.", ResourceErrorType.INVALID_BLUEPRINT);
        }

        String spaceKey = space.getKey();

        //permission checks
        if (!permissionManager.hasCreatePermission(creator, space, Page.class)) {
            throw new BlueprintIllegalArgumentException("No permission to create pages in space " + spaceKey + ".", ResourceErrorType.PERMISSION_USER_CREATE_PAGE, spaceKey);
        }

        if (parentPage != null && !permissionManager.hasPermission(creator, Permission.VIEW, parentPage)) {
            throw new BlueprintIllegalArgumentException("No permission to create child pages of " + parentPage.getDisplayTitle() + ".", ResourceErrorType.PERMISSION_USER_VIEW_PAGE, parentPage.getId());
        }
    }

    private ContentTemplateRef getContentTemplateRef(ContentBlueprint blueprint, CreateBlueprintPageEntity entity) {
        if (blueprint == null)
            return null;

        String overrideTemplateId = entity.getContentTemplateId();
        if (isNotBlank(overrideTemplateId)) {
            UUID contentTemplateRefId = UUID.fromString(overrideTemplateId);
            return findContentTemplateRefInBlueprint(blueprint, contentTemplateRefId);
        }

        ContentTemplateRef ref;
        String overrideTemplateKey = entity.getContentTemplateKey();
        if (isNotBlank(overrideTemplateKey)) {
            ref = findContentTemplateRefInBlueprint(blueprint, overrideTemplateKey);
        } else {
            ref = blueprint.getFirstContentTemplateRef();
        }

        return blueprintResolver.resolveTemplateRef(ref);
    }

    private ContentTemplateRef findContentTemplateRefInBlueprint(ContentBlueprint contentBlueprint, UUID refId) {
        for (ContentTemplateRef ref : contentBlueprint.getContentTemplateRefs()) {
            if (ref.getId().equals(refId)) {
                return ref;
            }
        }
        throw new IllegalStateException("Content blueprint has no ContentTemplateRef with id: " + refId);
    }

    private ContentTemplateRef findContentTemplateRefInBlueprint(ContentBlueprint contentBlueprint, String key) {
        ModuleCompleteKey blueprintKey = new ModuleCompleteKey(contentBlueprint.getModuleCompleteKey());
        ModuleCompleteKey templateKey = getModuleCompleteKeyFromRelative(blueprintKey.getPluginKey(), key);
        String overrideTemplateKey = templateKey.getCompleteKey();

        for (ContentTemplateRef ref : contentBlueprint.getContentTemplateRefs()) {
            String blueprintTemplateKey = ref.getModuleCompleteKey();
            if (blueprintTemplateKey.equals(overrideTemplateKey)) {
                return ref;
            }
        }
        throw new IllegalStateException("Content blueprint has no ContentTemplateRef with key: " + key);
    }
}
