package com.atlassian.confluence.plugins.createcontent.extensions;

import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.actions.BPUserPreferences;
import com.atlassian.confluence.plugins.createcontent.impl.ContentBlueprint;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.user.UserAccessor;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.spring.scanner.annotation.export.ExportAsService;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.google.common.collect.Sets;
import com.opensymphony.module.propertyset.PropertySet;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;

import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toList;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * Default implementation of UserBlueprintConfigManager.
 */
@Component
@ExportAsService(UserBlueprintConfigManager.class)
public class DefaultUserBlueprintConfigManager implements UserBlueprintConfigManager {
    private static final Logger log = getLogger(DefaultUserBlueprintConfigManager.class);

    private static final String CREATED_BLUEPRINT_KEYS = "createdBlueprints";
    private static final String SKIP_HOW_TO_USE_BLUEPRINT_KEYS = "skip-how-to-use-blueprint-keys";

    private final UserAccessor userAccessor;
    private final ContentBlueprintManager contentBlueprintManager;

    @Autowired
    public DefaultUserBlueprintConfigManager(
            final @ComponentImport UserAccessor userAccessor,
            final ContentBlueprintManager contentBlueprintManager) {
        this.userAccessor = userAccessor;
        this.contentBlueprintManager = contentBlueprintManager;
    }

    @Override
    public Set<UUID> getSkipHowToUseKeys(ConfluenceUser user) {
        BPUserPreferences userPreferences = getUserPreferences(user);
        return getSkipKeys(userPreferences);
    }

    @Override
    public void setSkipHowToUse(ConfluenceUser user, UUID contentBlueprintId, boolean skip) {
        BPUserPreferences userPreferences = getUserPreferences(user);
        Set<UUID> skipKeys = getSkipKeys(userPreferences);
        if (skip)
            skipKeys.add(contentBlueprintId);
        else
            skipKeys.remove(contentBlueprintId);

        String keysStr = StringUtils.join(skipKeys, ",");
        userPreferences.setText(SKIP_HOW_TO_USE_BLUEPRINT_KEYS, keysStr);
    }

    @Override
    public boolean isFirstBlueprintOfTypeForUser(final UUID id, final ConfluenceUser user) {
        BPUserPreferences userPreferences = getUserPreferences(user);

        String blueprintKeys = userPreferences.getText(CREATED_BLUEPRINT_KEYS);
        return blueprintKeys == null || !blueprintKeys.contains(id.toString());
    }

    @Override
    public void setBlueprintCreatedByUser(final UUID id, final ConfluenceUser user) {
        BPUserPreferences userPreferences = getUserPreferences(user);
        String blueprintKeys = userPreferences.getText(CREATED_BLUEPRINT_KEYS);

        String idStr = id.toString();

        boolean modified = false;
        if (blueprintKeys == null) {
            blueprintKeys = idStr;
            modified = true;
        }
        if (!blueprintKeys.contains(idStr)) {
            blueprintKeys += ',' + idStr;
            modified = true;
        }
        if (modified) {
            userPreferences.setText(CREATED_BLUEPRINT_KEYS, blueprintKeys);
        }
    }

    public BPUserPreferences getUserPreferences(ConfluenceUser user) {
        PropertySet propertySet = userAccessor.getPropertySet(user);

        migratePrefs(propertySet, CREATED_BLUEPRINT_KEYS);
        migratePrefs(propertySet, SKIP_HOW_TO_USE_BLUEPRINT_KEYS);

        // TODO - cache these prefs per user. EG
        return new BPUserPreferences(propertySet);
    }

    private void migratePrefs(final PropertySet propertySet, final String key) {
        try {
            int type = propertySet.getType(key);
            if (type != PropertySet.STRING) {
                return;
            }

            final String string = propertySet.getString(key);

            /*
             * CONFSERVER-37166 we have updated the implementation of PropertySet#setText to apply a little optimisation
             * that turns the property to a STRING type instead of TEXT if the length of the value is less than 255 chars.
             *
             * because of that second time we go through here we need to check if the migration happened already in which
             * case we need to just stop the processing
             *
             *  the migration happens when we have processed blueprint module keys and turned them into their corresponding UUID
             *  one way to test for it is check if the string contains the following "plugin-key:module-key"
             */
            if (!string.contains(":")) {
                return;
            }

            String[] bpKeys = string.split(",");
            final List<String> uids = stream(bpKeys)
                    .map(bpKey -> {
                        if (!bpKey.contains(":")) {
                            return bpKey;
                        }
                        ContentBlueprint blueprint = contentBlueprintManager.getPluginBlueprint(new ModuleCompleteKey(bpKey));
                        return blueprint != null ? blueprint.getId().toString() : null;
                    })
                    .filter(Objects::nonNull)
                    .collect(toList());
            propertySet.remove(key);
            propertySet.setText(key, StringUtils.join(uids, ','));

        } catch (NullPointerException ex) {
            // Property not found. Some propertySet implementations are safe and return type 0, but some others just explode
        } catch (Exception ex) {
            log.warn("Unable to migrate Confluence Blueprint user preferences", ex);
        }
    }

    private Set<UUID> getSkipKeys(BPUserPreferences userPreferences) {
        String keysStr = userPreferences.getText(SKIP_HOW_TO_USE_BLUEPRINT_KEYS);
        if (StringUtils.isBlank(keysStr))
            return Sets.newHashSet();

        final String[] split = keysStr.split(",");
        Set<UUID> ids = Sets.newHashSet();
        for (String id : split) {
            ids.add(UUID.fromString(id));
        }

        return ids;
    }
}
