package com.atlassian.confluence.plugins.createcontent.extensions;

import com.atlassian.confluence.plugins.createcontent.actions.BlueprintManager;
import com.atlassian.confluence.plugins.dialog.wizard.api.DialogWizard;
import com.atlassian.confluence.plugins.dialog.wizard.api.DialogWizardModuleDescriptor;
import com.atlassian.confluence.util.i18n.DocumentationBeanFactory;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginParseException;
import com.atlassian.plugin.descriptors.AbstractModuleDescriptor;
import com.atlassian.plugin.module.ModuleFactory;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * Represents a blueprint that creates Confluence content in a pluggable fashion.
 *
 * @since 5.0
 */
public class BlueprintModuleDescriptor extends AbstractModuleDescriptor<Void> implements BlueprintDescriptor {
    private static final Logger log = LoggerFactory.getLogger(BlueprintModuleDescriptor.class);

    private final DocumentationBeanFactory documentationBeanFactory;

    private ModuleCompleteKey indexTemplateKey;
    private ModuleCompleteKey blueprintKey;
    private String indexKey;
    private boolean indexDisabled;
    private String createResult;
    private String howToUseTemplate;
    private String indexTitleI18nKey;
    private DialogWizard dialogWizard;
    private List<ModuleCompleteKey> contentTemplateKeys;
    private Element tempWizardElement;

    public BlueprintModuleDescriptor(
            final @ComponentImport ModuleFactory moduleFactory,
            final @ComponentImport DocumentationBeanFactory documentationBeanFactory) {
        super(moduleFactory);

        this.documentationBeanFactory = documentationBeanFactory;
    }

    @Override
    public void init(Plugin plugin, Element element) throws PluginParseException {
        super.init(plugin, element);

        if (isBlank(getKey()))
            throw new PluginParseException("key is a required attribute of <blueprint>.");
        if (isBlank(getI18nNameKey())) {
            // CONFDEV-18602 Should throw exception once our Blueprint plugins are updated
            log.warn("i18n-name-key is a required attribute of <blueprint> for module: " + getCompleteKey());
//            throw new PluginParseException("i18n-name-key is a required attribute of <blueprint>.");
        }

        blueprintKey = new ModuleCompleteKey(getCompleteKey());

        contentTemplateKeys = Lists.newArrayList();

        // Since version 1.3.8 of this plugin, the content-template-key attribute can be replaced with one or more
        // content-template-key *elements*
        String attrContentTemplateKey = element.attributeValue("content-template-key");
        if (StringUtils.isNotBlank(attrContentTemplateKey)) {
            contentTemplateKeys.add(new ModuleCompleteKey(plugin.getKey(), attrContentTemplateKey));
        } else {
            List<Element> templateElements = element.elements("content-template");
            for (Element keyElement : templateElements) {
                contentTemplateKeys.add(new ModuleCompleteKey(plugin.getKey(), keyElement.attributeValue("ref")));
            }
        }

        indexDisabled = Boolean.parseBoolean(element.attributeValue("disable-index-page"));
        indexTemplateKey = null;
        if (StringUtils.isNotBlank(element.attributeValue("index-template-key")))
            indexTemplateKey = new ModuleCompleteKey(plugin.getKey(), element.attributeValue("index-template-key"));
        else
            indexTemplateKey = new ModuleCompleteKey(BlueprintManager.PLUGIN_KEY, "default-index-template");

        indexKey = element.attributeValue("index-key");
        // An optional override - if none is supplied we fall back to the blueprint title that appears in the dialog.
        indexTitleI18nKey = element.attributeValue("i18n-index-title-key");

        createResult = element.attributeValue("create-result");
        howToUseTemplate = element.attributeValue("how-to-use-template");

        tempWizardElement = element.element("dialog-wizard");
    }

    /**
     * @deprecated since 1.3.11
     */
    @Deprecated
    @Override
    public ModuleCompleteKey getContentTemplateKey() {
        return getFirstContentTemplateKey();
    }

    @Override
    public ModuleCompleteKey getFirstContentTemplateKey() {
        return contentTemplateKeys.size() > 0 ? contentTemplateKeys.get(0) : null;
    }

    @Override
    public ModuleCompleteKey getContentTemplateKey(String contentTemplateModuleKey) {
        return new ModuleCompleteKey(getPluginKey(), contentTemplateModuleKey);
    }

    @Override
    public List<ModuleCompleteKey> getContentTemplates() {
        return contentTemplateKeys;
    }

    @Override
    public ModuleCompleteKey getIndexTemplate() {
        return indexTemplateKey;
    }

    @Override
    public ModuleCompleteKey getBlueprintKey() {
        return blueprintKey;
    }

    @Override
    public String getIndexKey() {
        return indexKey;
    }

    @Override
    public String getCreateResult() {
        return createResult;
    }

    @Override
    public String getIndexTitleI18nKey() {
        return indexTitleI18nKey;
    }

    @Override
    public Void getModule() {
        return null;
    }

    @Override
    public String getHowToUseTemplate() {
        return howToUseTemplate;
    }

    @Override
    public DialogWizard getDialogWizard() {
        // important that we calculate the dialog wizard and pass in the documentation bean factory after the blueprint plugin has initialised
        if (dialogWizard == null && tempWizardElement != null) {
            DialogWizardModuleDescriptor moduleDescriptor = new DialogWizardModuleDescriptor(moduleFactory, documentationBeanFactory);
            moduleDescriptor.init(plugin, tempWizardElement);
            dialogWizard = moduleDescriptor.getModule();
            tempWizardElement = null;
        }
        return dialogWizard;
    }

    @Override
    public boolean isIndexDisabled() {
        return indexDisabled;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;

        if (!(obj instanceof BlueprintModuleDescriptor))
            return false;

        BlueprintModuleDescriptor that = (BlueprintModuleDescriptor) obj;

        return Objects.equal(blueprintKey, that.blueprintKey);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(blueprintKey);
    }
}
