package com.atlassian.confluence.plugins.createcontent.actions;

import com.atlassian.confluence.event.events.template.TemplateUpdateEvent;
import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.pages.templates.PluginTemplateReference;
import com.atlassian.confluence.plugins.createcontent.ContentTemplateRefManager;
import com.atlassian.confluence.plugins.createcontent.api.events.BlueprintTemplateUpdateEvent;
import com.atlassian.confluence.plugins.createcontent.impl.ContentBlueprint;
import com.atlassian.confluence.plugins.createcontent.impl.ContentTemplateRef;
import com.atlassian.confluence.plugins.createcontent.services.BlueprintResolver;
import com.atlassian.confluence.plugins.createcontent.template.PluginPageTemplateHelper;
import com.atlassian.confluence.plugins.templates.actions.AbstractEditPageTemplateAction;
import com.atlassian.confluence.plugins.templates.actions.AbstractPageTemplateAction;
import com.atlassian.confluence.security.Permission;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.plugin.ModuleCompleteKey;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nullable;
import java.util.UUID;

import static com.atlassian.confluence.pages.templates.PluginTemplateReference.spaceTemplateReference;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

public class EditTemplateAction extends AbstractEditPageTemplateAction {
    private PluginPageTemplateHelper pluginPageTemplateHelper;
    private ContentTemplateRef templateRef;
    private ContentTemplateRefManager contentTemplateRefManager;
    private BlueprintResolver blueprintResolver;

    private String pluginKey;
    private String moduleKey;
    @Nullable
    private String referencingPluginKey;
    @Nullable
    private String referencingModuleKey;
    private int pageTemplateVersion;

    /**
     * This is overridden because the method in {@link AbstractPageTemplateAction} expects the template to have a space
     * and if it does not, it checks administration permissions on the application instead of the space.
     */
    @Override
    public boolean isPermitted() {
        Space space = getSpace();
        Object target = (space != null) ? space : PermissionManager.TARGET_APPLICATION;
        return permissionManager.hasPermission(getRemoteUser(), Permission.ADMINISTER, target);
    }

    @Override
    public String doDefault() throws Exception {
        pageTemplate = getPageTemplate();

        if (pageTemplate == null)
            return "notfound";

        PageTemplate usedTemplate = pageTemplate;

        // We might use an override template ref for the used content, title and description.
        if (templateRef != null) {
            ContentTemplateRef usedTemplateRef = blueprintResolver.resolveTemplateRef(templateRef);
            usedTemplate = pluginPageTemplateHelper.getPageTemplate(usedTemplateRef);
        }

        title = usedTemplate.getName();
        description = usedTemplate.getDescription();
        wysiwygContent = formatConverter.convertToEditorFormat(usedTemplate.getContent(), getRenderContext());
        pageTemplateVersion = pageTemplate.getVersion();    // must come from original PageTemplate

        return super.doDefault();
    }

    /**
     * Called when saving an Edit to a PageTemplate that is:
     * a) attached to a ContentBlueprint, and
     * b) has never been edited before
     */
    @Override
    public String execute() throws Exception {
        if (isNotBlank(back))
            return INPUT;

        if (isNotBlank(preview))
            return PREVIEW;

        pageTemplate = getPageTemplate();

        if (pageTemplate == null)
            return "notfound";

        if (pageTemplate.getVersion() != pageTemplateVersion) {
            addActionError(getText("create.content.plugin.template.updated.since.edit"));
            return ERROR;
        }

        Space space = getSpace();
        if (space != null) {
            if (pageTemplate.getSpace() == null && !pageTemplate.isNew()) {
                // This template is an edited global one - don't change it. And don't clone it - evil!
                // This is just to make following line savePageTemplate(pageTemplate, null) to execute in order to
                // create a copy of page template for space
                //This is the page we should use as the original template.
                pageTemplate = new PageTemplate(pageTemplate);
            }
            space.addPageTemplate(pageTemplate);
        }

        if (pageTemplate.isNew())
            pageTemplateManager.savePageTemplate(pageTemplate, null); // copy the first version to database first

        PageTemplate originalVersion;
        try {
            originalVersion = (PageTemplate) pageTemplate.clone();
        } catch (CloneNotSupportedException e) {
            throw new AssertionError("Should not happen");
        }

        pageTemplate.setName(title);
        pageTemplate.setDescription(description);
        pageTemplate.setContent(formatConverter.convertToStorageFormat(wysiwygContent, getRenderContext()));

        // system page templates will not have templateRefs
        if (templateRef != null) {
            // User-generated Templates will not have module keys.
            ModuleCompleteKey templateModuleKey = null;
            String templateModuleKeyStr = templateRef.getModuleCompleteKey();
            if (isNotBlank(templateModuleKeyStr)) {
                templateModuleKey = new ModuleCompleteKey(templateModuleKeyStr);
            }
            pageTemplate.setModuleCompleteKey(templateModuleKey);

            final ContentBlueprint blueprint = templateRef.getParent();
            ModuleCompleteKey blueprintModuleKey = new ModuleCompleteKey(blueprint.getModuleCompleteKey());
            pageTemplate.setReferencingModuleCompleteKey(blueprintModuleKey);

            pageTemplateManager.savePageTemplate(pageTemplate, originalVersion);

            String pluginKey = templateModuleKey.getPluginKey();
            String moduleKey = templateModuleKey.getModuleKey();
            BlueprintTemplateUpdateEvent templateUpdatedEvent = new BlueprintTemplateUpdateEvent(this, pluginKey, moduleKey, space);
            eventManager.publishEvent(templateUpdatedEvent);
        }

        TemplateUpdateEvent templateEvent = new TemplateUpdateEvent(this, null, pageTemplate);
        eventManager.publishEvent(templateEvent);

        return super.execute();
    }

    @Override
    public PageTemplate getPageTemplate() {
        if (entityId > 0) {
            // An existing template - can be found without the Blueprint system.
            return pageTemplateManager.getPageTemplate(entityId);
        } else if (StringUtils.isNotBlank(pluginKey) && StringUtils.isNotBlank(moduleKey)) {
            ModuleCompleteKey referencingModuleCompleteKey = null;
            if (StringUtils.isNotBlank(referencingPluginKey) && StringUtils.isNotBlank(referencingModuleKey)) {
                referencingModuleCompleteKey = new ModuleCompleteKey(referencingPluginKey, referencingModuleKey);
            }

            PluginTemplateReference pluginTemplateReference = spaceTemplateReference(
                    new ModuleCompleteKey(pluginKey, moduleKey),
                    referencingModuleCompleteKey,
                    getSpace());
            return pluginPageTemplateHelper.getPageTemplate(pluginTemplateReference);
        }

        if (templateRef == null)
            throw new IllegalStateException("Action should be passed enough data to locate a PageTemplate.");

        return pluginPageTemplateHelper.getPageTemplate(templateRef);
    }

    public String getPluginKey() {
        return pluginKey;
    }

    public void setPluginKey(String pluginKey) {
        this.pluginKey = pluginKey;
    }

    public String getModuleKey() {
        return moduleKey;
    }

    public void setModuleKey(String moduleKey) {
        this.moduleKey = moduleKey;
    }

    @Nullable
    public String getReferencingPluginKey() {
        return referencingPluginKey;
    }

    public void setReferencingPluginKey(@Nullable String referencingPluginKey) {
        this.referencingPluginKey = referencingPluginKey;
    }

    @Nullable
    public String getReferencingModuleKey() {
        return referencingModuleKey;
    }

    public void setReferencingModuleKey(@Nullable String referencingModuleKey) {
        this.referencingModuleKey = referencingModuleKey;
    }

    public int getPageTemplateVersion() {
        return pageTemplateVersion;
    }

    public void setPageTemplateVersion(int pageTemplateVersion) {
        this.pageTemplateVersion = pageTemplateVersion;
    }

    public void setSpaceKey(String key) {
        // Because key is not a very helpful name for a parameter.
        setKey(key);
    }

    public String getContentTemplateRefId() {
        return templateRef != null ? templateRef.getId().toString() : null;
    }

    // System templates don't have template ref ids - could be empty!
    public void setContentTemplateRefId(String templateRefId) {
        if (StringUtils.isNotBlank(templateRefId))
            templateRef = contentTemplateRefManager.getById(UUID.fromString(templateRefId));
    }

    public void setPluginPageTemplateHelper(PluginPageTemplateHelper pluginPageTemplateHelper) {
        this.pluginPageTemplateHelper = pluginPageTemplateHelper;
    }

    public void setContentTemplateRefManager(ContentTemplateRefManager contentTemplateRefManager) {
        this.contentTemplateRefManager = contentTemplateRefManager;
    }

    public void setBlueprintResolver(BlueprintResolver blueprintResolver) {
        this.blueprintResolver = blueprintResolver;
    }
}
