package com.atlassian.confluence.plugins.createcontent.actions;

import com.atlassian.confluence.content.render.xhtml.DefaultConversionContext;
import com.atlassian.confluence.content.render.xhtml.XhtmlException;
import com.atlassian.confluence.core.ContextPathHolder;
import com.atlassian.confluence.labels.Label;
import com.atlassian.confluence.labels.Labelable;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.pages.PageManager;
import com.atlassian.confluence.pages.actions.CreatePageAction;
import com.atlassian.confluence.plugin.webresource.ConfluenceWebResourceManager;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.api.events.BlueprintPageCreateEvent;
import com.atlassian.confluence.plugins.createcontent.impl.ContentBlueprint;
import com.atlassian.confluence.plugins.createcontent.impl.ContentTemplateRef;
import com.atlassian.confluence.renderer.PageContext;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.xhtml.api.EditorFormatService;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import org.apache.commons.lang3.StringUtils;

import javax.xml.stream.XMLStreamException;
import java.lang.reflect.Type;
import java.util.Map;
import java.util.UUID;

import static com.google.common.collect.Maps.newHashMap;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * Provides base services needed by the concrete create-related actions for blueprints.
 */
public abstract class AbstractCreateBlueprintPageAction extends CreatePageAction {
    @ComponentImport
    private EditorFormatService editorFormatService;
    @ComponentImport
    private ConfluenceWebResourceManager confluenceWebResourceManager;
    @ComponentImport
    private ContextPathHolder contextPathHolder;
    private BlueprintManager blueprintManager;
    private BlueprintContentGenerator contentGenerator;

    protected ContentBlueprint contentBlueprint;
    protected Map<String, Object> context = newHashMap();

    private boolean goToIndexPage;

    /**
     * The ref of the ContentTemplate that will be used to populate this Blueprint instance. If not set by the POST,
     * the template key of the {@link ContentBlueprint} will be used.
     */
    private ContentTemplateRef contentTemplateRef;
    protected ContentBlueprintManager contentBlueprintManager;

    protected void validatePageTitleAgainstIndexPageTitle() {
        final String indexPageTitle = blueprintManager.getIndexPageTitle(contentBlueprint);
        if (indexPageTitle.equalsIgnoreCase(getTitle())) {
            addActionError("create.content.plugin.index.page.title.clash", getTitle());
        }
    }

    private void initialiseWysiwygContent(String contentBody) throws XhtmlException {
        final DefaultConversionContext conversionContext = new DefaultConversionContext(new PageContext(getSpace().getKey()));
        try {
            final String wysiwygContent = editorFormatService.convertStorageToEdit(contentBody, conversionContext);
            setWysiwygContent(wysiwygContent);
        } catch (XMLStreamException e) {
            throw new XhtmlException(e);
        }
    }

    protected Page getOrCreateIndexPage() {
        return blueprintManager.createAndPinIndexPage(contentBlueprint, getSpace());
    }

    private Label getBlueprintIndexLabel() {
        return new Label(contentBlueprint.getIndexKey());
    }

    /**
     * Loads the Blueprint content template and generates the Blueprint Page from the template and the context.
     */
    protected Page populateBlueprintPage() throws XhtmlException {
        ContentTemplateRef contentTemplateRef = getContentTemplateRef();
        final Page blueprintPage = contentGenerator.generateBlueprintPageObject(contentTemplateRef, getSpace(), context);

        // Override the default title if one was passed in the request
        String blueprintPageDefaultTitle = blueprintPage.getTitle();
        String blueprintPageCustomTitle = getTitle();
        if (isNotBlank(blueprintPageCustomTitle)) {
            blueprintPage.setTitle(blueprintPageCustomTitle);
        } else {
            setTitle(blueprintPageDefaultTitle);
        }
        final Label blueprintIndexLabel = getBlueprintIndexLabel();
        getLabelManager().addLabel((Labelable) blueprintPage, blueprintIndexLabel);

        // Init the action fields from the blueprint, to be passed to the editor if necessary
        initialiseWysiwygContent(blueprintPage.getBodyAsString());
        setLabelsString(blueprintIndexLabel.getName());

        return blueprintPage;
    }

    /**
     * We need to pass information from this action call to the one where the form is submitted, so
     * that we know which 'simple' blueprint module is being used and which index page should be created, etc.
     * Ideally we'd have some sort of flash-storage to pass this info without a metadata-JS-hidden-field
     * browser round trip.
     */
    public void storeBlueprintKeyInEditorContext() {
        // blueprint may be null if editing an existing draft that has the Blueprint Id stored in the contentPropertyManager.
        if (contentBlueprint != null) {
            UUID aoId = contentBlueprint.getId();
            confluenceWebResourceManager.putMetadata("content-blueprint-id", String.valueOf(aoId));
        }
        confluenceWebResourceManager.requireResourcesForContext("editor-blueprint");
    }

    public void storeBlueprintPageIndicatorInEditorContext() {
        confluenceWebResourceManager.putMetadata("is-blueprint-page", "true");
    }

    protected void sendBlueprintPageCreateEvent(final Page page) {
        ConfluenceUser user = AuthenticatedUserThreadLocal.get();
        getEventManager().publishEvent(new BlueprintPageCreateEvent(this, page, contentBlueprint, user, context));
    }

    public void setContextJson(final String contextJson) {
        if (isNotBlank(contextJson)) {
            final Type type = new TypeToken<Map<String, String>>() {
            }.getType();
            final Map<String, Object> newContext = new Gson().fromJson(contextJson, type);
            setContext(newContext);
        }
    }

    protected void setContext(Map<String, Object> context) {
        this.context = context;
    }

    public void setContentBlueprintId(final String id) {
        this.contentBlueprint = contentBlueprintManager.getById(UUID.fromString(id));
    }

    /**
     * Permissions checking is currently done against the space associated with "newSpaceKey" (as opposed to "spaceKey").
     * To avoid passing the newSpaceKey as a request parameter, we override setSpaceKey to also set the newSpaceKey.
     * <p>
     * Failing to do this will lead to strange permission issues.
     */
    @Override
    public void setSpaceKey(final String spaceKey) {
        super.setSpaceKey(spaceKey);
        setNewSpaceKey(spaceKey);
    }

    private ContentTemplateRef getContentTemplateRef() {
        if (contentTemplateRef == null) {
            // Not specified in the UI - use the default for the Blueprint
            contentTemplateRef = contentBlueprint.getFirstContentTemplateRef();
            assert (contentTemplateRef != null);
        }
        return contentTemplateRef;
    }

    /**
     * Allows create POSTs to override the plugin descriptor's defined template key.
     */
    public void setContentTemplateRefId(final String contentTemplateRefId) {
        if (contentTemplateRefId != null) {
            // NOTE - we could allow for this being a completely-qualified plugin module key in future, but
            // for now a *relative* key should do us fine.
            // Overridden from the front end, use the template matching the id.
            contentTemplateRef = findContentTemplateRefInBlueprint(UUID.fromString(contentTemplateRefId));
        }
    }

    // Dialog wizards that choose templates should do it with ids that the JS logic gets from the loaded
    // ContentBlueprint.

    /**
     * @deprecated since 2.0.0
     */
    @Deprecated
    public void setContentTemplateKey(String contentTemplateKey) {
        if (StringUtils.isBlank(contentTemplateKey))
            return;

        // NOTE - we could allow for this being a completely-qualified plugin module key in future, but
        // for now a *relative* key should do us fine.
        // Overridden from the front end, use the template matching the id.
        final ModuleCompleteKey contentBlueprintKey = new ModuleCompleteKey(contentBlueprint.getModuleCompleteKey());
        final ModuleCompleteKey contentTemplateModuleKey = new ModuleCompleteKey(contentBlueprintKey.getPluginKey(), contentTemplateKey);

        contentTemplateRef = findContentTemplateRefInBlueprint(contentTemplateModuleKey.getCompleteKey());
    }

    private ContentTemplateRef findContentTemplateRefInBlueprint(UUID refId) {
        for (ContentTemplateRef ref : contentBlueprint.getContentTemplateRefs()) {
            if (ref.getId().equals(refId)) {
                return ref;
            }
        }
        throw new IllegalStateException("Content blueprint has no ContentTemplateRef with id: " + refId);
    }

    private ContentTemplateRef findContentTemplateRefInBlueprint(String moduleCompleteKey) {
        for (ContentTemplateRef ref : contentBlueprint.getContentTemplateRefs()) {
            if (moduleCompleteKey.equals(ref.getModuleCompleteKey())) {
                return ref;
            }
        }
        throw new IllegalStateException("Content blueprint has no ContentTemplateRef with moduleCompleteKey: " + moduleCompleteKey);
    }

    public String getFormaction() {
        return contextPathHolder.getContextPath() + "/plugins/createcontent/docreatepage.action";
    }

    public void setContextPathHolder(final ContextPathHolder contextPathHolder) {
        this.contextPathHolder = contextPathHolder;
    }

    public void setBlueprintManager(final BlueprintManager blueprintManager) {
        this.blueprintManager = blueprintManager;
    }

    public void setContentGenerator(BlueprintContentGenerator contentGenerator) {
        this.contentGenerator = contentGenerator;
    }

    public void setEditorFormatService(final EditorFormatService editorFormatService) {
        this.editorFormatService = editorFormatService;
    }

    public void setConfluenceWebResourceManager(final ConfluenceWebResourceManager confluenceWebResourceManager) {
        this.confluenceWebResourceManager = confluenceWebResourceManager;
    }

    public void setContentBlueprintManager(ContentBlueprintManager contentBlueprintManager) {
        this.contentBlueprintManager = contentBlueprintManager;
    }

    protected PageManager getPageManager() {
        return pageManager;
    }

    public boolean getGoToIndexPage() {
        return goToIndexPage;
    }

    public void setGoToIndexPage(boolean goToIndexPage) {
        this.goToIndexPage = goToIndexPage;
    }
}
