package com.atlassian.confluence.plugins.createcontent.rest;

import com.atlassian.activeobjects.external.ActiveObjects;
import com.atlassian.confluence.api.service.accessmode.AccessModeService;
import com.atlassian.confluence.api.service.accessmode.ReadOnlyAccessAllowed;
import com.atlassian.confluence.plugins.createcontent.ContentTemplateRefManager;
import com.atlassian.confluence.plugins.createcontent.activeobjects.ContentTemplateRefAo;
import com.atlassian.confluence.plugins.createcontent.impl.ContentTemplateRef;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;

import javax.annotation.Nonnull;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import java.util.UUID;

import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static javax.ws.rs.core.MediaType.APPLICATION_XML;

@Path("/templatesRefs")
public class ContentTemplateRefResource extends AbstractRestResource {
    public static final String PARAM_ID = "id";

    private final ContentTemplateRefManager contentTemplateRefManager;
    private final ActiveObjects activeObjects;

    public ContentTemplateRefResource(
            final @ComponentImport PermissionManager permissionManager,
            final @ComponentImport SpaceManager spaceManager,
            final ContentTemplateRefManager contentTemplateRefManager,
            final @ComponentImport ActiveObjects activeObjects,
            final @ComponentImport AccessModeService accessModeService) {
        super(permissionManager, spaceManager, accessModeService);

        this.contentTemplateRefManager = contentTemplateRefManager;
        this.activeObjects = activeObjects;
    }

    @DELETE
    @ReadOnlyAccessAllowed
    @Path("deleteAll")
    public Integer deleteAll() {
        checkAdminPermission();

        return contentTemplateRefManager.deleteAll();
    }

    @GET
    @Path("{" + PARAM_ID + "}")
    public ContentTemplateRef get(@PathParam(PARAM_ID) UUID uuid) {
        checkNullParameter(uuid, PARAM_ID);

        return contentTemplateRefManager.getById(uuid);
    }

    @DELETE
    @Path("{" + PARAM_ID + "}")
    public boolean delete(@PathParam(PARAM_ID) UUID uuid) {
        checkNullParameter(uuid, PARAM_ID);

        return contentTemplateRefManager.delete(uuid);
    }

    @POST
    @Consumes({APPLICATION_JSON, APPLICATION_XML})
    public UUID create(final ContentTemplateRef contentTemplateRef) {
        checkNullEntity(contentTemplateRef);

        ContentTemplateRefAo ao = activeObjects.executeInTransaction(() -> saveContentTemplateRef(contentTemplateRef));
        if (ao != null) {
            return UUID.fromString(ao.getUuid());
        }
        return null;
    }

    @PUT
    @Consumes({APPLICATION_JSON, APPLICATION_XML})
    public void update(final ContentTemplateRef contentTemplateRef) {
        contentTemplateRefManager.update(contentTemplateRef);
        for (ContentTemplateRef child : contentTemplateRef.getChildren()) {
            update(child);
        }
    }

    @Nonnull
    private ContentTemplateRefAo saveContentTemplateRef(@Nonnull ContentTemplateRef contentTemplateRef) {
        final ContentTemplateRefAo parentAo = contentTemplateRefManager.createAo(contentTemplateRef);

        for (ContentTemplateRef child : contentTemplateRef.getChildren()) {
            ContentTemplateRefAo childAo = saveContentTemplateRef(child);
            childAo.setParent(parentAo);
            childAo.save();
        }

        return parentAo;
    }
}
