package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.content.ContentBody;
import com.atlassian.confluence.api.model.content.ContentRepresentation;
import com.atlassian.confluence.api.model.content.Space;
import com.atlassian.confluence.api.model.content.template.ContentTemplate;
import com.atlassian.confluence.api.model.content.template.ContentTemplateId;
import com.atlassian.confluence.api.model.content.template.ContentTemplateType;
import com.atlassian.confluence.api.model.pagination.LimitedRequest;
import com.atlassian.confluence.api.model.pagination.LimitedRequestImpl;
import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.model.pagination.PageResponse;
import com.atlassian.confluence.api.model.pagination.PageResponseImpl;
import com.atlassian.confluence.api.model.pagination.SimplePageRequest;
import com.atlassian.confluence.api.model.validation.SimpleValidationResult;
import com.atlassian.confluence.api.model.validation.ValidationResult;
import com.atlassian.confluence.api.service.content.ContentBodyConversionService;
import com.atlassian.confluence.api.service.exceptions.BadRequestException;
import com.atlassian.confluence.api.service.exceptions.NotFoundException;
import com.atlassian.confluence.api.service.exceptions.ServiceException;
import com.atlassian.confluence.api.service.exceptions.unchecked.NotImplementedServiceException;
import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.pages.templates.PageTemplateManager;
import com.atlassian.confluence.pages.templates.PluginTemplateReference;
import com.atlassian.confluence.plugins.createcontent.BlueprintStateController;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.ContentTemplateRefManager;
import com.atlassian.confluence.plugins.createcontent.api.events.BlueprintTemplateUpdateEvent;
import com.atlassian.confluence.plugins.createcontent.api.services.BlueprintContentTemplateService;
import com.atlassian.confluence.plugins.createcontent.factory.ContentTemplateFactory;
import com.atlassian.confluence.plugins.createcontent.factory.FinderFactory;
import com.atlassian.confluence.plugins.createcontent.model.BlueprintState;
import com.atlassian.confluence.plugins.createcontent.template.PageTemplateGrouper;
import com.atlassian.confluence.plugins.createcontent.template.PluginPageTemplateHelper;
import com.atlassian.confluence.security.Permission;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.fugue.Option;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

import static com.atlassian.confluence.api.model.content.template.ContentTemplateId.ContentTemplateIdWithKeys;
import static com.atlassian.confluence.api.model.content.template.ContentTemplateId.ContentTemplateIdWithUUID;
import static com.atlassian.confluence.api.model.messages.SimpleMessage.withTranslation;
import static com.atlassian.confluence.pages.templates.PluginTemplateReference.spaceTemplateReference;
import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.CREATE_DIALOG_CONTENT_SECTION;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;

/**
 * Implement this service to provide CRUD for blueprint template
 */
@Component
public class DefaultBlueprintContentTemplateService implements BlueprintContentTemplateService {
    public static final int MAX_LIMIT = 200;
    private final PageTemplateGrouper pageTemplateGrouper;
    private final BlueprintStateController blueprintStateController;
    private final SpaceManager spaceManager;
    private final PluginPageTemplateHelper pluginPageTemplateHelper;
    private final ContentTemplateRefManager contentTemplateRefManager;
    private final PageTemplateManager pageTemplateManager;
    private final ContentBodyConversionService contentBodyConversionService;
    private final EventPublisher eventPublisher;
    private final ContentBlueprintManager contentBlueprintManager;
    private final PermissionManager permissionManager;
    private final FinderFactory templateFinderFactory;

    @Autowired
    public DefaultBlueprintContentTemplateService(
            final PageTemplateGrouper pageTemplateGrouper,
            final BlueprintStateController blueprintStateController,
            final @ComponentImport SpaceManager spaceManager,
            final PluginPageTemplateHelper pluginPageTemplateHelper,
            final ContentTemplateRefManager contentTemplateRefManager,
            final @ComponentImport PageTemplateManager pageTemplateManager,
            final @ComponentImport ContentBodyConversionService contentBodyConversionService,
            final @ComponentImport EventPublisher eventPublisher,
            final ContentBlueprintManager contentBlueprintManager,
            final @ComponentImport PermissionManager permissionManager,
            final FinderFactory templateFinderFactory) {
        this.pageTemplateGrouper = pageTemplateGrouper;
        this.blueprintStateController = blueprintStateController;
        this.spaceManager = spaceManager;
        this.pluginPageTemplateHelper = pluginPageTemplateHelper;
        this.contentTemplateRefManager = contentTemplateRefManager;
        this.pageTemplateManager = pageTemplateManager;
        this.contentBodyConversionService = contentBodyConversionService;
        this.eventPublisher = eventPublisher;
        this.contentBlueprintManager = contentBlueprintManager;
        this.permissionManager = permissionManager;
        this.templateFinderFactory = templateFinderFactory;
    }

    @Override
    public PageResponse<ContentTemplate> getTemplates(ContentTemplateType contentTemplateType, Option<Space> space, PageRequest pageRequest, Expansion... expansions) {
        validator().validateGet(space).throwIfInvalid("Cannot get templates");
        com.atlassian.confluence.spaces.Space mySpace = null;
        if (space.isDefined()) {
            mySpace = spaceManager.getSpace(space.get().getKey());
        }

        final ConfluenceUser currentUser = AuthenticatedUserThreadLocal.get();
        final Map<UUID, BlueprintState> blueprintStateMap =
                blueprintStateController.getAllContentBlueprintState(CREATE_DIALOG_CONTENT_SECTION, currentUser, mySpace);

        // Gets all blueprints for that context
        final Collection<ContentBlueprint> contentBlueprints = pageTemplateGrouper.getSpaceContentBlueprints(mySpace);
        // Gets all blueprints to be displayed in the Admin page (enabled or disabled)
        final Collection<ContentBlueprint> displayedBlueprints = getDisplayableBlueprints(contentBlueprints,
                blueprintStateMap, space.isDefined());
        List<ContentTemplate> contentTemplates = convertToContentTemplates(space, displayedBlueprints, expansions);

        if (pageRequest == null) {
            pageRequest = new SimplePageRequest(0, MAX_LIMIT);
        }
        LimitedRequest limitedRequest = LimitedRequestImpl.create(pageRequest, MAX_LIMIT);
        return PageResponseImpl.filteredPageResponse(limitedRequest, contentTemplates, contentTemplate -> true);
    }

    public PageResponse<ContentTemplate> getTemplates(Option<Space> space, PageRequest pageRequest, Expansion... expansions) {
        return getTemplates(ContentTemplateType.BLUEPRINT, space, pageRequest, expansions);
    }

    private List<ContentTemplate> convertToContentTemplates(Option<Space> space, Collection<ContentBlueprint> displayedBlueprints, Expansion... expansions) {
        //flat blueprint into template list, first order by ContentBlueprint, then order by ContentTemplateRef.
        List<ContentTemplate> contentTemplates = new ArrayList<>();
        for (ContentBlueprint blueprint : displayedBlueprints) {
            blueprint.getContentTemplateRefs().forEach(templateRef -> contentTemplates.add(getContentTemplate(space, templateRef, expansions)));

            if (blueprint.getIndexPageTemplateRef() != null) {
                contentTemplates.add(getContentTemplate(space, blueprint.getIndexPageTemplateRef(), expansions));
            }
        }
        return contentTemplates;
    }

    private ContentTemplate getContentTemplate(Option<Space> space, ContentTemplateRef templateRef, Expansion... expansions) {
        PageTemplate pageTemplate = pluginPageTemplateHelper.getPageTemplate(templateRef);
        if (space.isDefined()) {
            //find and use the possibly edited pageTemplate on global level
            pageTemplate = resolvePageTemplate(templateRef);
        }

        return ContentTemplateFactory.buildFrom(space, templateRef, pageTemplate, expansions);
    }

    @Override
    public ContentTemplate getTemplate(ContentTemplateId contentTemplateId, Expansion... expansions) {

        validator().validateGet(contentTemplateId).throwIfInvalid("Cannot get template");

        ContentTemplateRef templateRef;
        if (contentTemplateId instanceof ContentTemplateIdWithUUID) {
            templateRef = getContentTemplateRefWithUuid(contentTemplateId);
        } else if (contentTemplateId instanceof ContentTemplateIdWithKeys) {
            templateRef = getContentTemplateRefWithKeys(contentTemplateId);
        } else {
            throw new NotImplementedServiceException("contentTemplateId is not supported: " + contentTemplateId);
        }

        ContentBlueprint blueprint = templateRef.getParent();
        PageTemplate pageTemplate = resolvePageTemplate(templateRef);

        Option<Space> space = Strings.isNullOrEmpty(blueprint.getSpaceKey()) ?
                Option.none() : Option.some(Space.builder().key(blueprint.getSpaceKey()).build());
        return ContentTemplateFactory.buildFrom(space, templateRef, pageTemplate, expansions);
    }

    private PageTemplate resolvePageTemplate(ContentTemplateRef templateRef) {
        ContentBlueprint blueprint = templateRef.getParent();
        PluginTemplateReference pluginTemplateReference = spaceTemplateReference(
                new ModuleCompleteKey(templateRef.getModuleCompleteKey()),
                new ModuleCompleteKey(blueprint.getModuleCompleteKey()),
                Strings.isNullOrEmpty(blueprint.getSpaceKey()) ? null : spaceManager.getSpace(blueprint.getSpaceKey()));
        return pluginPageTemplateHelper.getPageTemplate(pluginTemplateReference);
    }

    @Override
    public ContentTemplate create(ContentTemplate contentTemplate, Expansion... expand) {
        validator().validateCreate(contentTemplate).throwIfInvalid("Cannot create contentTemplate.");

        PageTemplate editingPageTemplate = getPageTemplate(contentTemplate.getTemplateId());
        if (editingPageTemplate == null) {
            throw new IllegalStateException("Cannot find the contentTemplateId: " + contentTemplate.getTemplateId());
        }

        editingPageTemplate = createFirstVersionForPageTemplate(contentTemplate, editingPageTemplate);

        PageTemplate firstVersionPageTemplate;
        try {
            firstVersionPageTemplate = (PageTemplate) editingPageTemplate.clone();
        } catch (CloneNotSupportedException ex) {
            throw new ServiceException("Should not happened error happened. ", ex);
        }

        editingPageTemplate.setName(contentTemplate.getName());
        editingPageTemplate.setDescription(contentTemplate.getDescription());
        setTemplateBody(contentTemplate, editingPageTemplate);

        ContentTemplateRef templateRef;
        if (contentTemplate.getTemplateId() instanceof ContentTemplateIdWithUUID) {
            templateRef = getContentTemplateRefWithUuid(contentTemplate.getTemplateId());
        } else {
            templateRef = getContentTemplateRefWithKeys(contentTemplate.getTemplateId());
        }

        ModuleCompleteKey templateModuleKey = new ModuleCompleteKey(templateRef.getModuleCompleteKey());
        editingPageTemplate.setModuleCompleteKey(templateModuleKey);
        editingPageTemplate.setReferencingModuleCompleteKey(new ModuleCompleteKey(templateRef.getParent().getModuleCompleteKey()));

        pageTemplateManager.savePageTemplate(editingPageTemplate, firstVersionPageTemplate);
        publishUpdatedEvent(templateModuleKey, contentTemplate.getSpace());

        return ContentTemplateFactory.buildFromNewPageTemplate(contentTemplate.getSpace(), templateRef, editingPageTemplate, expand);
    }

    @Override
    public ContentTemplate update(ContentTemplate contentTemplate, Expansion... expansions) {
        throw new NotImplementedServiceException("This is not needed, as editing is handled by ContentTemplateService"
                + ". Please use ContentTemplateIdWithId to identify template (long) for editing.");
    }

    @Override
    public TemplateFinder find(Expansion... expansions) {
        return templateFinderFactory.createFinder(this, expansions);
    }

    //TODO CRA-1230 delete this method after ContentTemplateService removed this method.
    public ValidatorImpl validator() {
        return new ValidatorImpl();
    }

    @Override
    public ValidatorImpl validator(ContentTemplateType contentTemplateType) {
        return new ValidatorImpl();
    }

    class ValidatorImpl implements Validator {

        private ValidatorImpl() {
        }

        @Override
        public ValidationResult validateDelete(ContentTemplateId contentTemplateId) {
            return SimpleValidationResult.FORBIDDEN;
        }

        @Override
        public ValidationResult validateCreate(ContentTemplate newContentTemplate) throws ServiceException {
            SimpleValidationResult.Builder simpleValidationResultBuilder = new SimpleValidationResult.Builder();
            if (newContentTemplate == null) {
                return simpleValidationResultBuilder
                        .addMessage(withTranslation("Missing contentTemplate"))
                        .build();
            }

            if (!newContentTemplate.getTemplateType().equals(ContentTemplateType.BLUEPRINT)) {
                return simpleValidationResultBuilder
                        .addMessage(withTranslation("Unsupported ContentTemplateType: " + newContentTemplate.getTemplateType()))
                        .build();
            }

            checkAdminPermission(simpleValidationResultBuilder, getSpace(newContentTemplate));

            String errorMessage = validateTemplate(newContentTemplate);
            if (errorMessage != null) {
                return simpleValidationResultBuilder
                        .addMessage(withTranslation(errorMessage))
                        .build();
            }

            return simpleValidationResultBuilder.build();
        }

        @Override
        public ValidationResult validateUpdate(ContentTemplate contentTemplate) {
            return SimpleValidationResult.FORBIDDEN;
        }

        public ValidationResult validateGet(Option<Space> space) {
            com.atlassian.confluence.spaces.Space entitySpace = null;
            if (space.isDefined()) {
                entitySpace = spaceManager.getSpace(space.get().getKey());
            }

            SimpleValidationResult.Builder simpleValidationResultBuilder = new SimpleValidationResult.Builder();
            checkViewPermission(simpleValidationResultBuilder, entitySpace);
            return simpleValidationResultBuilder.build();
        }

        @Override
        public ValidationResult validateGet(ContentTemplateId contentTemplateId) {
            SimpleValidationResult.Builder simpleValidationResultBuilder = new SimpleValidationResult.Builder();
            try {
                PageTemplate pageTemplate;
                pageTemplate = getPageTemplate(contentTemplateId);
                if (pageTemplate != null) {
                    com.atlassian.confluence.spaces.Space space = pageTemplate.getSpace();
                    Option<Space> templateSpace = Option.none();
                    if (space != null) {
                        templateSpace = Option.some(Space.builder().name(space.getName()).build());
                    }

                    return validateGet(templateSpace);
                }
            } catch (Exception ex) {
                //Some exception happened during locating this contentTemplateId. Following code will return error
                //identical to the case that page cannot be found.
            }

            simpleValidationResultBuilder
                    .addMessage(withTranslation("Cannot find ContentTemplateId: " + contentTemplateId))
                    .build();
            return simpleValidationResultBuilder.build();
        }

        private void checkAdminPermission(SimpleValidationResult.Builder simpleValidationResultBuilder, com.atlassian.confluence.spaces.Space space) {
            checkPermission(simpleValidationResultBuilder, space, Permission.ADMINISTER);
        }

        private void checkViewPermission(SimpleValidationResult.Builder simpleValidationResultBuilder, com.atlassian.confluence.spaces.Space space) {
            checkPermission(simpleValidationResultBuilder, space, Permission.VIEW);
        }

        private void checkPermission(SimpleValidationResult.Builder simpleValidationResultBuilder, com.atlassian.confluence.spaces.Space space, Permission permission) {
            Object target = (space != null) ? space : PermissionManager.TARGET_APPLICATION;
            boolean authorized = permissionManager.hasPermission(AuthenticatedUserThreadLocal.get(), permission, target);
            simpleValidationResultBuilder.authorized(authorized);
        }

        private String validateTemplate(ContentTemplate contentTemplate) {
            String title = contentTemplate.getName();

            if (!isNotEmpty(title))
                return "Template title cannot be null or empty string.";
            else if (title.length() > 255)
                return "Template title must be less than 255 characters.";
            else if (!isTemplateTitleUnique(contentTemplate))
                return "Template title already used, please use a different name.";
            else
                return null;
        }

        //Template name must be unique of the same template type. e.g.: Blueprint template could have the same name as ContentTemplate as long as they are different type
        //of template.
        private boolean isTemplateTitleUnique(ContentTemplate contentTemplate) {
            long templateId = 0;
            if (contentTemplate.getTemplateId() instanceof ContentTemplateId.ContentTemplateIdWithId) {
                templateId = ((ContentTemplateId.ContentTemplateIdWithId) contentTemplate.getTemplateId()).getId();
            }

            PageTemplate otherTemplate = pageTemplateManager.getPageTemplate(contentTemplate.getName(), getSpace(contentTemplate));
            return otherTemplate == null //There is no other template in the same space with same name
                    || getContentTemplateType(otherTemplate) != contentTemplate.getTemplateType()
                    || templateId == otherTemplate.getId(); //If id matches, we are editing.
        }

        private ContentTemplateType getContentTemplateType(PageTemplate otherTemplate) {
            if (StringUtils.isNotBlank(otherTemplate.getModuleKey()))
                return ContentTemplateType.BLUEPRINT;
            else
                return ContentTemplateType.PAGE;
        }
    }


    @Override
    public void delete(ContentTemplateId contentTemplateId) {
        throw new NotImplementedServiceException("This is not needed, as revoking is handled by "
                + "ContentTemplateService");
    }

    private void publishUpdatedEvent(ModuleCompleteKey templateModuleKey, Option<Space> space) {
        com.atlassian.confluence.spaces.Space templateSpace = space.isDefined() ? spaceManager.getSpace(space.get()
                .getKey()) : null;
        String pluginKey = templateModuleKey.getPluginKey();
        String moduleKey = templateModuleKey.getModuleKey();
        BlueprintTemplateUpdateEvent templateUpdatedEvent = new BlueprintTemplateUpdateEvent(this, pluginKey,
                moduleKey, templateSpace);
        eventPublisher.publish(templateUpdatedEvent);
    }

    private void setTemplateBody(ContentTemplate contentTemplate, PageTemplate pageTemplate) {
        ContentBody body;
        if (contentTemplate.getBody().containsKey(ContentRepresentation.STORAGE)) {
            body = contentTemplate.getBody().get(ContentRepresentation.STORAGE);
        } else {
            //There is not a STORAGE representation exists in request, we will find the first convertible representation and convert it to storage.
            Optional<ContentRepresentation> firstConvertiblePresentation =
                    contentTemplate.getBody().keySet().stream().filter(ContentRepresentation::convertsToStorage).findFirst();
            if (!firstConvertiblePresentation.isPresent()) {
                throw new BadRequestException("Cannot find valid content for template from contentTemplate");
            }

            ContentBody bodyToCreate = contentTemplate.getBody().get(firstConvertiblePresentation.get());
            body = contentBodyConversionService.convert(bodyToCreate, ContentRepresentation.STORAGE);
        }

        pageTemplate.setContent(body.getValue());
    }

    private PageTemplate createFirstVersionForPageTemplate(ContentTemplate contentTemplate, PageTemplate
            editingPageTemplate) {
        PageTemplate firstVersion = editingPageTemplate;
        if (contentTemplate.getSpace().isDefined()) {
            if (isEditingTemplateAModifiedGlobalVersion(editingPageTemplate)) {
                firstVersion = new PageTemplate(editingPageTemplate);
            }
            spaceManager.getSpace(contentTemplate.getSpace().get().getKey()).addPageTemplate(firstVersion);
        }

        if (firstVersion.isNew()) {
            pageTemplateManager.savePageTemplate(firstVersion, null);
        }

        return firstVersion;
    }

    private boolean isEditingTemplateAModifiedGlobalVersion(PageTemplate originalPageTemplate) {
        return originalPageTemplate.getSpace() == null && !originalPageTemplate.isNew();
    }

    private PageTemplate getPageTemplate(ContentTemplateId contentTemplateId) {
        if (contentTemplateId instanceof ContentTemplateIdWithUUID) {
            return pluginPageTemplateHelper.getPageTemplate(getContentTemplateRefWithUuid(contentTemplateId));
        } else if (contentTemplateId instanceof ContentTemplateIdWithKeys) {
            ContentTemplateRef templateRef = getContentTemplateRefWithKeys(contentTemplateId);
            return resolvePageTemplate(templateRef);
        }

        return null;
    }

    private ContentTemplateRef getContentTemplateRefWithUuid(ContentTemplateId contentTemplateId) {
        ContentTemplateIdWithUUID uuidId = (ContentTemplateIdWithUUID) contentTemplateId;
        ContentTemplateRef templateRef = contentTemplateRefManager.getById(uuidId.getUuid());
        if (templateRef == null) {
            throw new NotFoundException("No content template ref found with UUID: " + uuidId);
        }

        return templateRef;
    }

    //This is an interesting helper method to find the ContentTemplateRef of space, or global as
    // ContentTemplateIdWithKeys requested.
    private ContentTemplateRef getContentTemplateRefWithKeys(ContentTemplateId contentTemplateId) {
        ContentTemplateIdWithKeys keysId = (ContentTemplateIdWithKeys) contentTemplateId;
        ModuleCompleteKey moduleCompleteKey = new ModuleCompleteKey(keysId.getModuleCompleteKey());
        ContentTemplateRef clonedTemplateRef = contentTemplateRefManager.getCloneByModuleCompleteKey(moduleCompleteKey);
        if (clonedTemplateRef == null) {
            throw new NotFoundException("Cannot locate contentTemplateId: " + contentTemplateId);
        }

        ContentBlueprint clonedBlueprint = clonedTemplateRef.getParent();

        ContentBlueprint blueprintOfSpaceOrGlobal = contentBlueprintManager.getPluginBackedContentBlueprint(
                new ModuleCompleteKey(clonedBlueprint.getModuleCompleteKey()),
                keysId.getSpaceKey().getOrNull());

        if (blueprintOfSpaceOrGlobal.getIndexPageTemplateRef() != null &&
                blueprintOfSpaceOrGlobal.getIndexPageTemplateRef().getModuleCompleteKey().equals(keysId.getModuleCompleteKey())) {
            return blueprintOfSpaceOrGlobal.getIndexPageTemplateRef();
        }

        Optional<ContentTemplateRef> foundRef = blueprintOfSpaceOrGlobal.getContentTemplateRefs()
                .stream()
                .filter(ref -> ref.getModuleCompleteKey().equals(keysId.getModuleCompleteKey()))
                .findFirst();

        if (foundRef.isPresent()) {
            return foundRef.get();
        }

        throw new NotFoundException("Cannot locate contentTemplateId: " + contentTemplateId);
    }

    private com.atlassian.confluence.spaces.Space getSpace(ContentTemplate contentTemplate) {
        com.atlassian.confluence.spaces.Space spaceEntity = null;
        if (contentTemplate.getSpace().isDefined()) {
            spaceEntity = spaceManager.getSpace(contentTemplate.getSpace().get().getKey());
        }

        return spaceEntity;
    }

    private <T extends PluginBackedBlueprint> Collection<T> getDisplayableBlueprints(Collection<T> blueprints, final
    Map<UUID, BlueprintState> blueprintStateMap, final boolean isViewingSpaceTemplateAdmin) {
        return Collections2.filter(blueprints,
                input -> {
                    final UUID blueprintId = input.getId();
                    final BlueprintState blueprintState = blueprintStateMap.get(blueprintId);

                    // Ignore blueprints that are added between the call to get state and the call to the grouper
                    if (blueprintState == null) {
                        return false;
                    }

                    if (blueprintState.isDisabledInPluginSystem()) {
                        return false;
                    }

                    if (blueprintState.isDisabledByWebInterfaceManager()) {
                        return false;
                    }

                    //noinspection RedundantIfStatement
                    if (isViewingSpaceTemplateAdmin && blueprintState.isDisabledGlobally()) {
                        return false;
                    }

                    return true;
                });
    }
}
