package com.atlassian.confluence.plugins.createcontent.impl;


import com.atlassian.confluence.api.impl.service.content.factory.ContentBlueprintInstanceFactory;
import com.atlassian.confluence.api.impl.service.content.factory.ContentTemplateFactory;
import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.Expansions;
import com.atlassian.confluence.api.model.content.ContentBody;
import com.atlassian.confluence.api.model.content.ContentRepresentation;
import com.atlassian.confluence.api.model.content.ContentStatus;
import com.atlassian.confluence.api.model.content.Space;
import com.atlassian.confluence.api.model.content.id.ContentId;
import com.atlassian.confluence.api.model.content.template.ContentBlueprintInstance;
import com.atlassian.confluence.api.model.content.template.ContentTemplate;
import com.atlassian.confluence.api.model.content.template.ContentTemplateId;
import com.atlassian.confluence.api.model.content.template.ContentTemplateType;
import com.atlassian.confluence.api.model.pagination.LimitedRequest;
import com.atlassian.confluence.api.model.pagination.LimitedRequestImpl;
import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.model.pagination.PageResponse;
import com.atlassian.confluence.api.model.pagination.PageResponseImpl;
import com.atlassian.confluence.api.model.validation.SimpleValidationResult;
import com.atlassian.confluence.api.model.validation.ValidationResult;
import com.atlassian.confluence.api.service.content.ContentBodyConversionService;
import com.atlassian.confluence.api.service.exceptions.BadRequestException;
import com.atlassian.confluence.api.service.exceptions.ConflictException;
import com.atlassian.confluence.api.service.exceptions.InternalServerException;
import com.atlassian.confluence.api.service.exceptions.ServiceException;
import com.atlassian.confluence.content.render.xhtml.XhtmlException;
import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.core.ContentPermissionManager;
import com.atlassian.confluence.core.DefaultSaveContext;
import com.atlassian.confluence.core.persistence.confluence.StaleObjectStateException;
import com.atlassian.confluence.labels.Label;
import com.atlassian.confluence.labels.LabelManager;
import com.atlassian.confluence.labels.Labelable;
import com.atlassian.confluence.pages.Draft;
import com.atlassian.confluence.pages.DraftManager;
import com.atlassian.confluence.pages.DraftsTransitionHelper;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.pages.PageManager;
import com.atlassian.confluence.pages.actions.PagePermissionsActionHelper;
import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.pages.templates.PageTemplateManager;
import com.atlassian.confluence.pages.templates.variables.StringVariable;
import com.atlassian.confluence.pages.templates.variables.Variable;
import com.atlassian.confluence.plugins.createcontent.api.services.PageContentTemplateService;
import com.atlassian.confluence.plugins.createcontent.factory.FinderFactory;
import com.atlassian.confluence.security.ContentPermission;
import com.atlassian.confluence.security.Permission;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.user.UserAccessor;
import com.atlassian.confluence.util.LabelUtil;
import com.atlassian.fugue.Option;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.atlassian.confluence.api.model.content.ContentStatus.CURRENT;
import static com.atlassian.confluence.api.model.content.ContentStatus.DRAFT;
import static com.atlassian.confluence.api.model.messages.SimpleMessage.withTranslation;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;


/**
 * Default implementation for TemplateService. Service templates in confluence.
 * The template include page templates and blueprint templates. Currently they are managed by PageTemplateManager
 * and
 *
 * @since 6.0.0
 */
@Component
public class DefaultPageContentTemplateService implements PageContentTemplateService {
    private static final int MAX_PAGE_LIMIT = 200;
    private final PageTemplateManager pageTemplateManager;
    private final ContentTemplateFactory contentTemplateFactory;
    private final SpaceManager spaceManager;
    private final ContentBodyConversionService contentBodyConversionService;
    private final LabelManager labelManager;
    private final PermissionManager permissionManager;
    private final ContentPermissionManager contentPermissionManager;
    private final FinderFactory templateFinderFactory;
    private final DraftsTransitionHelper draftsTransitionHelper;
    private final DraftManager draftManager;
    private final PageManager pageManager;
    private final UserAccessor userAccessor;
    private final ContentBlueprintInstanceFactory contentBlueprintInstanceFactory;

    @Autowired
    public DefaultPageContentTemplateService(
            final @ComponentImport PageTemplateManager pageTemplateManager,
            final @ComponentImport ContentTemplateFactory contentTemplateFactory,
            final @ComponentImport SpaceManager spaceManager,
            final @ComponentImport ContentBodyConversionService contentBodyConversionService,
            final @ComponentImport LabelManager labelManager,
            final @ComponentImport PermissionManager permissionManager,
            final FinderFactory templateFinderFactory,
            final @ComponentImport DraftsTransitionHelper draftTransactionHelper,
            final @ComponentImport DraftManager draftManager,
            final @ComponentImport PageManager pageManager,
            final @ComponentImport UserAccessor userAccessor,
            final @ComponentImport ContentPermissionManager contentPermissionManager,
            final @ComponentImport ContentBlueprintInstanceFactory contentBlueprintInstanceFactory) {
        this.pageTemplateManager = pageTemplateManager;
        this.contentTemplateFactory = contentTemplateFactory;
        this.spaceManager = spaceManager;
        this.contentBodyConversionService = contentBodyConversionService;
        this.labelManager = labelManager;
        this.permissionManager = permissionManager;
        this.templateFinderFactory = templateFinderFactory;
        this.draftsTransitionHelper = draftTransactionHelper;
        this.draftManager = draftManager;
        this.pageManager = pageManager;
        this.userAccessor = userAccessor;
        this.contentPermissionManager = contentPermissionManager;
        this.contentBlueprintInstanceFactory = contentBlueprintInstanceFactory;
    }

    @Override
    public PageResponse<ContentTemplate> getTemplates(ContentTemplateType contentTemplateType, Option<Space> space, PageRequest pageRequest, Expansion... expansions) {
        validator().validateGet(space).throwIfInvalid("Cannot get templates");

        List<PageTemplate> pageTemplates = getPageTemplates(space);

        Function<PageTemplate, ContentTemplate> mapPageTemplateToContentTemplate = pageTemplate ->
                contentTemplateFactory.buildFrom(pageTemplate, new Expansions(expansions));

        List<ContentTemplate> contentTemplates = pageTemplates.stream().map(mapPageTemplateToContentTemplate).collect(Collectors.toList());

        if (pageRequest == null) {
            throw new BadRequestException("PageRequest cannot be null");
        }
        LimitedRequest limitedRequest = LimitedRequestImpl.create(pageRequest, MAX_PAGE_LIMIT);
        return PageResponseImpl.filteredPageResponse(limitedRequest, contentTemplates, null);
    }

    public PageResponse<ContentTemplate> getTemplates(Option<Space> space, PageRequest pageRequest, Expansion... expansions) {
        return getTemplates(ContentTemplateType.PAGE, space, pageRequest, expansions);
    }

    @Override
    public ContentTemplate getTemplate(ContentTemplateId contentTemplateId, Expansion... expansions) {
        validator().validateGet(contentTemplateId).throwIfInvalid("Cannot get template with id: " + contentTemplateId);
        return contentTemplateFactory.buildFrom(getPageTemplate(contentTemplateId), new Expansions(expansions));
    }

    /**
     * Retrieve all the page template from Site or from space if spaceKey is given.
     *
     * @param space
     * @return
     */
    private List<PageTemplate> getPageTemplates(Option<Space> space) {
        if (space.isEmpty()) {
            return pageTemplateManager.getGlobalPageTemplates();
        } else {
            com.atlassian.confluence.spaces.Space spaceEntity = spaceManager.getSpace(space.get().getKey());
            if (spaceEntity == null) {
                throw new IllegalArgumentException("Cannot find the space. spaceKey = " + space.get().getKey());
            }

            return spaceEntity.getPageTemplates();
        }
    }

    private PageTemplate getPageTemplate(ContentTemplateId contentTemplateId) {
        if (contentTemplateId instanceof ContentTemplateId.ContentTemplateIdWithId) {
            ContentTemplateId.ContentTemplateIdWithId id =
                    (ContentTemplateId.ContentTemplateIdWithId) contentTemplateId;
            return pageTemplateManager.getPageTemplate(id.getId());
        } else {
            throw new BadRequestException("ContentTemplateId is not supported: " + contentTemplateId);
        }
    }

    @Override
    public ContentTemplate create(ContentTemplate contentTemplate, Expansion... expansions) {
        validator().validateCreate(contentTemplate).throwIfInvalid("Cannot create ContentTemplate");

        PageTemplate pageTemplate = new PageTemplate();
        pageTemplate.setName(contentTemplate.getName());
        pageTemplate.setDescription(contentTemplate.getDescription());

        setTemplateBody(contentTemplate, pageTemplate);

        com.atlassian.confluence.spaces.Space spaceEntity = getSpace(contentTemplate);
        if (spaceEntity != null) {
            spaceEntity.addPageTemplate(pageTemplate);
        }

        pageTemplateManager.savePageTemplate(pageTemplate, null);
        LabelUtil.syncState(contentTemplate.getLabels(), labelManager, AuthenticatedUserThreadLocal.get(), pageTemplate);

        PageTemplate createdTemplate = pageTemplateManager.getPageTemplate(contentTemplate.getName(), spaceEntity);
        return contentTemplateFactory.buildFrom(createdTemplate, new Expansions(expansions));
    }

    @Override
    public ContentTemplate update(ContentTemplate contentTemplate, Expansion... expansions) {
        validator().validateUpdate(contentTemplate).throwIfInvalid("Cannot update contentTemplate");

        PageTemplate updatingTemplate, originalTemplate;
        updatingTemplate = getPageTemplate(contentTemplate.getTemplateId());
        try {
            originalTemplate = (PageTemplate) updatingTemplate.clone();
        } catch (CloneNotSupportedException ex) {
            throw new ServiceException("Cannot clone PageTemplate: " + updatingTemplate);
        }

        updatingTemplate.setName(contentTemplate.getName());
        updatingTemplate.setDescription(contentTemplate.getDescription());
        setTemplateBody(contentTemplate, updatingTemplate);

        try {
            pageTemplateManager.savePageTemplate(updatingTemplate, originalTemplate);
        } catch (StaleObjectStateException ex) {
            throw new ConflictException("The template being updated has been modified in the database.", ex);
        }

        LabelUtil.syncState(contentTemplate.getLabels(), labelManager, AuthenticatedUserThreadLocal.get(), updatingTemplate);

        PageTemplate updatedTemplate = pageTemplateManager.getPageTemplate(contentTemplate.getName(), getSpace(contentTemplate));
        return contentTemplateFactory.buildFrom(updatedTemplate, new Expansions(expansions));
    }

    @Override
    public void delete(ContentTemplateId contentTemplateId) {
        validator().validateDelete(contentTemplateId).throwIfInvalid("Cannot delete contentTemplateId: " + contentTemplateId);
        pageTemplateManager.removePageTemplate(getPageTemplate(contentTemplateId));
    }

    //TODO CRA-1230 delete this method after ContentTemplateService removed this method.
    public ValidatorImpl validator() {
        return new ValidatorImpl();
    }

    @Override
    public ValidatorImpl validator(ContentTemplateType contentTemplateType) {
        return new ValidatorImpl();
    }

    @Override
    public TemplateFinder find(Expansion... expansions) {
        return templateFinderFactory.createFinder(this, expansions);
    }

    @Override
    public ContentBlueprintInstance createInstance(ContentBlueprintInstance blueprintInstance, Expansion... expansions) {
        validator().validateCreateInstance(blueprintInstance).throwIfInvalid("Cannot create instance from template");

        ContentTemplateId.ContentTemplateIdWithId contentTemplateId = (ContentTemplateId.ContentTemplateIdWithId)
                blueprintInstance.getContentBlueprintSpec().getContentTemplateId().get();
        PageTemplate template = pageTemplateManager.getPageTemplate(contentTemplateId.getId());

        String contentString = generateContentString(blueprintInstance, template);
        List<Label> labels = template.getLabels();

        ContentStatus status = blueprintInstance.getContent().getStatus();
        ContentEntityObject createdEntity;
        if (status.equals(DRAFT)) {
            createdEntity = createDraft(blueprintInstance);

            createdEntity.setTitle(blueprintInstance.getContent().getTitle());
            createdEntity.setBodyAsString(contentString);
            if (DraftsTransitionHelper.isLegacyDraft(createdEntity)) {
                draftManager.saveDraft((Draft) createdEntity);
            } else {
                pageManager.saveContentEntity(createdEntity, DefaultSaveContext.DRAFT);
            }
        } else if (status.equals(CURRENT)) {
            final Page page = new Page();

            page.setTitle(blueprintInstance.getContent().getTitle());
            page.setBodyAsString(contentString);
            com.atlassian.confluence.spaces.Space spaceEntity = spaceManager.getSpace(blueprintInstance.getContent().getSpace().getKey());
            page.setSpace(spaceEntity);
            if (blueprintInstance.getContent().getParent().isDefined()) {
                Page parent = pageManager.getPage(blueprintInstance.getContent().getParent().get().getId().asLong());
                page.setParentPage(parent);
            }
            pageManager.saveContentEntity(page, DefaultSaveContext.DEFAULT);

            createdEntity = page;
        } else {
            //Should never happen, as validator would have rejected invalid content status.
            throw new BadRequestException("Cannot handle content status: " + blueprintInstance.getContent().getStatus());
        }

        addLabels(labels, createdEntity, blueprintInstance.getContentBlueprintSpec().getLabelsString());
        addPermission(blueprintInstance.getContentBlueprintSpec().getViewPermissionUsersString(), createdEntity);
        return contentBlueprintInstanceFactory.convertToInstance(createdEntity, blueprintInstance, expansions);
    }


    private ContentEntityObject createDraft(ContentBlueprintInstance blueprintInstance) {
        ContentEntityObject contentDraft;
        if (blueprintInstance.getContent().getParentId() == ContentId.UNSET) {
            contentDraft = draftsTransitionHelper.createDraft(Page.CONTENT_TYPE,
                    blueprintInstance.getContent().getSpaceRef().exists() ? blueprintInstance.getContent().getSpace().getKey() : null);
        } else {
            contentDraft = draftsTransitionHelper.createDraft(Page.CONTENT_TYPE,
                    blueprintInstance.getContent().getSpaceRef().exists() ? blueprintInstance.getContent().getSpace().getKey() : null,
                    blueprintInstance.getContent().getParentId().asLong());
        }
        return contentDraft;
    }

    private String generateContentString(ContentBlueprintInstance blueprintInstance, PageTemplate template) {
        String contentString;
        try {
            List templateVariables = pageTemplateManager.getTemplateVariables(template);
            if (templateVariables == null || templateVariables.size() == 0) {
                contentString = template.getContent();
            } else {
                List<Variable> variables = convertContextToVariables(blueprintInstance.getContentBlueprintSpec().getContext());
                contentString = pageTemplateManager.mergeVariables(template, variables, null);
            }
        } catch (XhtmlException ex) {
            throw new InternalServerException(ex);
        }

        return contentString;
    }

    private void addLabels(List<Label> label, Labelable labelable, Option<String> labelsString) {
        //First apply all labels from template
        label.forEach(l -> labelManager.addLabel(labelable, l));
        //Second apply all labels from spec.
        if (labelsString.isDefined()) {
            List<String> labelsFromString = LabelUtil.split(labelsString.get());
            labelsFromString.forEach(l -> LabelUtil.addLabel(l, labelManager, labelable));
        }
    }

    private void addPermission(Option<String> viewPermissionsUsers, ContentEntityObject contentEntityObject) {
        if (viewPermissionsUsers.isDefined()) {
            PagePermissionsActionHelper permissionHelper = new PagePermissionsActionHelper(AuthenticatedUserThreadLocal.get(), userAccessor);
            List<ContentPermission> viewPermissions = permissionHelper.createPermissions(ContentPermission.VIEW_PERMISSION, null, viewPermissionsUsers.get());
            contentPermissionManager.setContentPermissions(viewPermissions, contentEntityObject, ContentPermission.VIEW_PERMISSION);
        }
    }

    private List<Variable> convertContextToVariables(Map<String, Object> context) {
        List<Variable> variables = new ArrayList();
        context.forEach((key, value) -> {
            Variable variable = new StringVariable(key, value.toString());
            variables.add(variable);
        });

        return variables;
    }

    class ValidatorImpl implements PageContentTemplateService.Validator {

        private ValidatorImpl() {
        }

        @Override
        public ValidationResult validateDelete(ContentTemplateId contentTemplateId) {
            SimpleValidationResult.Builder simpleValidationResultBuilder = new SimpleValidationResult.Builder();
            simpleValidationResultBuilder.authorized(true);

            if (contentTemplateId == null) {
                return simpleValidationResultBuilder
                        .addMessage(withTranslation("Missing contentTemplateId"))
                        .build();
            }

            com.atlassian.confluence.spaces.Space space;
            PageTemplate pageTemplate = getPageTemplate(contentTemplateId);
            if (pageTemplate == null) {
                return simpleValidationResultBuilder
                        .addMessage(withTranslation("Template doesn't exist"))
                        .build();
            }

            checkAdminPermission(simpleValidationResultBuilder, pageTemplate.getSpace());
            return simpleValidationResultBuilder.build();
        }

        @Override
        public ValidationResult validateCreate(ContentTemplate newContentTemplate) throws ServiceException {
            SimpleValidationResult.Builder simpleValidationResultBuilder = new SimpleValidationResult.Builder();
            simpleValidationResultBuilder.authorized(true);

            if (newContentTemplate == null) {
                return simpleValidationResultBuilder
                        .addMessage(withTranslation("Missing contentTemplate"))
                        .build();
            }

            String errorMessage = validateTemplate(newContentTemplate);
            if (errorMessage != null) {
                return simpleValidationResultBuilder
                        .addMessage(withTranslation(errorMessage))
                        .build();
            }

            checkAdminPermission(simpleValidationResultBuilder, getSpace(newContentTemplate));
            return simpleValidationResultBuilder.build();
        }

        @Override
        public ValidationResult validateUpdate(ContentTemplate contentTemplate) {
            ValidationResult result = validateGet(contentTemplate.getTemplateId());
            if (!result.isValid()) return result;

            return validateCreate(contentTemplate);
        }

        public ValidationResult validateGet(Option<Space> space) {
            com.atlassian.confluence.spaces.Space entitySpace = null;
            if (space.isDefined()) {
                entitySpace = spaceManager.getSpace(space.get().getKey());
            }

            SimpleValidationResult.Builder simpleValidationResultBuilder = new SimpleValidationResult.Builder();
            checkViewPermission(simpleValidationResultBuilder, entitySpace);
            return simpleValidationResultBuilder.build();
        }

        @Override
        public ValidationResult validateGet(ContentTemplateId contentTemplateId) {
            SimpleValidationResult.Builder simpleValidationResultBuilder = new SimpleValidationResult.Builder();
            try {
                PageTemplate pageTemplate = getPageTemplate(contentTemplateId);
                if (pageTemplate != null) {
                    com.atlassian.confluence.spaces.Space space = pageTemplate.getSpace();
                    Option<Space> templateSpace = Option.none();
                    if (space != null) {
                        templateSpace = Option.some(Space.builder().name(space.getName()).build());
                    }

                    return validateGet(templateSpace);
                }
            } catch (Exception ex) {
                //Some exception happened during locating this contentTemplateId. Following code will return error
                //identical to the case that page cannot be found.
            }

            simpleValidationResultBuilder
                    .addMessage(withTranslation("Cannot find ContentTemplateId: " + contentTemplateId))
                    .build();
            return simpleValidationResultBuilder.build();
        }

        @Override
        public ValidationResult validateCreateInstance(ContentBlueprintInstance instance) {
            SimpleValidationResult.Builder simpleValidationResultBuilder = new SimpleValidationResult.Builder();
            simpleValidationResultBuilder.authorized(true);

            if (instance == null ||
                    instance.getContentBlueprintSpec().getContentTemplateId().isEmpty() ||
                    !(instance.getContentBlueprintSpec().getContentTemplateId().get() instanceof ContentTemplateId.ContentTemplateIdWithId)) {
                return simpleValidationResultBuilder
                        .addMessage(withTranslation("Instance doesn't have a Valid ContentTemplateId provided"))
                        .build();
            }

            ContentStatus status = instance.getContent().getStatus();
            if (!ImmutableList.of(DRAFT, CURRENT).contains(status)) {
                return simpleValidationResultBuilder
                        .addMessage(withTranslation("Status of content must be DRAFT or CURRENT, supplied value: " + status))
                        .build();
            }

            if (instance.getContent().getSpace() == null) {
                return simpleValidationResultBuilder
                        .addMessage(withTranslation("Content must have a space"))
                        .build();
            }

            com.atlassian.confluence.spaces.Space entitySpace = spaceManager.getSpace(instance.getContent().getSpace().getKey());
            checkEditPermission(simpleValidationResultBuilder, entitySpace);

            return simpleValidationResultBuilder.build();
        }

        private void checkAdminPermission(SimpleValidationResult.Builder simpleValidationResultBuilder, com.atlassian.confluence.spaces.Space space) {
            checkPermission(simpleValidationResultBuilder, space, Permission.ADMINISTER);
        }

        private void checkViewPermission(SimpleValidationResult.Builder simpleValidationResultBuilder, com.atlassian.confluence.spaces.Space space) {
            checkPermission(simpleValidationResultBuilder, space, Permission.VIEW);
        }

        private void checkEditPermission(SimpleValidationResult.Builder simpleValidationResultBuilder, com.atlassian.confluence.spaces.Space space) {
            checkPermission(simpleValidationResultBuilder, space, Permission.EDIT);
        }

        private void checkPermission(SimpleValidationResult.Builder simpleValidationResultBuilder, com.atlassian.confluence.spaces.Space space, Permission permission) {
            Object target = (space != null) ? space : PermissionManager.TARGET_APPLICATION;
            boolean authorized = permissionManager.hasPermission(AuthenticatedUserThreadLocal.get(), permission, target);
            simpleValidationResultBuilder.authorized(authorized);
        }

        private String validateTemplate(ContentTemplate contentTemplate) {
            String title = contentTemplate.getName();

            if (!isNotEmpty(title))
                return "Template title cannot be null or empty string.";
            else if (title.length() > 255)
                return "Template title must be less than 255 characters.";
            else if (!isTemplateTitleUnique(contentTemplate))
                return "Template title already used, please use a different name.";
            else
                return null;
        }

        //Template name must be unique of the same template type. e.g.: Blueprint template could have the same name as ContentTemplate as long as they are different type
        //of template.
        private boolean isTemplateTitleUnique(ContentTemplate contentTemplate) {
            long templateId = 0;
            if (contentTemplate.getTemplateId() instanceof ContentTemplateId.ContentTemplateIdWithId) {
                templateId = ((ContentTemplateId.ContentTemplateIdWithId) contentTemplate.getTemplateId()).getId();
            }

            PageTemplate otherTemplate = pageTemplateManager.getPageTemplate(contentTemplate.getName(), getSpace(contentTemplate));
            return otherTemplate == null //There is no other template in the same space with same name
                    || getContentTemplateType(otherTemplate) != contentTemplate.getTemplateType()
                    || templateId == otherTemplate.getId(); //If id matches, we are editing.
        }

        private ContentTemplateType getContentTemplateType(PageTemplate otherTemplate) {
            if (StringUtils.isNotBlank(otherTemplate.getModuleKey()))
                return ContentTemplateType.BLUEPRINT;
            else
                return ContentTemplateType.PAGE;
        }
    }

    private void setTemplateBody(ContentTemplate contentTemplate, PageTemplate pageTemplate) {
        ContentBody body;
        if (contentTemplate.getBody().containsKey(ContentRepresentation.STORAGE)) {
            body = contentTemplate.getBody().get(ContentRepresentation.STORAGE);
        } else {
            //There is not a STORAGE representation exists in request, we will find the first convertible representation and convert it to storage.
            Optional<ContentRepresentation> firstConvertiblePresentation =
                    contentTemplate.getBody().keySet().stream().filter(ContentRepresentation::convertsToStorage).findFirst();
            if (!firstConvertiblePresentation.isPresent()) {
                throw new BadRequestException("Cannot find valid content for template from contentTemplate");
            }

            ContentBody bodyToCreate = contentTemplate.getBody().get(firstConvertiblePresentation.get());
            body = contentBodyConversionService.convert(bodyToCreate, ContentRepresentation.STORAGE);
        }

        pageTemplate.setContent(body.getValue());
    }

    private com.atlassian.confluence.spaces.Space getSpace(ContentTemplate contentTemplate) {
        com.atlassian.confluence.spaces.Space spaceEntity = null;
        if (contentTemplate.getSpace().isDefined()) {
            spaceEntity = spaceManager.getSpace(contentTemplate.getSpace().get().getKey());
        }

        return spaceEntity;
    }
}
