package com.atlassian.confluence.ext.usage.index;

import com.atlassian.bonnie.BonnieConstants;
import com.atlassian.bonnie.ILuceneConnection;
import com.atlassian.bonnie.LuceneConnection;
import com.atlassian.confluence.core.ContentEntityManager;
import com.atlassian.confluence.ext.usage.UsageConstants;
import com.atlassian.confluence.ext.usage.event.UsageEventWrapperTask;
import com.atlassian.confluence.ext.usage.query.ContentUsageQuery;
import com.atlassian.confluence.ext.usage.query.TopUserQuery;
import com.atlassian.confluence.ext.usage.query.UsageDataUtils;
import com.atlassian.confluence.labels.Label;
import com.atlassian.confluence.search.lucene.filter.ContentPermissionsFilter;
import com.atlassian.confluence.security.Permission;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.security.SpacePermission;
import com.atlassian.confluence.security.SpacePermissionManager;
import com.atlassian.confluence.setup.BootstrapManager;
import com.atlassian.confluence.setup.ConfluenceBootstrapConstants;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.user.UserAccessor;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.user.User;
import com.atlassian.util.profiling.UtilTimerStack;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.document.DateTools;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.AtomicReaderContext;
import org.apache.lucene.search.Collector;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Scorer;
import org.jfree.data.time.RegularTimePeriod;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import static java.util.Collections.emptyList;
import static org.apache.commons.lang3.StringUtils.isEmpty;

/**
 * Manage the interactions with the index
 */
@Component
public class UsageIndexManager implements InitializingBean, DisposableBean {
    private static final Logger log = LoggerFactory.getLogger(UsageIndexManager.class);
    private static final int OPTIMIZATION_FREQUENCY = 1000;

    private ILuceneConnection connection;
    private int optimizeCounter = 0;
    private final SpaceManager spaceManager;
    private final SpacePermissionManager spacePermissionManager;
    private final BootstrapManager bootstrapManager;
    private final ContentEntityManager contentEntityManager;
    private final UserAccessor userAccessor;
    private final PermissionManager permissionManager;

    @Autowired
    public UsageIndexManager(
            final @ComponentImport SpacePermissionManager spacePermissionManager,
            final @ComponentImport SpaceManager spaceManager,
            final @ComponentImport UserAccessor userAccessor,
            final @ComponentImport BootstrapManager bootstrapManager,
            final @ComponentImport ContentEntityManager contentEntityManager,
            final @ComponentImport PermissionManager permissionManager) {
        this.spacePermissionManager = spacePermissionManager;
        this.spaceManager = spaceManager;
        this.userAccessor = userAccessor;
        this.bootstrapManager = bootstrapManager;
        this.contentEntityManager = contentEntityManager;
        this.permissionManager = permissionManager;
    }

    @Override
    public void afterPropertiesSet() {
        String indexDir = bootstrapManager.getFilePathProperty(ConfluenceBootstrapConstants.LUCENE_INDEX_DIR_PROP) + File.separator + "plugin" + File.separator + "usage";
        File f = new File(indexDir);
        if (!f.exists()) {
            f.mkdirs();
        }
        connection = new LuceneConnection(f, new StandardAnalyzer(BonnieConstants.LUCENE_VERSION));
    }

    @Override
    public void destroy() {
        if (connection != null) {
            connection.close();
            connection = null;
        }
    }

    @SuppressWarnings("checkstyle:AnonInnerLength")
    public TimeSeriesCollection queryUsage(final ContentUsageQuery q) {
        final TimeSeriesCollection dataset = new TimeSeriesCollection();
        final RegularTimePeriod[] range = new RegularTimePeriod[]{null, null};

        UtilTimerStack.push("UsageIndexManager::queryUsage");

        /* Filter out spaces that the user is not permitted to see */
        User user = AuthenticatedUserThreadLocal.getUser();
        if (filterSpacesForUser(q.getSpaces(), user) != 0 && q.getSpaces().isEmpty()) {
            return dataset;
        }

        if (connection != null) {
            connection.withSearch(new LuceneConnection.SearcherAction() {
                public void perform(final IndexSearcher searcher) throws IOException {
                    searcher.search(q.getLuceneQuery(), new Collector() {
                        private AtomicReaderContext context;

                        @Override
                        public void collect(int docId) {
                            try {
                                Document d = context.reader().document(docId);
                                collectResult(dataset, d, q);
                            } catch (IOException e) {
                                log.error("IO error", e);
                            } catch (ParseException e) {
                                log.error("Parse error", e);
                            }
                        }

                        private void collectResult(TimeSeriesCollection collection, Document document, ContentUsageQuery q) throws ParseException {
                            Date date = DateTools.stringToDate(document.get(UsageConstants.FIELD_DATE));
                            RegularTimePeriod timePeriod = UsageDataUtils.getTimePeriod(q.getPeriod(), date);

                            String column = "Events";

                            if (q.getColumns() != null) {
                                if (UsageConstants.COLUMNS_EVENTS.equals(q.getColumns())) {
                                    column = document.get(UsageConstants.FIELD_EVENTTYPE);
                                } else if (UsageConstants.COLUMNS_SPACES.equals(q.getColumns())) {
                                    column = document.get(UsageConstants.FIELD_SPACEID);
                                } else if (UsageConstants.COLUMNS_TYPES.equals(q.getColumns())) {
                                    column = document.get(UsageConstants.FIELD_ENTITYTYPE);
                                }
                            }

                            TimeSeries series = collection.getSeries(column);

                            if (series == null) {
                                series = new TimeSeries(column, UsageDataUtils.getTimePeriodClass(q.getPeriod()));
                                collection.addSeries(series);
                            }

                            Number value = series.getValue(timePeriod);

                            if (value == null)
                                series.add(timePeriod, Integer.valueOf(1));
                            else
                                series.update(timePeriod, Integer.valueOf(1 + value.intValue()));

                            // now update the range
                            if (range[0] == null || timePeriod.compareTo(range[0]) < 0)
                                range[0] = timePeriod;

                            if (range[1] == null || timePeriod.compareTo(range[1]) > 0)
                                range[1] = timePeriod;
                        }

                        @Override
                        public void setNextReader(AtomicReaderContext context) {
                            this.context = context;
                        }

                        @Override
                        public boolean acceptsDocsOutOfOrder() {
                            return false;
                        }

                        @Override
                        public void setScorer(Scorer scorer) {
                        }
                    });
                }
            });
        }

        UsageDataUtils.normaliseDateRange(dataset, range);

        UtilTimerStack.pop("UsageIndexManager::queryUsage");

        return dataset;
    }

    /**
     * Filters a collection of spaces by removing those which the user does not have view permission for
     *
     * @param spaces Colelction to be filtered
     * @param user   User for permission checking
     * @return Number of spaces filtered
     */
    protected int filterSpacesForUser(Collection spaces, User user) {
        int filterCount = 0;
        for (Iterator iter = spaces.iterator(); iter.hasNext(); ) {
            Space space = (Space) iter.next();
            if (!hasViewSpacePermission(space, user)) {
                iter.remove();
                ++filterCount;
            }
        }
        return filterCount;
    }

    public void index(final UsageEventWrapperTask task) {
        if (task == null)
            return;

        if (connection != null) {
            connection.withWriter(indexWriter -> {
                if (log.isDebugEnabled())
                    log.debug("Indexing: " + task);
                indexWriter.addDocument(task.getDocument());
            });

            optimizeCounter--;

            if (optimizeCounter <= 0) {
                optimizeCounter = OPTIMIZATION_FREQUENCY;
                connection.optimize();
            }
        }
    }

    /**
     * Find the popular items that match a query.
     *
     * @param q   The query
     * @param max max
     * @return A list of the most popular content entities, wrapped in a small object.
     */
    @SuppressWarnings("checkstyle:AnonInnerLength")
    public List<PopularResult> queryPopular(ContentUsageQuery q, int max) {
        List<PopularResult> results = new LinkedList<>();

        UtilTimerStack.push("UsageIndexManager::queryPopular");

        if (connection != null) {
            final Map<String, PopularResult> unorderedResults = new HashMap<>();

            final ContentUsageQuery queryToRun;
            final boolean spaceQuery;

            // we need to rewrite the query if they're looking for popular spaces as this is really popular content _in_ spaces
            if (q.getContentTypes().contains(UsageConstants.SPACE_ENTITY_TYPE)) {
                spaceQuery = true;
                queryToRun = new ContentUsageQuery(q);
                queryToRun.setContentTypes(emptyList());
            } else {
                spaceQuery = false;
                queryToRun = q;
            }
            final User user = AuthenticatedUserThreadLocal.get();
            final List<String> groupNames = userAccessor.getGroupNames(user);

            connection.withSearch(new LuceneConnection.SearcherAction() {
                public void perform(final IndexSearcher searcher) throws IOException {
                    searcher.search(queryToRun.getLuceneQuery(), new ContentPermissionsFilter(user, groupNames), new Collector() {
                        private AtomicReaderContext context;

                        @Override
                        public void collect(int docId) throws IOException {
                            Document d = context.reader().document(docId);
                            String entityId, entityType;

                            if (spaceQuery) {
                                entityId = d.get(UsageConstants.FIELD_SPACEID);
                                entityType = UsageConstants.SPACE_ENTITY_TYPE;
                            } else {
                                entityId = d.get(UsageConstants.FIELD_ENTITYID);
                                entityType = d.get(UsageConstants.FIELD_ENTITYTYPE);
                            }

                            PopularResult cr = unorderedResults.get(entityId);

                            if (cr == null)
                                cr = new PopularResult(spaceManager, contentEntityManager, entityId, entityType);

                            // Ignore historical events (with no current space or content)
                            if (cr.getSpace() == null && cr.getContent() == null)
                                return;

                            cr.incrementCount();
                            unorderedResults.put(entityId, cr);
                        }

                        @Override
                        public void setNextReader(AtomicReaderContext context) {
                            this.context = context;
                        }

                        @Override
                        public void setScorer(Scorer scorer) {
                        }

                        @Override
                        public boolean acceptsDocsOutOfOrder() {
                            return false;
                        }
                    });
                }
            });

            Collection<PopularResult> filteredResults = filterPopularResults(unorderedResults.values(), user);
            if (!queryToRun.getLabels().isEmpty()) {
                filteredResults = filterPopularResultsByLabels(filteredResults, queryToRun.getLabels());
            }
            results.addAll(filteredResults);

            Collections.sort(results);

            if (results.size() > max)
                results = results.subList(0, max);
        }

        UtilTimerStack.pop("UsageIndexManager::queryPopular");

        return results;
    }

    protected Collection<PopularResult> filterPopularResults(Collection<PopularResult> results, User user) {
        List<PopularResult> filteredResults = new LinkedList<>();
        for (PopularResult result : results) {
            if (hasPermissionForResult(result, user))
                filteredResults.add(result);
        }
        return filteredResults;
    }

    /**
     * Filter list of popular results by labels set on query. Must match all labels supplied.
     *
     * @param results results
     * @param labels  labels
     * @return filtered labels
     */
    protected Collection<PopularResult> filterPopularResultsByLabels(Collection<PopularResult> results, Collection<Label> labels) {
        List<PopularResult> filteredResults = new LinkedList<>();
        for (PopularResult result : results) {
            if (result.getLabels().containsAll(labels))
                filteredResults.add(result);
        }
        return filteredResults;
    }


    /**
     * Find active users based on a query
     *
     * @param q   User query
     * @param max Maximum number of results to return
     * @return List of matching users in descending order of activity
     */
    public List<UserResult> queryTopUsers(final TopUserQuery q, int max) {
        List<UserResult> results = new ArrayList<>();

        if (connection != null) {
            final Map<String, UserResult> unorderedResults = new HashMap<>();

            connection.withSearch(new LuceneConnection.SearcherAction() {
                public void perform(final IndexSearcher searcher) throws IOException {
                    searcher.search(q.getLuceneQuery(), new Collector() {
                        private AtomicReaderContext context;

                        @Override
                        public void collect(int docId) throws IOException {
                            Document d = context.reader().document(docId);
                            String entityId = d.get("user");

                            UserResult cr = unorderedResults.get(entityId);

                            if (cr == null) {
                                cr = new UserResult(entityId, userAccessor);
                                unorderedResults.put(entityId, cr);
                            }

                            cr.incrementCount();
                        }

                        @Override
                        public void setNextReader(AtomicReaderContext context) {
                            this.context = context;
                        }

                        @Override
                        public boolean acceptsDocsOutOfOrder() {
                            return false;
                        }

                        @Override
                        public void setScorer(Scorer scorer) {
                        }
                    });
                }
            });

            results.addAll(unorderedResults.values());

            Collections.sort(results);

            if (results.size() > max) {
                return results.subList(0, max);
            }
        }

        return results;
    }

    protected boolean hasViewSpacePermission(Space space, User user) {
        return spacePermissionManager.hasPermission(SpacePermission.VIEWSPACE_PERMISSION, space, user);
    }

    /**
     * @param result Popular result to permission check
     * @param user   User for permission checking
     * @return True if the user has permission to see the result
     */
    protected boolean hasPermissionForResult(PopularResult result, User user) {
        if (UsageConstants.SPACE_ENTITY_TYPE.equalsIgnoreCase(result.getEntityType()))
            return hasViewSpacePermission(result.getSpace(), user);
        else
            return permissionManager.hasPermission(user, Permission.VIEW, result.getContent());
    }

    protected boolean hasViewPermissionInIndex(Document document, User user) {
        String userPermission = document.get("userpermission");
        String[] groupPermissions = document.getValues("grouppermission");

        if (isEmpty(userPermission) && (groupPermissions == null || groupPermissions.length == 0))
            return true;

        if (userPermission != null && user != null && userPermission.equals(user.getName()))
            return true;

        if (groupPermissions == null || groupPermissions.length == 0)
            return false;

        List groups = Arrays.asList(groupPermissions);

        for (String groupName : userAccessor.getGroupNames(user)) {
            if (groups.contains(groupName))
                return true;
        }

        return false;
    }

    public SpaceManager getSpaceManager() {
        return spaceManager;
    }

    public SpacePermissionManager getSpacePermissionManager() {
        return spacePermissionManager;
    }

    public BootstrapManager getBootstrapManager() {
        return bootstrapManager;
    }

    public ContentEntityManager getContentEntityManager() {
        return contentEntityManager;
    }

    public UserAccessor getUserAccessor() {
        return userAccessor;
    }

    public PermissionManager getPermissionManager() {
        return permissionManager;
    }
}

