package com.atlassian.confluence.extra.flyingpdf.util;

import com.atlassian.annotations.Internal;
import com.lowagie.text.pdf.PdfAction;
import com.lowagie.text.pdf.PdfArray;
import com.lowagie.text.pdf.PdfDestination;
import com.lowagie.text.pdf.PdfDictionary;
import com.lowagie.text.pdf.PdfName;
import com.lowagie.text.pdf.PdfObject;
import com.lowagie.text.pdf.PdfReader;
import com.lowagie.text.pdf.PdfStamper;
import com.lowagie.text.pdf.PdfString;

import java.util.Map;
import java.util.regex.Matcher;

/**
 * Fix links between pages like /display/FOO/bar to PDF bookmarks. It loads entire PDF file in memory, so it won't
 * work for really huge PDF files. We execute this under sandbox exactly for this reason.
 */
@Internal
public class BookmarksPageProcessor implements PdfPageProcessor {

    private final String spaceKey;
    private final Map<String, Integer> locationByTitle;
    private final String baseUrl;

    public BookmarksPageProcessor(String spaceKey, Map<String, Integer> locationByTitle, String baseUrl) {
        this.spaceKey = spaceKey;
        this.locationByTitle = locationByTitle;
        this.baseUrl = baseUrl;
    }

    @Override
    public void processPage(PdfReader reader, PdfStamper stamper, int page) {
        final PdfDictionary pageDict = reader.getPageN(page);
        final PdfArray annotations = pageDict.getAsArray(PdfName.ANNOTS);

        if (annotations == null || annotations.length() == 0) {
            return;
        }

        for (Object annotation : annotations.getArrayList()) {
            final PdfDictionary annotationDictionary = (PdfDictionary) PdfReader.getPdfObject((PdfObject) annotation);

            // Make sure this annotation has a link
            if (!annotationDictionary.get(PdfName.SUBTYPE).equals(PdfName.LINK)) {
                continue;
            }

            // Make sure this annotation has an ACTION
            if (annotationDictionary.get(PdfName.A) == null) {
                continue;
            }

            // Get the ACTION for the current annotation
            PdfDictionary action = (PdfDictionary) annotationDictionary.get(PdfName.A);

            // Test if it is a URI action
            if (action.get(PdfName.S).equals(PdfName.URI)) {
                final PdfString currentUri = (PdfString) action.get(PdfName.URI);

                final boolean isOnThisServer = currentUri.toString().trim().startsWith("/")
                        || currentUri.toString().trim().startsWith(baseUrl);

                if (!isOnThisServer) {
                    // We don't process links to another resources
                    continue;
                }

                final Matcher matcher = UrlUtils.pageDisplayUrlPattern.matcher(currentUri.toUnicodeString());

                // If URI is internal and matches space key, we can make a bookmark out of it
                if (matcher.find()) {
                    final String spaceKey = matcher.group(1);
                    final String pageTitle = UrlUtils.decodeTitle(matcher.group(2));

                    if (this.spaceKey.equals(spaceKey) && locationByTitle.containsKey(pageTitle)) {
                        final Integer location = locationByTitle.get(pageTitle);
                        final PdfAction newAction = PdfAction.gotoLocalPage(location,
                                new PdfDestination(0), stamper.getWriter());

                        annotationDictionary.put(PdfName.A, newAction);
                    }
                }
            }
        }
    }
}
