package com.atlassian.confluence.extra.flyingpdf.upgrade;

import com.atlassian.confluence.extra.flyingpdf.config.FontManager;
import com.atlassian.confluence.setup.settings.ConfluenceDirectories;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.sal.api.ApplicationProperties;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.core.io.FileSystemResource;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.IOException;

import static com.atlassian.confluence.extra.flyingpdf.config.FontsDirectoryFontDao.FONTS_DIR;

/**
 * This component acts like an upgrade task for the plugin. It will run when the plugin is first enabled and ensure that
 * if the Confluence instance was configured with a custom font for use in PDF Export then it will be made available to
 * this PDF Export plugin.
 */
@Component
public class UpgradePdfLanguageSupport implements InitializingBean {
    private static final Logger LOGGER = Logger.getLogger(UpgradePdfLanguageSupport.class);

    private final FontManager pdfExportFontManager;
    private final ConfluenceDirectories confluenceDirectories;
    private final ApplicationProperties applicationProperties;

    public UpgradePdfLanguageSupport(
            @ComponentImport ConfluenceDirectories confluenceDirectories,
            @ComponentImport ApplicationProperties applicationProperties,
            FontManager pdfExportFontManager) {
        this.applicationProperties = applicationProperties;
        this.pdfExportFontManager = pdfExportFontManager;
        this.confluenceDirectories = confluenceDirectories;
    }

    /**
     * Once this bean is fully initialized it will check the Confluence font directory and if a custom font is found,
     * ensure that it is made available to the new PDF Export mechanism.
     */
    @Override
    public void afterPropertiesSet() {
        if (pdfExportFontManager.isCustomFontInstalled()) {
            // this is not necessary - the custom font has been installed correctly at some previous point
            return;
        }
        File fontDir = new File(applicationProperties.getHomeDirectory(), FONTS_DIR);
        if (!fontDir.exists())
            return;

        File[] fontFiles = fontDir.listFiles();
        if ((fontFiles == null) || (fontFiles.length == 0))
            return;

        if (fontFiles.length > 1) {
            LOGGER.warn("More than one file was found in the font directory "
                    + "- custom font will not be migrated for use with the PDF Export plugin.");
            return;
        }

        try {
            // move the font to the temp directory so it can be installed again into the fonts directory
            final File tempFontDir = confluenceDirectories.getTempDirectory()
                    .resolve(String.valueOf(System.currentTimeMillis()))
                    .toFile();

            if (!tempFontDir.mkdir())
                throw new IOException("Failed to create the temporary font storage directory " + tempFontDir);

            File tempFontFile = new File(tempFontDir, fontFiles[0].getName());
            if (!fontFiles[0].renameTo(tempFontFile)) {
                throw new IOException("Could not move the currently installed font " + fontFiles[0].getAbsolutePath()
                        + " to a temporary location of " + tempFontDir);
            }

            pdfExportFontManager.installFont(new FileSystemResource(tempFontFile));
            LOGGER.info("Successfully made the font " + fontFiles[0] + " available for use in the PDF export plugin.");
        } catch (IOException ex) {
            LOGGER.error("Failed to make the font " + fontFiles[0]
                    + " available for use in the PDF export plugin. Please install the font manually.", ex);
        }
    }

}
