package com.atlassian.confluence.extra.flyingpdf.sandbox;

import com.atlassian.confluence.importexport.resource.DownloadResourceManager;
import com.atlassian.confluence.importexport.resource.DownloadResourceNotFoundException;
import com.atlassian.confluence.importexport.resource.DownloadResourceReader;
import com.atlassian.confluence.importexport.resource.UnauthorizedDownloadResourceException;
import com.atlassian.confluence.util.sandbox.SandboxCallback;
import com.atlassian.confluence.util.sandbox.SandboxCallbackContext;
import com.atlassian.confluence.util.sandbox.SandboxSerializer;
import com.atlassian.renderer.util.UrlUtil;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.util.Optional;

/**
 * This callback allows for Sandbox to ask Confluence for some resource during PDF rendering (e.g. image)
 */
public class GetResourceCallback implements SandboxCallback<GetResourceCallbackRequest, GetResourceCallbackResponse> {
    private static final Logger log = LoggerFactory.getLogger(GetResourceCallback.class);

    @Override
    public GetResourceCallbackResponse apply(SandboxCallbackContext context, GetResourceCallbackRequest request) {
        Optional<DownloadResourceManager> downloadResourceManager = context.get(DownloadResourceManager.class);

        if (!downloadResourceManager.isPresent()) {
            log.error("DownloadResourceManager is not registered in callback context");
            return new GetResourceCallbackResponse(new byte[0]);
        }

        String decodedUri = request.getUri();
        String strippedUri = stripQueryString(decodedUri);
        try {
            DownloadResourceReader resourceReader = downloadResourceManager.get().getResourceReader(
                    request.getUsername(), strippedUri, UrlUtil.getQueryParameters(decodedUri));

            try (InputStream inputStream = resourceReader.getStreamForReading()) {
                if (inputStream == null) {
                    log.warn("There was a problem with fetching attachment for {}", request.getUri());
                    return new GetResourceCallbackResponse(new byte[0]);
                }

                return new GetResourceCallbackResponse(IOUtils.toByteArray(inputStream));
            }
        } catch (UnauthorizedDownloadResourceException | DownloadResourceNotFoundException | IOException e) {
            return new GetResourceCallbackResponse(new byte[0]);
        }
    }

    @Override
    public SandboxSerializer<GetResourceCallbackRequest> inputSerializer() {
        return GetResourceCallbackRequest.serializer();
    }

    @Override
    public SandboxSerializer<GetResourceCallbackResponse> outputSerializer() {
        return GetResourceCallbackResponse.serializer();
    }

    private String stripQueryString(String uri) {
        int queryIndex = uri.indexOf('?');
        if (queryIndex > 0)
            uri = uri.substring(0, queryIndex);
        return uri;
    }
}
