package com.atlassian.confluence.extra.flyingpdf.sandbox;

import org.apache.commons.lang3.SerializationUtils;
import org.w3c.dom.Document;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSSerializer;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.IOException;
import java.io.Serializable;
import java.io.StringReader;

class DocumentSerialisation {

    private static final DocumentBuilder documentBuilder = createDocumentBuilder();

    /**
     * We might use java serialisation instead of DOM -> String -> DOM because the latter is not idempotent.
     * If the document is malformed (we do have those), desearialising SAX parser will be unable to parse the String.
     * It is a good question whether it is OK to care about malformed documents, but we don't want to introduce
     * export regressions right now and that's why we go a bit hacky, but more resilient java serialisation way.
     */
    static byte[] serializeViaJava(Document document) {
        return SerializationUtils.serialize((Serializable) document);
    }

    static Document deserializeViaJava(byte[] bytes) {
        return (Document) SerializationUtils.deserialize(bytes);
    }

    static byte[] serializeViaString(Document document) {
        DOMImplementationLS implementation = (DOMImplementationLS) document.getImplementation();
        LSSerializer serializer = implementation.createLSSerializer();
        return serializer.writeToString(document).getBytes();
    }

    static Document deserializeViaString(byte[] bytes) {
        InputSource inputSource = new InputSource();
        inputSource.setCharacterStream(new StringReader(new String(bytes)));

        Document document;
        try {
            document = documentBuilder.parse(inputSource);
        } catch (SAXException | IOException e) {
            throw new RuntimeException(e);
        }

        return document;
    }

    private static DocumentBuilder createDocumentBuilder() {
        try {
            DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();

            // disable DTD validation
            builder.setEntityResolver((publicId, systemId) -> new InputSource(new StringReader("")));
            return builder;
        } catch (ParserConfigurationException e) {
            throw new RuntimeException(e);
        }
    }
}
