package com.atlassian.confluence.extra.flyingpdf.impl;

import com.atlassian.confluence.core.persistence.confluence.SessionFactoryTypeEnum;
import com.atlassian.confluence.core.persistence.confluence.SessionFactoryTypeThreadLocal;
import com.atlassian.confluence.extra.flyingpdf.PdfExportProgressMonitor;
import com.atlassian.confluence.extra.flyingpdf.PdfExporterService;
import com.atlassian.confluence.extra.flyingpdf.analytic.SpaceExportMetrics;
import com.atlassian.confluence.extra.flyingpdf.html.DecorationPolicy;
import com.atlassian.confluence.importexport.ImportExportManager;
import com.atlassian.confluence.pages.ContentTree;
import com.atlassian.confluence.security.GateKeeper;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.util.RequestCacheThreadLocal;
import com.atlassian.confluence.util.longrunning.ConfluenceAbstractLongRunningTask;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import com.atlassian.user.User;
import org.apache.commons.lang3.StringUtils;

import java.io.File;
import java.util.List;

public class PdfExportLongRunningTask extends ConfluenceAbstractLongRunningTask {

    private TransactionTemplate transactionTemplate;

    private ImportExportManager importExportManager;

    private SpaceManager spaceManager;

    private PdfExporterService flyingPdfExporterService;

    private GateKeeper gateKeeper;

    private final List<String> contentToBeExported;

    private final long spaceId;

    private final User remoteUser;

    private final String contextPath;

    private final PdfExportSemaphore pdfExportSemaphore;

    private final DecorationPolicy decorations;

    /**
     * Output from the task - the generated PDF
     */
    private String downloadPath;

    public PdfExportLongRunningTask(List<String> contentToBeExported, Space space, User remoteUser,
                                    String contextPath, PdfExportSemaphore pdfExportSemaphore, DecorationPolicy decorations) {
        this.contentToBeExported = contentToBeExported;
        this.spaceId = space.getId();
        this.remoteUser = remoteUser;
        this.contextPath = contextPath;
        this.pdfExportSemaphore = pdfExportSemaphore;
        this.decorations = decorations;
    }

    @Override
    protected void runInternal() {
        try {
            pdfExportSemaphore.run(this::doRunInternal);
        } catch (RuntimeException e) {
            progress.setCompletedSuccessfully(false);
            progress.setStatus(e.getMessage());
        }
    }

    private void doRunInternal() {
        SessionFactoryTypeThreadLocal.set(SessionFactoryTypeEnum.NON_CACHING);
        try {
            transactionTemplate.execute(() -> {
                AuthenticatedUserThreadLocal.setUser(remoteUser);
                RequestCacheThreadLocal.getRequestCache().put(RequestCacheThreadLocal.CONTEXT_PATH_KEY, contextPath);

                PdfExportProgressMonitor monitor = flyingPdfExporterService.createProgressMonitor(progress);

                File pdfFile;
                try {
                    monitor.beginCalculationOfContentTree();
                    Space space = spaceManager.getSpace(spaceId);
                    ContentTree contentTree = importExportManager.getContentTree(remoteUser, space);
                    if (contentToBeExported != null && !contentToBeExported.isEmpty()) {
                        contentTree.filter(contentToBeExported);
                    }
                    monitor.completedCalculationOfContentTree(contentTree.size());

                    pdfFile = flyingPdfExporterService.createPdfForSpace(remoteUser, space, contentTree,
                            monitor, contextPath, new SpaceExportMetrics(), decorations);

                    downloadPath = importExportManager.prepareDownloadPath(pdfFile.getAbsolutePath());
                    gateKeeper.addKey(downloadPath, remoteUser);
                    downloadPath = contextPath + downloadPath;
                    monitor.completed(downloadPath);
                } catch (Exception ex) {
                    log.error("Error during PDF export", ex);
                    String exceptionMessage = ex.getMessage();
                    if (StringUtils.isBlank(exceptionMessage))
                        exceptionMessage = ex.getClass().getName();
                    monitor.errored(exceptionMessage);
                }
                return null;
            });

        } finally {
            SessionFactoryTypeThreadLocal.clear();
        }
    }

    public String getDownloadPath() {
        return downloadPath;
    }

    public String getName() {
        return "PDF Space Export";
    }

    public String getNameKey() {
        return "com.atlassian.confluence.extra.flyingpdf.exporttaskname";
    }

    public void setTransactionTemplate(TransactionTemplate transactionTemplate) {
        this.transactionTemplate = transactionTemplate;
    }

    public void setImportExportManager(ImportExportManager importExportManager) {
        this.importExportManager = importExportManager;
    }

    public void setSpaceManager(SpaceManager spaceManager) {
        this.spaceManager = spaceManager;
    }

    public void setFlyingPdfExporterService(PdfExporterService flyingPdfExporterService) {
        this.flyingPdfExporterService = flyingPdfExporterService;
    }

    public void setGateKeeper(GateKeeper gateKeeper) {
        this.gateKeeper = gateKeeper;
    }

}
