package com.atlassian.confluence.extra.flyingpdf.impl;

import com.atlassian.confluence.core.persistence.confluence.SessionFactoryTypeEnum;
import com.atlassian.confluence.core.persistence.confluence.SessionFactoryTypeThreadLocal;
import com.atlassian.confluence.importexport.ImportExportManager;
import com.atlassian.confluence.importexport.impl.ExportFileNameGenerator;
import com.atlassian.confluence.pages.ContentTree;
import com.atlassian.confluence.renderer.radeox.macros.MacroUtils;
import com.atlassian.confluence.security.GateKeeper;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.util.i18n.I18NBean;
import com.atlassian.confluence.util.longrunning.ConfluenceAbstractLongRunningTask;
import com.atlassian.confluence.util.velocity.VelocityUtils;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import com.atlassian.user.User;
import org.apache.commons.lang3.StringUtils;
import org.apache.velocity.context.Context;

import java.io.File;
import java.io.FileWriter;

import static java.util.Collections.singletonList;

/**
 * Long running task, which is used to build and render a content tree, if user wants to choose content to be
 * exported to pdf.
 *
 * @since 3.4.5
 */
public class ContentTreeLongRunningTask extends ConfluenceAbstractLongRunningTask {
    private final ImportExportManager importExportManager;
    private final TransactionTemplate transactionTemplate;
    private final I18NBean i18NBean;
    private final Space space;
    private final User user;
    private final GateKeeper gateKeeper;
    private final String contextPath;
    private final ExportFileNameGenerator pdfExportFileNameGenerator;
    private final PdfExportSemaphore pdfExportSemaphore;
    private String downloadPath;

    public ContentTreeLongRunningTask(ImportExportManager importExportManager,
                                      TransactionTemplate transactionTemplate, I18NBean i18NBean, Space space, User user,
                                      GateKeeper gateKeeper, String contextPath, ExportFileNameGenerator pdfExportFileNameGenerator,
                                      PdfExportSemaphore pdfExportSemaphore) {
        this.importExportManager = importExportManager;
        this.transactionTemplate = transactionTemplate;
        this.i18NBean = i18NBean;
        this.space = space;
        this.user = user;
        this.gateKeeper = gateKeeper;
        this.contextPath = contextPath;
        this.pdfExportFileNameGenerator = pdfExportFileNameGenerator;
        this.pdfExportSemaphore = pdfExportSemaphore;
    }

    @Override
    protected void runInternal() {
        try {
            pdfExportSemaphore.run(this::doRunInternal);
        } catch (RuntimeException e) {
            progress.setCompletedSuccessfully(false);
            progress.setStatus(e.getMessage());
        }
    }

    private void doRunInternal() {
        SessionFactoryTypeThreadLocal.set(SessionFactoryTypeEnum.NON_CACHING);
        try {
            transactionTemplate.execute(() -> {
                try {
                    initProgress();

                    ContentTree contentTree = importExportManager.getContentTree(user, space);
                    Context context = MacroUtils.createDefaultVelocityContext();
                    context.put("contentTree", contentTree);

                    File file = pdfExportFileNameGenerator.getExportFile("confluence.extra.content-tree-builder-");
                    FileWriter writer = new FileWriter(file);

                    try {
                        VelocityUtils.writeRenderedTemplate(writer, "/templates/extra/pdfexport/export-space-common-tree.vm",
                                context);
                    } finally {
                        writer.close();
                    }

                    downloadPath = importExportManager.prepareDownloadPath(file.getAbsolutePath());
                    gateKeeper.addKey(downloadPath, user);
                    downloadPath = contextPath + downloadPath;

                    updateProgress(downloadPath);
                } catch (Exception e) {
                    log.error("Error during building content tree for PDF export", e);
                    updateProgress(e);
                }
                return null;
            });
        } finally {
            SessionFactoryTypeThreadLocal.clear();
        }
    }

    private void updateProgress(Exception e) {
        String exceptionMessage = e.getMessage();
        if (StringUtils.isBlank(exceptionMessage)) {
            exceptionMessage = e.getClass().getName();
        }

        progress.setPercentage(100);
        progress.setCompletedSuccessfully(false);
        progress.setStatus(i18NBean.getText("com.atlassian.confluence.extra.flyingpdf.progress.contenttreeerrored",
                singletonList(exceptionMessage)));
    }

    private void updateProgress(String downloadPath) {
        progress.setPercentage(100);
        progress.setCompletedSuccessfully(true);
        progress.setStatus(downloadPath);
    }

    private void initProgress() {
        progress.setPercentage(0);
        progress.setStatus("");
    }

    @Override
    public String getName() {
        return "Content tree build for PDF export";
    }

    public String getDownloadPath() {
        return downloadPath;
    }
}
