package com.atlassian.confluence.extra.flyingpdf.impl;

import com.atlassian.confluence.cluster.ClusterManager;
import com.atlassian.confluence.cluster.ClusterNodeInformation;
import com.atlassian.confluence.extra.flyingpdf.PdfExportProgressMonitor;
import com.atlassian.confluence.extra.flyingpdf.PdfExporterService;
import com.atlassian.confluence.extra.flyingpdf.analytic.EnvironmentInfo;
import com.atlassian.confluence.extra.flyingpdf.analytic.ExportResults;
import com.atlassian.confluence.extra.flyingpdf.analytic.ExportScope;
import com.atlassian.confluence.extra.flyingpdf.analytic.ExportStatus;
import com.atlassian.confluence.extra.flyingpdf.analytic.FailureLocation;
import com.atlassian.confluence.extra.flyingpdf.analytic.PageExportMetrics;
import com.atlassian.confluence.extra.flyingpdf.analytic.PdfExportAnalyticEvent;
import com.atlassian.confluence.extra.flyingpdf.analytic.SpaceExportMetrics;
import com.atlassian.confluence.extra.flyingpdf.html.DecorationPolicy;
import com.atlassian.confluence.importexport.ImportExportException;
import com.atlassian.confluence.pages.AbstractPage;
import com.atlassian.confluence.pages.ContentTree;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.core.util.ProgressMeter;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.spring.scanner.annotation.export.ExportAsService;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.user.User;
import org.springframework.stereotype.Component;

import java.io.File;
import java.util.Optional;

import static com.atlassian.confluence.extra.flyingpdf.analytic.SandboxStatus.NOT_USED;
import static com.atlassian.confluence.extra.flyingpdf.analytic.SandboxStatus.USED;

/**
 * Wrapper around real PDF service that gathers analytic and audit data and pushes it to Analytics, Diagnostics and Auditing.
 * It is done by filling some info into {@link PageExportMetrics} or {@link SpaceExportMetrics}, passing them to real
 * services and then using this data for analytics.
 * <p>
 * Real PDF service should fill any known information to the Metrics objects. Using this pattern we can understand
 * real cause of problems.
 */
@Component
@ExportAsService(PdfExporterService.class)
public class BigBrotherPdfExporterService implements PdfExporterService {

    private final EventPublisher eventPublisher;
    private final ClusterManager clusterManager;
    private final DelegatingPdfExporterService delegate;

    public BigBrotherPdfExporterService(
            @ComponentImport EventPublisher eventPublisher,
            @ComponentImport ClusterManager clusterManager,
            DelegatingPdfExporterService delegate) {
        this.eventPublisher = eventPublisher;
        this.clusterManager = clusterManager;
        this.delegate = delegate;
    }

    @Override
    public File createPdfForSpace(User user, Space space, ContentTree contentTree, PdfExportProgressMonitor progress,
                                  String contextPath, SpaceExportMetrics spaceExportMetrics, DecorationPolicy decorations) throws ImportExportException {
        spaceExportMetrics.setConfluencePages(contentTree.size());
        fillEnvironmentData(space, spaceExportMetrics.getEnvironmentInfo());

        final long startTime = System.currentTimeMillis();
        try {
            final File result = delegate.createPdfForSpace(user, space, contentTree, progress, contextPath, spaceExportMetrics, decorations);
            spaceExportMetrics.getExportResults().setExportStatus(ExportStatus.OK);
            return result;
        } catch (Exception e) {
            setGenericError(spaceExportMetrics.getExportResults());
            throw e;
        } finally {
            spaceExportMetrics.setTotalTime((int) (System.currentTimeMillis() - startTime));
            eventPublisher.publish(new PdfExportEvent(space));
            eventPublisher.publish(new PdfExportAnalyticEvent(spaceExportMetrics));
        }
    }

    @Override
    public File createPdfForPage(User user, AbstractPage page, String contextPath, PageExportMetrics pageExportMetrics) throws ImportExportException {
        pageExportMetrics.setPageId(page.getId());
        pageExportMetrics.setPageRevision(page.getConfluenceRevision().hashCode());
        fillEnvironmentData(page, pageExportMetrics.getEnvironmentInfo());

        try {
            return delegate.createPdfForPage(user, page, contextPath, pageExportMetrics);
        } catch (Exception e) {
            pageExportMetrics.getExportResults().setFailureLocation(FailureLocation.PAGE);
            setGenericError(pageExportMetrics.getExportResults());
            throw e;
        } finally {
            eventPublisher.publish(new PdfExportEvent(page));
            eventPublisher.publish(new PdfExportAnalyticEvent(pageExportMetrics));
        }
    }

    private void setGenericError(ExportResults results) {
        if (results.getExportStatus() == null) {
            results.setExportStatus(ExportStatus.FAIL);
            results.setFailureLocation(FailureLocation.INTERNAL);
        }
    }

    private void fillEnvironmentData(Space space, EnvironmentInfo environmentInfo) {
        if (clusterManager.isClustered()) {
            environmentInfo.setDcNodeId(
                    Optional.ofNullable(clusterManager.getThisNodeInformation())
                            .map(ClusterNodeInformation::getAnonymizedNodeIdentifier)
                            .map(Object::hashCode)
                            .orElse(-1));
        }
        environmentInfo.setSpaceKey(space.getKey());
        environmentInfo.setSpaceName(space.getName());
        environmentInfo.setExportScope(ExportScope.SPACE);
        environmentInfo.setSandboxStatus(delegate.sandboxIsUsed() ? USED : NOT_USED);
    }

    private void fillEnvironmentData(AbstractPage page, EnvironmentInfo environmentInfo) {
        Space space = page.getSpace();
        environmentInfo.setPageName(page.getTitle());
        environmentInfo.setPageType(page.getType());
        environmentInfo.setSpaceName(space.getName());
        environmentInfo.setSpaceKey(space.getKey());
        environmentInfo.setExportScope(ExportScope.PAGE);
    }

    @Override
    public ContentTree getContentTree(User user, Space space) {
        return delegate.getContentTree(user, space);
    }

    @Override
    public boolean isPermitted(User user, AbstractPage page) {
        return delegate.isPermitted(user, page);
    }

    @Override
    public boolean isPermitted(User user, Space space) {
        return delegate.isPermitted(user, space);
    }

    @Override
    public boolean exportableContentExists(Space space) {
        return delegate.exportableContentExists(space);
    }

    @Override
    public PdfExportProgressMonitor createProgressMonitor(ProgressMeter progressMeter) {
        return delegate.createProgressMonitor(progressMeter);
    }
}


