package com.atlassian.confluence.extra.flyingpdf.config;

import com.atlassian.bandana.BandanaManager;
import com.atlassian.confluence.setup.bandana.ConfluenceBandanaContext;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Component;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

/**
 * An implementation of the font manager that will store a custom font to be used in PDF export in the Confluence home
 * directory.
 * <p>
 * The name of the custom font is stored in Map in bandana where the key is the name give to the font and the value is
 * the font file name. This isn't necessary at the moment since only one custom font is supported but should we later
 * extend the font manager to support multiple custom fonts it will save us the need to support two different bandana
 * storage formats in later implementations of this class.
 */
@Component
public class PdfExportFontManager implements FontManager {
    private static final String FONT_NAMES_KEY = "com.atlassian.confluence.extra.flyingpdf.fontname";
    private static final String DEFAULT_FONT_NAME = "customfont";

    private static final Logger LOGGER = Logger.getLogger(PdfExportFontManager.class);

    private final FontDao fontDao;
    private final BandanaManager bandanaManager;
    private final EventPublisher eventPublisher;

    public PdfExportFontManager(FontDao fontDao, @ComponentImport BandanaManager bandanaManager, @ComponentImport EventPublisher eventPublisher) {
        this.fontDao = fontDao;
        this.bandanaManager = bandanaManager;
        this.eventPublisher = eventPublisher;
    }

    public FileSystemResource getInstalledFont() {
        String customFont = getFontFileName(DEFAULT_FONT_NAME);
        if (StringUtils.isBlank(customFont))
            return null;
        Resource fontResource;
        try {
            fontResource = fontDao.getFont(customFont);
        } catch (IOException ex) {
            LOGGER.debug("No font resource could be found with the name " + customFont, ex);
            return null;
        }
        if (!(fontResource instanceof FileSystemResource)) {
            LOGGER.warn("The fontDao did not return the font " + customFont + " as a FileSystemResource");
            return null;
        }
        return (FileSystemResource) fontResource;
    }

    public void installFont(Resource fontResource) throws IOException {
        String fontName;
        try {
            fontName = fontResource.getFilename();
        } catch (IllegalStateException ex) {
            throw new IOException("The supplied fontResource did not include a filename property.");
        }

        fontDao.saveFont(fontName, fontResource);
        String oldFont = getFontFileName(DEFAULT_FONT_NAME);

        // if this new font has a different name from the old one then remove the old one (protects against removing
        // the font that has just been saved).
        if (StringUtils.isNotBlank(oldFont) && !oldFont.equals(fontName)) {
            fontDao.removeFont(oldFont);
        }
        // Only update bandana entry if the name is different from previously.
        if (!fontName.equals(oldFont))
            storeFontFileName(DEFAULT_FONT_NAME, fontName);
        // update the font data across the cluster regardless of name change.
        eventPublisher.publish(new CustomFontInstalledEvent("PDF Export Font Manager", fontName, getFontData(fontResource)));
    }

    /**
     * Get the font data.
     *
     * @param fontResource
     * @return
     * @throws IOException
     */
    private byte[] getFontData(Resource fontResource) throws IOException {
        if (!fontResource.exists())
            throw new IOException("The font resource cannot be found for transfer to other nodes in the cluster: "
                    + fontResource.getDescription());

        ByteArrayOutputStream fontDataStream = new ByteArrayOutputStream();
        InputStream istream = fontResource.getInputStream();
        try {
            IOUtils.copy(istream, fontDataStream);
        } finally {
            IOUtils.closeQuietly(fontDataStream);
        }
        return fontDataStream.toByteArray();
    }

    public boolean isCustomFontInstalled() {
        FileSystemResource customFont = getInstalledFont();
        return customFont != null;
    }

    public void removeInstalledFont() throws IOException {
        String customFont = getFontFileName(DEFAULT_FONT_NAME);
        if (StringUtils.isBlank(customFont)) {
            LOGGER.debug("No custom font is installed.");
            return;
        }
        try {
            fontDao.removeFont(customFont);
        } catch (IOException ex) {
            LOGGER.warn("Failed to remove the custom font " + customFont, ex);
            return;
        }
        // clear the stored font
        storeFontFileName(DEFAULT_FONT_NAME, "");
        // remove the font from the other nodes in the cluster
        eventPublisher.publish(new CustomFontRemovedEvent("PDF Export Font Manager", customFont));
    }

    /**
     * Get the file name for a named font from bandana
     *
     * @param fontName
     * @return the file name for the font or null if it is not stored.
     */
    private String getFontFileName(String fontName) {
        Map<String, String> fontNameMap = (Map<String, String>) bandanaManager.getValue(
                ConfluenceBandanaContext.GLOBAL_CONTEXT, FONT_NAMES_KEY);
        if (fontNameMap != null)
            return fontNameMap.get(fontName);
        return null;
    }

    /**
     * Store the font details in bandana.
     *
     * @param fontName
     * @param fontFileName
     */
    private void storeFontFileName(String fontName, String fontFileName) {
        Map<String, String> fontNameMap = (Map<String, String>) bandanaManager.getValue(
                ConfluenceBandanaContext.GLOBAL_CONTEXT, FONT_NAMES_KEY);
        if (fontNameMap == null)
            fontNameMap = new HashMap<>(1);
        fontNameMap.put(fontName, fontFileName);
        bandanaManager.setValue(ConfluenceBandanaContext.GLOBAL_CONTEXT, FONT_NAMES_KEY, fontNameMap);
    }

}
