/**
 *
 */
package com.atlassian.confluence.extra.flyingpdf;

import com.atlassian.confluence.extra.flyingpdf.analytic.PageExportMetrics;
import com.atlassian.confluence.extra.flyingpdf.analytic.SpaceExportMetrics;
import com.atlassian.confluence.extra.flyingpdf.html.DecorationPolicy;
import com.atlassian.confluence.extra.flyingpdf.impl.NoOpProgressMonitor;
import com.atlassian.confluence.importexport.ImportExportException;
import com.atlassian.confluence.pages.AbstractPage;
import com.atlassian.confluence.pages.ContentTree;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.core.util.ProgressMeter;
import com.atlassian.user.User;

import java.io.File;

/**
 * A service responsible for creating a PDF representation of Confluence content.
 */
public interface PdfExporterService {

    default File createPdfForSpace(User user, Space space, ContentTree contentTree, String contextPath,
                                   SpaceExportMetrics spaceExportMetrics)
            throws ImportExportException {
        return createPdfForSpace(user, space, contentTree, new NoOpProgressMonitor(), contextPath, spaceExportMetrics, DecorationPolicy.none());
    }

    File createPdfForSpace(User user, Space space, ContentTree contentTree,
                           PdfExportProgressMonitor progress, String contextPath,
                           SpaceExportMetrics spaceExportMetrics, DecorationPolicy decorations) throws ImportExportException;

    File createPdfForPage(User user, AbstractPage page, String contextPath, PageExportMetrics pageExportMetrics) throws ImportExportException;

    /**
     * Return the content tree for the given space. This can be used to generate a UI for selecting pages to be exported
     * for a space.
     * @param user  the user
     * @param space the space to be exported
     * @return content of space that needs to be exported.
     */
    ContentTree getContentTree(User user, Space space);

    /**
     * Check whether the specified user has permission to export the specified page.
     *
     * @param user the user
     * @param page the page to be exported
     * @return true if the user is permitted; otherwise false.
     */
    boolean isPermitted(User user, AbstractPage page);

    /**
     * Check whether the specified user has permission to export the specified space.
     *
     * @param user  the user
     * @param space the space to be exported
     * @return true if the user is permitted to export the space; otherwise false.
     */
    boolean isPermitted(User user, Space space);

    /**
     * Check if there is something to export for the specified user in the specified space.
     *
     * @param space the space to be exported
     * @return <code>true</code> if there is at least one page to export for given user in the current space,
     * <code>false</code> otherwise
     */
    boolean exportableContentExists(Space space);

    /**
     * Wrappers are different for different pdf exporter services (via regular one or via sandbox)
     * So only the service could create a properly progress monitor
     *
     * @param progressMeter progress meter from long running task
     * @return monitor wrapper
     * @since 6.0.0
     */
    PdfExportProgressMonitor createProgressMonitor(ProgressMeter progressMeter);
}
