package com.atlassian.confluence.extra.flyingpdf.sandbox;

import com.atlassian.confluence.extra.flyingpdf.util.BookmarksPageProcessor;
import com.atlassian.confluence.extra.flyingpdf.util.PageNumbersPageProcessor;
import com.atlassian.confluence.extra.flyingpdf.util.PdfPageProcessor;
import com.atlassian.confluence.extra.flyingpdf.util.PdfPostProcessor;
import com.atlassian.confluence.extra.flyingpdf.util.PdfJoiner;
import com.atlassian.confluence.extra.flyingpdf.util.PdfUtils;
import com.atlassian.confluence.util.sandbox.SandboxSerializer;
import com.atlassian.confluence.util.sandbox.SandboxTask;
import com.atlassian.confluence.util.sandbox.SandboxTaskContext;
import com.google.common.collect.Lists;
import com.lowagie.text.DocumentException;

import java.io.File;
import java.io.IOException;
import java.util.List;

import static com.atlassian.confluence.extra.flyingpdf.html.DecorationPolicy.DecorationComponent.PAGE_NUMBERS;

/**
 * PDF join is done in sandbox because it uses 3rd party code and can potentially behave badly
 */
public class SandboxPdfJoinTask implements SandboxTask<SandboxPdfJoinRequest, SandboxPdfJoinResponse> {

    @Override
    public SandboxPdfJoinResponse apply(SandboxTaskContext context, SandboxPdfJoinRequest request) {
        final String intermediateFile = request.getOutputFile() + "-partial";
        try {
            // Join partial files together and write to intermediate file
            new PdfJoiner(intermediateFile, request.getExportedSpaceStructure())
                    .join();

            // Page processors will run on each individual page over one pass of the whole PDF file
            List<PdfPageProcessor> pageProcessors = Lists.newArrayList(
                    new BookmarksPageProcessor(request.getSpaceKey(), request.getExportedSpaceStructure().locationByTitleMap(), request.getBaseUrl()));

            if (request.getDecorationPolicy().components().contains(PAGE_NUMBERS)) {
                pageProcessors.add(new PageNumbersPageProcessor());
            }

            // Read intermediate file, apply page processors and write to final file
            new PdfPostProcessor(intermediateFile, request.getOutputFile(), pageProcessors)
                    .run();

        } catch (IOException | DocumentException e) {
            throw new RuntimeException(e);
        } finally {
            //noinspection ResultOfMethodCallIgnored
            new File(intermediateFile).delete();
        }

        final File outputFile = new File(request.getOutputFile());
        return new SandboxPdfJoinResponse(outputFile, PdfUtils.numberOfPages(outputFile));
    }

    @Override
    public SandboxSerializer<SandboxPdfJoinRequest> inputSerializer() {
        return SandboxPdfJoinRequest.serializer();
    }

    @Override
    public SandboxSerializer<SandboxPdfJoinResponse> outputSerializer() {
        return SandboxPdfJoinResponse.serializer();
    }
}
