package com.atlassian.confluence.extra.flyingpdf.sandbox;

import com.atlassian.confluence.util.sandbox.SandboxSerializer;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import static com.atlassian.confluence.util.sandbox.SandboxSerializers.compositeByteArraySerializer;
import static com.atlassian.confluence.util.sandbox.SandboxSerializers.intSerializer;
import static com.atlassian.confluence.util.sandbox.SandboxSerializers.stringSerializer;

public class SandboxPdfConversionResponse {

    private final boolean wasUnableToReadInputDocument;

    private final File resultingPdf;

    private final int pdfPagesNum;

    public static SandboxPdfConversionResponse ok(File resultingPdf, int pdfPagesNum) {
        return new SandboxPdfConversionResponse(false, resultingPdf, pdfPagesNum);
    }

    public static SandboxPdfConversionResponse couldNotReadInputDocument() {
        return new SandboxPdfConversionResponse(true, null, -1);
    }

    public SandboxPdfConversionResponse(boolean wasUnableToReadInputDocument, File resultingPdf, int pdfSize) {
        this.wasUnableToReadInputDocument = wasUnableToReadInputDocument;
        this.resultingPdf = resultingPdf;
        this.pdfPagesNum = pdfSize;
    }

    public boolean wasUnableToReadInputDocument() {
        return wasUnableToReadInputDocument;
    }

    public File getResultingPdf() {
        return resultingPdf;
    }

    public int getPdfPagesNum() {
        return pdfPagesNum;
    }

    static Serializer serializer() {
        return Serializer.instance;
    }

    static final class Serializer implements SandboxSerializer<SandboxPdfConversionResponse> {
        static final Serializer instance = new SandboxPdfConversionResponse.Serializer();

        private Serializer() {
        }

        @Override
        public byte[] serialize(SandboxPdfConversionResponse conversionResponse) {
            List<byte[]> fields = new ArrayList<>();

            fields.add(intSerializer().serialize(conversionResponse.wasUnableToReadInputDocument ? 1 : 0));

            final String filePathOrNull = Optional.ofNullable(conversionResponse.getResultingPdf())
                    .map(File::getAbsolutePath)
                    .orElse(null);
            fields.add(stringSerializer().serialize(filePathOrNull));
            fields.add(intSerializer().serialize(conversionResponse.pdfPagesNum));

            return compositeByteArraySerializer().serialize(fields);
        }

        @Override
        public SandboxPdfConversionResponse deserialize(byte[] bytes) {
            List<byte[]> fields = compositeByteArraySerializer().deserialize(bytes);
            return new SandboxPdfConversionResponse(
                    intSerializer().deserialize(fields.get(0)) == 1,
                    new File(stringSerializer().deserialize(fields.get(1))),
                    intSerializer().deserialize(fields.get(2)));
        }
    }
}
