package com.atlassian.confluence.extra.flyingpdf.sandbox;

import com.atlassian.confluence.util.sandbox.SandboxSerializer;
import org.w3c.dom.Document;

import java.util.ArrayList;
import java.util.List;

import static com.atlassian.confluence.util.sandbox.SandboxSerializers.compositeByteArraySerializer;
import static com.atlassian.confluence.util.sandbox.SandboxSerializers.intSerializer;
import static com.atlassian.confluence.util.sandbox.SandboxSerializers.stringSerializer;

public class SandboxPdfConversionRequest {
    private final String baseUrl;
    private final String contextPath;
    private final String cdnUrl;
    private final String fontPath;
    private final String username;
    private final String exportFile;
    private final boolean encodeHtmlAsString;
    private final byte[] document;

    /**
     * Request instance where the document is encoded as printed XHTML
     */
    static SandboxPdfConversionRequest stringEncoded(String baseurl, String contextPath, String cdnUrl, String fontPath,
                                                     String username, String exportFile, Document document) {
        return new SandboxPdfConversionRequest(baseurl, contextPath, cdnUrl, fontPath, username, exportFile, true,
                DocumentSerialisation.serializeViaString(document));
    }

    /**
     * Request instance where the document is encoded like Java-serialised Document instance
     */
    static SandboxPdfConversionRequest serializationEncoded(String baseurl, String contextPath, String cdnUrl, String fontPath,
                                                            String username, String exportFile, Document document) {
        return new SandboxPdfConversionRequest(baseurl, contextPath, cdnUrl, fontPath, username, exportFile, false,
                DocumentSerialisation.serializeViaJava(document));
    }

    public SandboxPdfConversionRequest(String baseUrl, String contextPath, String cdnUrl, String fontPath,
                                       String username, String exportFile, boolean encodeHtmlAsString, byte[] document) {
        this.baseUrl = baseUrl;
        this.contextPath = contextPath;
        this.cdnUrl = cdnUrl;
        this.fontPath = fontPath;
        this.username = username;
        this.exportFile = exportFile;
        this.encodeHtmlAsString = encodeHtmlAsString;
        this.document = document;
    }

    public String getBaseUrl() {
        return baseUrl;
    }

    public String getContextPath() {
        return contextPath;
    }

    public String getCdnUrl() {
        return cdnUrl;
    }

    public String getFontPath() {
        return fontPath;
    }

    public String getUsername() {
        return username;
    }

    public String getExportFile() {
        return exportFile;
    }

    public byte[] getDocument() {
        return document;
    }

    public boolean isEncodeHtmlAsString() {
        return encodeHtmlAsString;
    }

    static Serializer serializer() {
        return Serializer.instance;
    }

    static final class Serializer implements SandboxSerializer<SandboxPdfConversionRequest> {
        static final Serializer instance = new Serializer();

        private Serializer() {
        }

        @Override
        public byte[] serialize(SandboxPdfConversionRequest request) {
            List<byte[]> fields = new ArrayList<>();

            fields.add(stringSerializer().serialize(request.getBaseUrl()));
            fields.add(stringSerializer().serialize(request.getContextPath()));
            fields.add(stringSerializer().serialize(request.getCdnUrl()));
            fields.add(stringSerializer().serialize(request.getFontPath()));
            fields.add(stringSerializer().serialize(request.getUsername()));
            fields.add(stringSerializer().serialize(request.getExportFile()));
            fields.add(intSerializer().serialize(request.isEncodeHtmlAsString() ? 1 : 0));
            fields.add(request.getDocument());

            return compositeByteArraySerializer().serialize(fields);
        }

        @Override
        public SandboxPdfConversionRequest deserialize(byte[] bytes) {
            final List<byte[]> fields = compositeByteArraySerializer().deserialize(bytes);
            final String baseUrl = stringSerializer().deserialize(fields.get(0));
            final String contextPath = stringSerializer().deserialize(fields.get(1));
            final String cdnUrl = stringSerializer().deserialize(fields.get(2));
            final String fontPath = stringSerializer().deserialize(fields.get(3));
            final String username = stringSerializer().deserialize(fields.get(4));
            final String exportFile = stringSerializer().deserialize(fields.get(5));
            final boolean documentEncodedAsString = intSerializer().deserialize(fields.get(6)) == 1;
            final byte[] document = fields.get(7);

            return new SandboxPdfConversionRequest(
                    baseUrl,
                    contextPath,
                    cdnUrl,
                    fontPath,
                    username,
                    exportFile,
                    documentEncodedAsString,
                    document);
        }

    }
}
