package com.atlassian.confluence.extra.flyingpdf.impl;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.StringUtils;
import org.apache.commons.codec.net.URLCodec;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.xhtmlrenderer.pdf.ITextOutputDevice;
import org.xhtmlrenderer.pdf.ITextUserAgent;
import org.xhtmlrenderer.resource.ImageResource;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

public abstract class AbstractExportUserAgent extends ITextUserAgent {
    private static final String FILE_PREFIX = "file:";
    private static final String DATA_PREFIX = "data:";
    private static final Pattern RESOURCE_PATH_PATTERN = Pattern.compile("/s/(.*)/_/");

    private final String baseUrl;
    private final String cdnUrl;

    private final boolean bypassCdn = !Boolean.getBoolean("pdf.export.allow.cdn.access");

    public AbstractExportUserAgent(ITextOutputDevice device, String baseUrl, @Nullable String cdnUrl) {
        super(device);
        this.baseUrl = baseUrl;
        this.cdnUrl = cdnUrl;
    }

    public ImageResource getImageResource(final String uri) {
        try {
            return new ImageResource(uri, new LightITextFSImage(() -> resolveAndOpenStream(uri),
                    getSharedContext().getDotsPerPixel(), baseUrl, uri));
        } catch (Exception e) {
            log(Level.SEVERE, "Can't get image resource for uri" + uri + ", error: " + e.getMessage());
        }

        // failure while reading resource
        return new ImageResource(null, null);
    }

    @Override
    protected InputStream resolveAndOpenStream(String uri) {
        if (uri == null) {
            return null;
        }

        if (shrinkImageCacheBeforeFetching()) {
            shrinkImageCache();
        }

        String effectiveUri = uri;
        if (bypassCdn && isNotBlank(cdnUrl) && effectiveUri.startsWith(cdnUrl)) {
            // CDN configured to serve static resources, we want to point it back to the local host rather than hit the CDN,
            // potentially polluting the CDN cache and requiring two way access to it CONFSRVDEV-11389
            effectiveUri = effectiveUri.replace(cdnUrl, baseUrl);
        }

        String relativeUri = effectiveUri;
        if (relativeUri.startsWith(FILE_PREFIX)) {
            relativeUri = relativeUri.substring(FILE_PREFIX.length());
        } else if (relativeUri.startsWith(baseUrl)) {
            relativeUri = relativeUri.substring(baseUrl.length());
        }

        Matcher matcher = RESOURCE_PATH_PATTERN.matcher(relativeUri);
        relativeUri = matcher.replaceFirst("/");

        String decodedUri = relativeUri;
        try {
            decodedUri = URLDecoder.decode(relativeUri, "UTF8");
        } catch (UnsupportedEncodingException e) {
            log(Level.SEVERE, "Can't decode uri" + effectiveUri + ", error: " + e.getMessage());
        }

        InputStream resource = fetchResourceFromConfluence(relativeUri, decodedUri);

        if (resource != null) {
            return resource;
        } else if (effectiveUri.startsWith(DATA_PREFIX)) {
            return streamDataUrl(effectiveUri);
        }

        return super.resolveAndOpenStream(effectiveUri);
    }

    protected boolean shrinkImageCacheBeforeFetching() {
        return false;
    }

    protected abstract InputStream fetchResourceFromConfluence(String relativeUri, String decodedUri);

    private InputStream streamDataUrl(String dataUrl) {
        int dataIndex = dataUrl.indexOf(',');
        String data = dataUrl.substring(dataIndex + 1);
        byte[] bytes;
        if (dataUrl.substring(0, dataIndex).endsWith(";base64")) {
            bytes = Base64.decodeBase64(data);
        } else {
            try {
                bytes = URLCodec.decodeUrl(StringUtils.getBytesUsAscii(data));
            } catch (DecoderException e) {
                throw new IllegalArgumentException("Invalid data URL: \"" + dataUrl + "\".", e);
            }
        }
        return new ByteArrayInputStream(bytes);
    }

    protected void log(Level level, String message) {
    }
}