package com.atlassian.confluence.extra.flyingpdf.html;

import com.atlassian.confluence.extra.flyingpdf.util.UrlUtils;
import com.atlassian.renderer.util.UrlUtil;
import org.apache.commons.lang3.StringUtils;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.regex.Matcher;

import static com.atlassian.confluence.extra.flyingpdf.util.UrlUtils.encodeTitle;


/**
 * When you do a space export, you probably have links on the pages that references another pages.
 * This is where the fun starts. If referenced pages are included in space export, you want the links to become bookmarks.
 * Otherwise, you need to keep them as ordinary web links.
 * <p>
 * Here you have 2 options. If space export was done by legacy code (one huge space HTML translated to PDF in memory),
 * you have a luxury of using HTML anchors. Flyingsaucer library understands them and converts to PDF bookmarks automatically.
 * <p>
 * On the other hand, if you use sandbox and do conversion page-by-page, you can't use anchors. In this situation there's
 * an additional post processing steps that converts /display/FOO/bar to bookmark. And what you need to do in this case here
 * is to convert /pages/viewpage.action to /display/FOO/bar links to pick them later by postprocessing
 * {@link com.atlassian.confluence.extra.flyingpdf.util.PdfPostProcessor}
 * <p>
 * That's why we have {@link InternalPageStrategy} -- to distinguish these two cases.
 *
 * @author Ryan
 */

public class LinkFixer {

    /**
     * All pages to be exported
     */
    private final HashMap<String, String> pageTitleById = new HashMap<>();

    /**
     * Set of all page titles
     */
    private final HashSet<String> pageTitles = new HashSet<>();

    private final String baseUrl;

    /**
     * Export always runs in single space, so we can know space key
     */
    private final String spaceKey;

    /**
     * How we process links to another pages
     */
    private final InternalPageStrategy pageStrategy;

    public LinkFixer(String spaceKey, String baseUrl, InternalPageStrategy pageStrategy) {
        this.baseUrl = trimEndingSlash(baseUrl);
        this.spaceKey = spaceKey;
        this.pageStrategy = pageStrategy;
    }

    void addPage(String pageId, String pageTitle) {
        pageTitleById.put(pageId, pageTitle);
        pageTitles.add(pageTitle);
    }

    /**
     * How web link should be converted
     */
    String convertLink(String url) {
        if (StringUtils.isBlank(url))
            return null;

        final boolean isOnThisServer = url.trim().startsWith("/") || url.trim().startsWith(baseUrl);

        if (!isOnThisServer) {
            // We don't process links to another resources
            return null;
        }

        // Is this a view page url
        final Matcher matcher = UrlUtils.pageDisplayUrlPattern.matcher(url);
        if (matcher.find()) {
            final String spaceKey = matcher.group(1);
            final String pageTitle = UrlUtils.decodeTitle(matcher.group(2));

            if (this.spaceKey.equalsIgnoreCase(spaceKey) && pageTitles.contains(pageTitle)) {
                return pageStrategy.generate(baseUrl, spaceKey, pageTitle);
            }
        } else {
            if (url.contains("/pages/viewpage.action?")) {
                final Map queryParams = UrlUtil.getQueryParameters(url);
                final String spaceKey = (String) queryParams.get("spaceKey");
                final String pageId = (String) queryParams.get("pageId");

                String title = UrlUtils.decodeTitle((String) queryParams.get("pageTitle"));
                if (pageId != null) {
                    title = pageTitleById.get(pageId);
                    if (title != null) {
                        return pageStrategy.generate(baseUrl, this.spaceKey, title);
                    }
                } else if (this.spaceKey.equalsIgnoreCase(spaceKey) &&
                        pageTitles.contains(title)) {
                    return pageStrategy.generate(baseUrl, spaceKey, title);
                }
            }
        }

        // Keep the URL unchanged
        return null;
    }

    private static String trimEndingSlash(String baseUrl) {
        return baseUrl.endsWith("/") ? baseUrl.substring(0, baseUrl
                .length() - 1) : baseUrl;
    }

    public enum InternalPageStrategy {
        /**
         * Convert links to pages from the same export to anchors
         */
        ANCHOR {
            @Override
            String generate(String baseUrl, String spaceKey, String title) {
                return "#" + title;
            }
        },

        /**
         * Don't use anchors, just always normalise page links to /display/FOO/bar form
         */
        NORMALISE {
            @Override
            String generate(String baseUrl, String spaceKey, String title) {
                return baseUrl + "/display/" + spaceKey + "/" + encodeTitle(title);
            }
        };

        abstract String generate(String baseUrl, String spaceKey, String title);
    }

}
