package com.atlassian.confluence.extra.flyingpdf.analytic;

import java.util.Arrays;
import java.util.Collection;

/**
 * Holder of timing values which can calculate various statistics
 */
class TimeStatistics {

    private final int minPageTime;

    private final int maxPageTime;

    private final int meanPageTime;

    private final int p50PageTime;

    private final int p95PageTime;

    private final int p98PageTime;

    private final int p99PageTime;

    private final int p999PageTime;

    TimeStatistics(Collection<PageExportMetrics> pages) {
        if (pages.isEmpty()) {
            minPageTime = maxPageTime = meanPageTime = p50PageTime
                    = p95PageTime = p98PageTime = p99PageTime = p999PageTime = -1;
            return;
        }

        minPageTime = pages.stream()
                .mapToInt(PageExportMetrics::getTimeMs)
                .min()
                .orElse(-1);

        maxPageTime = pages.stream()
                .mapToInt(PageExportMetrics::getTimeMs)
                .max()
                .orElse(-1);

        final int totalTime = pages.stream()
                .mapToInt(PageExportMetrics::getTimeMs)
                .sum();
        meanPageTime = totalTime / pages.size();

        final int[] timings = pages.stream().mapToInt(PageExportMetrics::getTimeMs).toArray();
        final int[] quantiles = percentiles(timings, .50, .95, .98, .99, .999);

        p50PageTime = quantiles[0];
        p95PageTime = quantiles[1];
        p98PageTime = quantiles[2];
        p99PageTime = quantiles[3];
        p999PageTime = quantiles[4];
    }

    int getMinPageTime() {
        return minPageTime;
    }

    int getMaxPageTime() {
        return maxPageTime;
    }

    int getMeanPageTime() {
        return meanPageTime;
    }

    int getP50PageTime() {
        return p50PageTime;
    }

    int getP95PageTime() {
        return p95PageTime;
    }

    int getP98PageTime() {
        return p98PageTime;
    }

    int getP99PageTime() {
        return p99PageTime;
    }

    int getP999PageTime() {
        return p999PageTime;
    }

    /**
     * This can be done more efficiently in O(N), but this ugly v1 plugin can't use late guava or commons-math
     */
    private static int[] percentiles(int[] numbers, double... percentiles) {
        Arrays.sort(numbers);
        int[] values = new int[percentiles.length];
        for (int i = 0; i < percentiles.length; i++) {
            int index =
                    Math.min(numbers.length - 1,
                            Math.max(0,
                                    (int) (percentiles[i] * numbers.length)));
            values[i] = numbers[index];
        }
        return values;
    }
}
