package com.atlassian.confluence.extra.flyingpdf.sandbox;

import com.atlassian.confluence.api.model.Depth;
import com.atlassian.confluence.api.model.Expansions;
import com.atlassian.confluence.api.model.content.ContentType;
import com.atlassian.confluence.api.model.pagination.SimplePageRequest;
import com.atlassian.confluence.api.service.content.SpaceService;
import com.atlassian.confluence.core.ApiRestEntityFactory;
import com.atlassian.confluence.extra.flyingpdf.PdfExportProgressMonitor;
import com.atlassian.confluence.extra.flyingpdf.PdfExporterService;
import com.atlassian.confluence.extra.flyingpdf.analytic.ExportStatus;
import com.atlassian.confluence.extra.flyingpdf.analytic.FailureLocation;
import com.atlassian.confluence.extra.flyingpdf.analytic.PageExportMetrics;
import com.atlassian.confluence.extra.flyingpdf.analytic.SpaceExportMetrics;
import com.atlassian.confluence.extra.flyingpdf.html.DecorationPolicy;
import com.atlassian.confluence.extra.flyingpdf.html.LinkRenderingDetails;
import com.atlassian.confluence.extra.flyingpdf.html.TocBuilder;
import com.atlassian.confluence.extra.flyingpdf.html.XhtmlBuilder;
import com.atlassian.confluence.extra.flyingpdf.impl.ExportPermissionChecker;
import com.atlassian.confluence.extra.flyingpdf.impl.SandboxProgressMonitor;
import com.atlassian.confluence.extra.flyingpdf.util.ErrorMessages;
import com.atlassian.confluence.extra.flyingpdf.util.ExportedSpaceStructure;
import com.atlassian.confluence.extra.flyingpdf.util.PdfNode;
import com.atlassian.confluence.extra.flyingpdf.util.RenderedPdfFile;
import com.atlassian.confluence.importexport.ImportExportException;
import com.atlassian.confluence.importexport.ImportExportManager;
import com.atlassian.confluence.importexport.impl.ExportFileNameGenerator;
import com.atlassian.confluence.pages.AbstractPage;
import com.atlassian.confluence.pages.BlogPost;
import com.atlassian.confluence.pages.ContentNode;
import com.atlassian.confluence.pages.ContentTree;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.setup.settings.SettingsManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.util.GeneralUtil;
import com.atlassian.confluence.util.i18n.I18NBeanFactory;
import com.atlassian.core.util.ProgressMeter;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.user.User;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.w3c.dom.Document;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import static com.atlassian.confluence.extra.flyingpdf.html.DecorationPolicy.headerAndFooter;
import static com.atlassian.confluence.extra.flyingpdf.html.DecorationPolicy.none;
import static com.atlassian.confluence.extra.flyingpdf.html.LinkFixer.InternalPageStrategy.NORMALISE;
import static com.atlassian.confluence.extra.flyingpdf.util.UrlUtils.getFullUrl;
@Component
public class SandboxPdfExporterService implements PdfExporterService {
    private static final Logger log = LoggerFactory.getLogger(SandboxPdfExporterService.class);

    private final I18NBeanFactory i18NBeanFactory;
    private final ApiRestEntityFactory apiRestEntityFactory;
    private final ErrorMessages errorMessages;
    private final SpaceService apiSpaceService;
    private final SettingsManager settingsManager;
    private final SandboxPdfJoiner sandboxPdfJoiner;
    private final XhtmlBuilder intermediateHtmlBuilder;
    private final ImportExportManager importExportManager;
    private final ExportPermissionChecker exportPermissionChecker;
    private final SandboxXmlToPdfConverter sandboxXmlToPdfConverter;
    private final ExportFileNameGenerator pdfExportFileNameGenerator;

    public SandboxPdfExporterService(@ComponentImport I18NBeanFactory i18NBeanFactory,
                                     @ComponentImport ApiRestEntityFactory apiRestEntityFactory,
                                     @ComponentImport SpaceService apiSpaceService,
                                     @ComponentImport SettingsManager settingsManager,
                                     @ComponentImport ImportExportManager importExportManager,
                                     ErrorMessages errorMessages,
                                     SandboxPdfJoiner sandboxPdfJoiner,
                                     XhtmlBuilder intermediateHtmlBuilder,
                                     ExportPermissionChecker exportPermissionChecker,
                                     SandboxXmlToPdfConverter sandboxXmlToPdfConverter,
                                     ExportFileNameGenerator pdfExportFileNameGenerator) {
        this.i18NBeanFactory = i18NBeanFactory;
        this.apiRestEntityFactory = apiRestEntityFactory;
        this.errorMessages = errorMessages;
        this.apiSpaceService = apiSpaceService;
        this.settingsManager = settingsManager;
        this.sandboxPdfJoiner = sandboxPdfJoiner;
        this.intermediateHtmlBuilder = intermediateHtmlBuilder;
        this.importExportManager = importExportManager;
        this.exportPermissionChecker = exportPermissionChecker;
        this.sandboxXmlToPdfConverter = sandboxXmlToPdfConverter;
        this.pdfExportFileNameGenerator = pdfExportFileNameGenerator;
    }

    @Override
    public File createPdfForSpace(User user, Space space, ContentTree contentTree,
                                  PdfExportProgressMonitor progress, String contextPath,
                                  SpaceExportMetrics spaceExportMetrics, DecorationPolicy decorations) throws ImportExportException {
        exportPermissionChecker.checkAuthorization(user, space);

        if (contentTree.size() == 1) {
            // Shortcut for space exports consisting of only one page.
            // In this case we want to achieve behavior similar to single page export
            final Page page = Iterables.getOnlyElement(contentTree.getPages());
            String tmpFilePrefix = getTmpFilePrefix(page);
            final File outputFile = createPdf(
                    page,
                    contextPath,
                    LinkRenderingDetails.anchors(),
                    tmpFilePrefix,
                    new PageExportMetrics(),
                    DecorationPolicy.space().combine(decorations)).getFile();
            progress.completed(outputFile.getAbsolutePath());
            return outputFile;
        }

        progress.completedCalculationOfPdfPages(contentTree.size());
        progress.beginHtmlToPdfConversion();

        final List<PdfNode> pagesForest = new ArrayList<>();
        if (contentTree.getPages() != null) {
            for (ContentNode topLevel : contentTree.getRootNodes()) {
                final PageExportMetrics rootPageInfo = new PageExportMetrics();
                spaceExportMetrics.getPageExportMetrics().add(rootPageInfo);
                pagesForest.add(createPdfTreeNode(topLevel, contextPath, progress, contentTree.getPages(),
                        spaceExportMetrics, rootPageInfo));
            }
        }

        final ExportedSpaceStructure pdfStructure;
        final PageExportMetrics initialTocInfo = new PageExportMetrics();
        final PageExportMetrics finalTocInfo = new PageExportMetrics();
        final long tocStartTime = System.currentTimeMillis();
        try {
            final PdfNode initialToc = generateTableOfContents(contextPath, space, pagesForest, Collections.emptyMap(), initialTocInfo);
            pdfStructure = new ExportedSpaceStructure(initialToc, pagesForest);
            final PdfNode finalToc = generateTableOfContents(contextPath, space, pagesForest, pdfStructure.locationByTitleMap(), finalTocInfo);
            pdfStructure.replaceToc(finalToc);
            // We don't need initial TOC anymore, so we can delete it here.
            removeParentDirectory(initialToc.getRenderedPdfFile().getFile());
        } catch (Exception e) {
            for (PageExportMetrics tocInfo : ImmutableList.of(initialTocInfo, finalTocInfo)) {
                if (ExportStatus.isFail(tocInfo.getExportResults().getExportStatus())) {
                    spaceExportMetrics.getExportResults().setExportStatus(tocInfo.getExportResults().getExportStatus());
                    spaceExportMetrics.getExportResults().setFailureLocation(FailureLocation.TOC);
                }
            }
            throw e;
        } finally {
            spaceExportMetrics.setTocBuildTime((int) (System.currentTimeMillis() - tocStartTime));
        }

        String exportFileName = getSpaceExportFileName(space.getKey());

        final long joinStartTime = System.currentTimeMillis();
        try {
            final SandboxPdfJoinResponse result = sandboxPdfJoiner.join(spaceExportMetrics, space.getKey(), pdfStructure, exportFileName,
                    settingsManager.getGlobalSettings().getBaseUrl(), decorations);
            spaceExportMetrics.getExportResults().setPdfFileSizeBytes(result.getPdf().length());
            spaceExportMetrics.getExportResults().setPdfPagesTotal(result.getNumberOfPages());
        } finally {
            spaceExportMetrics.setJoinTime((int) (System.currentTimeMillis() - joinStartTime));
            cleanAllFiles(pdfStructure);
        }

        return new File(exportFileName);
    }

    private void cleanAllFiles(ExportedSpaceStructure structure) {
        removeParentDirectory(structure.getTableOfContents().getRenderedPdfFile().getFile());
        for (PdfNode root : structure.getConfluencePages()) {
            cleanupSubtree(root);
        }
    }

    private void cleanupSubtree(PdfNode root) {
        removeParentDirectory(root.getRenderedPdfFile().getFile());
        for (PdfNode child : root.getChildren()) {
            cleanupSubtree(child);
        }
    }

    private void removeParentDirectory(File file) {
        FileUtils.deleteQuietly(file.getParentFile());
    }

    private PdfNode generateTableOfContents(String contextPath, Space space, List<PdfNode> pages,
                                            Map<String, Integer> locations, PageExportMetrics pageExportMetrics) throws ImportExportException {
        final Document xhtml = renderToc(contextPath, space, pages, locations);
        final String tocString = i18NBeanFactory.getI18NBean().getText("com.atlassian.confluence.extra.flyingpdf.toc");
        return new PdfNode(tocString, renderPdf(contextPath, tocString, xhtml, "toc", pageExportMetrics));
    }

    private Document renderToc(String contextPath, Space space, List<PdfNode> pages, Map<String, Integer> locations) throws ImportExportException {
        final TocBuilder tocBuilder = new TocBuilder(contextPath, space.getKey());
        pages.forEach(p -> populateTocBuilder(tocBuilder, p, 0, locations));

        final String baseUrl = getFullUrl(settingsManager.getGlobalSettings().getBaseUrl(), contextPath);
        return intermediateHtmlBuilder.generateTableOfContents(baseUrl, space, tocBuilder);
    }

    private void populateTocBuilder(TocBuilder tocBuilder, PdfNode node, int level, Map<String, Integer> locations) {
        final String title = node.getPageTitle();
        if (locations.containsKey(title)) {
            tocBuilder.addEntry(level, title, locations.get(node.getPageTitle()));
        } else {
            tocBuilder.addEntry(level, title);
        }

        node.getChildren().forEach(n -> populateTocBuilder(tocBuilder, n, level + 1, locations));
    }

    @Override
    public File createPdfForPage(User user, AbstractPage page, String contextPath, PageExportMetrics pageExportMetrics) throws ImportExportException {
        exportPermissionChecker.checkAuthorization(user, page);
        String tmpFilePrefix = getTmpFilePrefix(page);
        return createPdf(page, contextPath, LinkRenderingDetails.anchors(), tmpFilePrefix, pageExportMetrics, none()).getFile();
    }

    public ContentTree getContentTree(User user, Space space) {
        return importExportManager.getContentTree(user, space);
    }

    public boolean isPermitted(User user, AbstractPage page) {
        return exportPermissionChecker.isPermitted(user, page);
    }

    public boolean isPermitted(User user, Space space) {
        return exportPermissionChecker.isPermitted(user, space);
    }

    @Override
    public boolean exportableContentExists(Space space) {
        return space != null && apiSpaceService
                .findContent(apiRestEntityFactory.buildRestEntityFrom(space, Expansions.EMPTY).getDelegate())
                .withDepth(Depth.ROOT)
                .fetchMany(ContentType.PAGE, new SimplePageRequest(0, 10))
                .size() > 0;
    }

    @Override
    public PdfExportProgressMonitor createProgressMonitor(ProgressMeter progressMeter) {
        return new SandboxProgressMonitor(i18NBeanFactory.getI18NBean(), errorMessages, progressMeter);
    }

    private PdfNode createPdfTreeNode(ContentNode contentNode,
                                      String contextPath,
                                      PdfExportProgressMonitor progress,
                                      Collection<Page> internalPages,
                                      SpaceExportMetrics spaceExportMetrics,
                                      PageExportMetrics pageExportMetrics) throws ImportExportException {
        spaceExportMetrics.getPageExportMetrics().add(pageExportMetrics);

        final Page page = contentNode.getPage();
        progress.performingHtmlToPdfConversionForPage(page.getTitle());
        String tmpFilePrefix = getTmpFilePrefix(page);
        final RenderedPdfFile file = createPdf(page, contextPath,
                new LinkRenderingDetails(internalPages, NORMALISE), tmpFilePrefix, pageExportMetrics, headerAndFooter());

        PdfNode node = new PdfNode(page.getTitle(), file);
        for (ContentNode child : contentNode.getChildren()) {
            final PageExportMetrics childInfo = new PageExportMetrics();
            node.addChild(createPdfTreeNode(child, contextPath, progress, internalPages, spaceExportMetrics, childInfo));
        }

        return node;
    }

    private String getTmpFilePrefix(AbstractPage page) {
        String tmpFilePrefix = page.getTitle();
        if (!GeneralUtil.isSafeTitleForFilesystem(tmpFilePrefix)) {
            tmpFilePrefix = page.getIdAsString();
        }
        return tmpFilePrefix;
    }

    private String getSpaceExportFileName(String spaceKey) {
        String differentiators = spaceKey.startsWith("~") ? spaceKey.substring(1) : spaceKey;
        try {
            return pdfExportFileNameGenerator.getExportFile(differentiators).getAbsolutePath();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private Document createDocument(Page page, LinkRenderingDetails linkDetails, DecorationPolicy decoration) {
        Document xhtml;
        ContentTree tree = new ContentTree();
        tree.addRootNode(new ContentNode(page));
        try {
            xhtml = intermediateHtmlBuilder.buildHtml(tree, page.getSpace(), linkDetails,
                    decoration);
        } catch (ImportExportException e) {

            log.error("error build xml dom", e);

            throw new RuntimeException(e);
        }
        return xhtml;
    }

    private Document createDocument(BlogPost blogPost) {
        Document xhtml;
        try {
            xhtml = intermediateHtmlBuilder.buildHtml(blogPost);
        } catch (ImportExportException e) {
            throw new RuntimeException(e);
        }
        return xhtml;
    }

    private RenderedPdfFile createPdf(AbstractPage page, String contextPath, LinkRenderingDetails linkDetails,
                                      String filePrefix, PageExportMetrics pageExportMetrics, DecorationPolicy decoration) throws ImportExportException {
        log.debug("Exporting page {}", page.getId());
        final Document xhtml;
        if (page instanceof Page) {
            xhtml = createDocument((Page) page, linkDetails, decoration);
        } else {
            xhtml = createDocument((BlogPost) page);
        }

        final long startTime = System.currentTimeMillis();
        try {
            pageExportMetrics.setPageId(page.getId());
            pageExportMetrics.setPageRevision(page.getConfluenceRevision().hashCode());

            final RenderedPdfFile result = renderPdf(contextPath, page.getTitle(), xhtml, filePrefix, pageExportMetrics);

            pageExportMetrics.getExportResults().setExportStatus(ExportStatus.OK);
            pageExportMetrics.getExportResults().setPdfPagesTotal(result.getNumPages());
            pageExportMetrics.getExportResults().setPdfFileSizeBytes(result.getFile().length());

            return result;
        } finally {
            pageExportMetrics.setTimeMs((int) (System.currentTimeMillis() - startTime));
        }
    }

    private RenderedPdfFile renderPdf(String contextPath, String pageTitle, Document xhtml, String filePrefix, PageExportMetrics pageExportMetrics) throws ImportExportException {
        final String filename = uniqueFilename(filePrefix);
        return sandboxXmlToPdfConverter.convertXhtmlToPdf(filename, pageTitle, xhtml, contextPath, getUsername(), pageExportMetrics);
    }

    private static String uniqueFilename(String filePrefix) {
        final String trimmedPrefix = filePrefix.substring(0, Math.min(filePrefix.length(), 32));
        return trimmedPrefix + "_" + UUID.randomUUID().toString().replaceAll("-", "");
    }

    private static String getUsername() {
        return AuthenticatedUserThreadLocal.getUsername();
    }

}
