package com.atlassian.confluence.extra.flyingpdf.impl;

import com.atlassian.confluence.extra.flyingpdf.util.ImageFileCacheUtils;
import com.atlassian.confluence.extra.flyingpdf.util.ImageInformation;
import com.atlassian.confluence.extra.flyingpdf.util.ImageInformationURICacheUtil;
import com.atlassian.confluence.extra.flyingpdf.util.ImageTranscoderCacheUtil;
import com.lowagie.text.Image;
import org.apache.batik.transcoder.TranscoderException;
import org.apache.batik.transcoder.TranscoderInput;
import org.apache.batik.transcoder.TranscoderOutput;
import org.apache.batik.transcoder.image.ImageTranscoder;
import org.apache.batik.transcoder.image.PNGTranscoder;
import org.apache.commons.io.IOUtils;
import org.xhtmlrenderer.pdf.ITextFSImage;

import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Iterator;
import java.util.function.Supplier;

/**
 * Implementation of ITextFSImage that does not keep the image in memory.
 */
public class LightITextFSImage extends ITextFSImage {
    private static final int DEFAULT_SVG_HEIGHT = 32;
    private static final int DEFAULT_SVG_WIDTH = 32;
    private static final String FILE_PREFIX = "file:";
    private final Supplier<InputStream> imgStreamSupplier;
    private int width;
    private int height;
    private final String uri;
    private final String baseUrl;

    /**
     * Creates a new LightITextFSImage.
     *
     * @param imgStreamSupplier supplier that provides InputStream to the underlying image file
     * @param dotsPerPixel      output resolution
     * @throws IOException if the image cannot be read
     */
    public LightITextFSImage(final Supplier<InputStream> imgStreamSupplier, final float dotsPerPixel,
                             final String baseUrl, final String uri) throws IOException {
        super(null);
        this.imgStreamSupplier = imgStreamSupplier;
        this.baseUrl = baseUrl;
        this.uri = uri;
        updateMeasurements(dotsPerPixel);
    }

    private void updateMeasurements(final float dotsPerPixel) throws IOException {
        // Check if uri is already cached, it means SVG
        ImageInformation imageInfo = ImageInformationURICacheUtil.getCacheURI(uri);
        if (imageInfo != null && imageInfo.getTempFileName() == null) {
            throw new IOException("Unknown image format");
        } else if (imageInfo != null) {
            // The first image with the same url will have been cached, then just set default width and height
            width = imageInfo.getWidth();
            height = imageInfo.getHeight();
        } else {
            File tempFile = createTempFileFromInputStream();

            ImageInformation cacheInformation;
            try (ImageInputStream in = ImageIO.createImageInputStream(new FileInputStream(tempFile))) {
                final Iterator readers = ImageIO.getImageReaders(in);
                if (readers.hasNext()) {
                    ImageReader reader = (ImageReader) readers.next();
                    try {
                        reader.setInput(in);
                        width = (int) (dotsPerPixel * reader.getWidth(0));
                        height = (int) (dotsPerPixel * reader.getHeight(0));
                        cacheInformation = new ImageInformation(height, width, tempFile.getAbsolutePath(), false);
                    } finally {
                        reader.dispose();
                    }
                } else {
                    // It could be SVG format and it is web url images, if not then exception will be thrown
                    try(InputStream imgSvgStream = new FileInputStream(tempFile)) {
                        PNGTranscoder transcoder = new PNGTranscoder();
                        ByteArrayOutputStream out = new ByteArrayOutputStream();
                        transcoder.transcode(new TranscoderInput(new InputStreamReader(imgSvgStream,
                                "UTF-8")), new TranscoderOutput(out));
                        // If there is no exception, then set default with and height for transcoder image
                        width = (int) (dotsPerPixel * DEFAULT_SVG_WIDTH);
                        height = (int) (dotsPerPixel * DEFAULT_SVG_HEIGHT);
                        cacheInformation = new ImageInformation(height, width, tempFile.getAbsolutePath(), true);
                    } catch (TranscoderException te) {
                        // In case, the uri is not svg file
                        cacheInformation = new ImageInformation(0, 0, null, false);
                        ImageInformationURICacheUtil.setCacheURI(uri, cacheInformation);
                        throw new IOException("Unknown image format", te);
                    } catch (MalformedURLException te) {
                        cacheInformation = new ImageInformation(0, 0, null, false);
                        ImageInformationURICacheUtil.setCacheURI(uri, cacheInformation);
                        throw new IOException("Malformed url " + uri, te);
                    }
                }
                ImageInformationURICacheUtil.setCacheURI(uri, cacheInformation);
            }
        }
    }

    private File createTempFileFromInputStream() throws IOException {
        InputStream imgStream = imgStreamSupplier.get();
        try {
            return ImageFileCacheUtils.createTempFile(imgStream);
        } finally {
            IOUtils.closeQuietly(imgStream);
        }
    }

    @Override
    public int getWidth() {
        return width;
    }

    @Override
    public int getHeight() {
        return height;
    }

    @Override
    public void scale(int width, int height) {
        int targetWidth = width;
        int targetHeight = height;

        if (targetWidth == -1) {
            targetWidth = (int) (getWidth() * ((double) targetHeight / getHeight()));
        }

        if (targetHeight == -1) {
            targetHeight = (int) (getHeight() * ((double) targetWidth / getWidth()));
        }

        this.width = targetWidth != 0? targetWidth : getWidth();
        this.height = targetHeight != 0? targetHeight : getHeight();
    }

    /**
     * Returns a new {@link Image} that is scaled according to current width and height.
     *
     * @return a new {@link Image} that is scaled according to current width and height
     */
    @Override
    public Image getImage() {
        String imageTempFile = ImageTranscoderCacheUtil.getCacheImage(uri);
        if (imageTempFile == null) {
            ImageInformation imageInformation = ImageInformationURICacheUtil.getCacheURI(uri);
            // There are 3 cases: an SVG image, An normal image, and a dummy image
            if (imageInformation.isSVGImage()) {
                try(InputStream imgSvgStream = new FileInputStream(imageInformation.getTempFileName())) {
                    PNGTranscoder transcoder = new PNGTranscoder();
                    // Set final png export image with and height, if img tags do not specifly the width and height then using default value above
                    transcoder.addTranscodingHint(ImageTranscoder.KEY_WIDTH, (float) width);
                    transcoder.addTranscodingHint(ImageTranscoder.KEY_HEIGHT, (float) height);
                    ByteArrayOutputStream out = new ByteArrayOutputStream();
                    transcoder.transcode(new TranscoderInput(new InputStreamReader(imgSvgStream,"UTF-8")), new TranscoderOutput(out));
                    //create a temp file to store transcoder result and put file name into the cache
                    File transcoderTempFile = ImageFileCacheUtils.createTempFile(out.toByteArray());
                    ImageTranscoderCacheUtil.setCacheImage(uri, transcoderTempFile.getAbsolutePath());
                    // reset imageTempFile to new path
                    imageTempFile = transcoderTempFile.getAbsolutePath();
                } catch (Exception tx) {
                    throw new RuntimeException("Failed to read image", tx);
                }
            } else {
                imageTempFile = imageInformation.getTempFileName();
            }
        }

        InputStream imageInputStream;
        // SVG file or normal image but not the dummy one
        try {
            if (imageTempFile != null) {
                imageInputStream = new FileInputStream(imageTempFile);
            } else if (!isExternalResource()) {
                imageInputStream = imgStreamSupplier.get();
            } else {
                throw new RuntimeException("Failed to read image");
            }


            try {
                BufferedImage jImage = ImageIO.read(imageInputStream);
                Image image = Image.getInstance(jImage, null);
                image.scaleAbsolute(width, height);
                return image;
            } catch (Exception e) {
                // This wil handle for internal svg files and it won't be cached
                try (InputStream imgSvgStream = new URL(uri).openStream()) {
                    PNGTranscoder transcoder = new PNGTranscoder();
                    // Set final png export image with and height, if img tags do not specify the width and height then using default value above
                    transcoder.addTranscodingHint(ImageTranscoder.KEY_WIDTH, (float) width);
                    transcoder.addTranscodingHint(ImageTranscoder.KEY_HEIGHT, (float) height);
                    ByteArrayOutputStream out = new ByteArrayOutputStream();
                    transcoder.transcode(new TranscoderInput(new InputStreamReader(imgSvgStream, "UTF-8")), new TranscoderOutput(out));
                    Image svgImage = Image.getInstance(out.toByteArray());
                    svgImage.scaleAbsolute(width, height);
                    return svgImage;
                } catch (Exception tx) {
                    throw new RuntimeException("Failed to read image", tx);
                }
            } finally {
                IOUtils.closeQuietly(imageInputStream);
            }

        } catch (Exception e) {
            throw new RuntimeException("Failed to read image", e);
        }
    }

    private boolean isExternalResource() {
        return uri != null && !uri.startsWith(FILE_PREFIX) && !uri.startsWith(baseUrl) && !uri.startsWith("/");
    }
}
