package com.atlassian.confluence.extra.flyingpdf.impl;

import com.atlassian.confluence.importexport.resource.DownloadResourceManager;
import com.atlassian.confluence.importexport.resource.DownloadResourceNotFoundException;
import com.atlassian.confluence.importexport.resource.DownloadResourceReader;
import com.atlassian.confluence.importexport.resource.UnauthorizedDownloadResourceException;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.renderer.util.UrlUtil;
import org.apache.log4j.Logger;
import org.xhtmlrenderer.pdf.ITextOutputDevice;

import java.io.InputStream;
import java.util.logging.Level;

public class ConfluenceExportUserAgent extends AbstractExportUserAgent {
    private static final Logger LOG = Logger.getLogger(ConfluenceExportUserAgent.class);

    private DownloadResourceManager resourceManager;

    public ConfluenceExportUserAgent(ITextOutputDevice device, String baseUrl, String cdnUrl, DownloadResourceManager resourceManager) {
        super(device, baseUrl, cdnUrl);

        this.resourceManager = resourceManager;
    }

    @Override
    protected InputStream fetchResourceFromConfluence(String relativeUri, String decodedUri) {
        if (resourceManager.matches(decodedUri)) {
            // so this isn't a request for a resource across the network
            // try getting it from the resource manager
            String userName = AuthenticatedUserThreadLocal.getUsername();
            String strippedUri = stripQueryString(decodedUri);

            DownloadResourceReader downloadResourceReader = getResourceReader(decodedUri, userName, strippedUri);

            // Some plugins (Gliffy) store the file as encoded so we have to re-check for the encoded version of the file
            if (downloadResourceReader == null) {
                strippedUri = stripQueryString(relativeUri);
                downloadResourceReader = getResourceReader(relativeUri, userName, strippedUri);
            }

            if (downloadResourceReader != null) {
                try {
                    return downloadResourceReader.getStreamForReading();
                } catch (Exception e) {
                    log(Level.SEVERE, "Couldn't retrieve image resource " + decodedUri + " during Confluence export");
                }
            }
        }

        return null;
    }

    @Override
    protected void log(Level level, String message) {
        if (level.equals(Level.SEVERE)) {
            LOG.error(message);
        } else {
            LOG.debug(message);
        }
    }

    private DownloadResourceReader getResourceReader(String uri,
                                                     String userName,
                                                     String strippedUri) {
        DownloadResourceReader downloadResourceReader = null;
        try {
            downloadResourceReader = resourceManager.getResourceReader(userName, strippedUri, UrlUtil.getQueryParameters(uri));
        } catch (UnauthorizedDownloadResourceException ex) {
            log(Level.WARNING, "Not authorized to download resource " + uri + ", error: " + ex.getMessage());
        } catch (DownloadResourceNotFoundException ex) {
            log(Level.WARNING, "No resource found for resource " + uri + ", error: " + ex.getMessage());
        }
        return downloadResourceReader;
    }

    private String stripQueryString(String uri) {
        int queryIndex = uri.indexOf('?');
        if (queryIndex > 0)
            uri = uri.substring(0, queryIndex);
        return uri;
    }
}