package com.atlassian.cache.hazelcast;

import java.util.SortedMap;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.CacheSettingsBuilder;
import com.atlassian.cache.CacheStatisticsKey;
import com.atlassian.cache.ManagedCache;
import java.util.function.Supplier;

import com.google.common.collect.ImmutableSortedMap;
import com.hazelcast.config.MapConfig;
import com.hazelcast.map.IMap;

import static java.util.concurrent.TimeUnit.SECONDS;

/**
 * Common implementation of {@link com.atlassian.cache.ManagedCache} methods for caches and cached references that are
 * backed by {@link IMap}s.
 */
@ParametersAreNonnullByDefault
public abstract class ManagedCacheSupport implements ManagedCache
{
    private final String name;
    protected final HazelcastCacheManager cacheManager;

    public ManagedCacheSupport(String name, HazelcastCacheManager cacheManager)
    {
        this.cacheManager = cacheManager;
        this.name = name;
    }

    @Nullable
    @Override
    public Long currentExpireAfterAccessMillis()
    {
        long maxIdle = SECONDS.toMillis(getConfig().getMaxIdleSeconds());
        return (maxIdle > 0 ? maxIdle : null);
    }

    @Nullable
    @Override
    public Long currentExpireAfterWriteMillis()
    {
        long timeToLive = SECONDS.toMillis(getConfig().getTimeToLiveSeconds());
        return (timeToLive > 0 ? timeToLive : null);
    }

    @Nullable
    @Override
    public Integer currentMaxEntries()
    {
        int maxSize = getConfig().getEvictionConfig().getSize();
        return maxSize > 0 ? maxSize : null;
    }

    @Override
    public boolean isReplicateViaCopy()
    {
        return true;
    }

    @Nonnull
    @Override
    public String getName()
    {
        return name;
    }

    @Override
    public boolean isFlushable()
    {
        final CacheSettings cacheSettings = getCacheSettings();
        return cacheSettings == null || cacheSettings.getFlushable(true);
    }

    @Override
    public boolean isLocal()
    {
        return false;
    }

    @Override
    public boolean isReplicateAsynchronously()
    {
        CacheSettings cacheSettings = getCacheSettings();
        return cacheSettings == null || cacheSettings.getReplicateAsynchronously(true);
    }

    @Override
    public boolean updateExpireAfterAccess(long expireAfter, TimeUnit timeUnit)
    {
        CacheSettings newCacheSettings = new CacheSettingsBuilder(getCacheSettings()).expireAfterAccess(expireAfter, timeUnit).build();
        return cacheManager.updateCacheSettings(getHazelcastMapName(), newCacheSettings);
    }

    @Override
    public boolean updateExpireAfterWrite(long expireAfter, TimeUnit timeUnit)
    {
        CacheSettings newCacheSettings = new CacheSettingsBuilder(getCacheSettings()).expireAfterWrite(expireAfter, timeUnit).build();
        return cacheManager.updateCacheSettings(getHazelcastMapName(), newCacheSettings);
    }

    @Override
    public boolean updateMaxEntries(int newValue)
    {
        CacheSettings newCacheSettings = new CacheSettingsBuilder(getCacheSettings()).maxEntries(newValue).build();
        return cacheManager.updateCacheSettings(getHazelcastMapName(), newCacheSettings);
    }

    @Nonnull
    @Override
    public SortedMap<CacheStatisticsKey, Supplier<Long>> getStatistics()
    {
        return ImmutableSortedMap.of();
    }

    @Override
    public boolean isStatisticsEnabled()
    {
        return false;
    }

    @Override
    public void setStatistics(boolean enabled) { throw new UnsupportedOperationException("setStatistics() not implemented"); }

    @Nonnull
    protected abstract String getHazelcastMapName();

    @Nullable
    private MapConfig getConfig()
    {
        return cacheManager.getMapConfig(getHazelcastMapName());
    }

    @Nullable
    private CacheSettings getCacheSettings()
    {
        return cacheManager.getCacheSettings(getHazelcastMapName());
    }
}
