package com.atlassian.cache;

import com.atlassian.annotations.PublicApi;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Interface for creating and/or attaching to caches. Caches are identified by name,
 * and two caches with the same name always refer to the same backing structure.
 * <p>
 * This factory and its builders hide many of the implementation details of any cache.
 * The only real control you have is whether it is a local (JVM-local) or shared cache.
 * All of the following maybe be different in some execution environments, so you should
 * not assume any of them in your code:
 * <ul>
 *   <li>Size and expiry settings (e.g. maxsize, expire times) may be overridden by admin or deployment settings.
 *   <li>When you builder.build() a cache for the first time in your code, the cache may already have items in it.
 *       For example, you are in a cluster. If your plugin is reloaded, or even if a node is restarted,
 *       shared caches will already have data in them.
 *   <li>For shared caches, when a new version of your code/plugin is deployed, it may already exist with data in it
 *       from a previous version of your code/plugin.
 *   <li>The keys and values in your non-local cache are effectively persistent from the point of view of
 *       your code/plugin. Therefore, you should consider backward compatibility of data, or invalidate
 *       caches on start/upgrade.
 * </ul>
 * The getCache methods called with a CacheLoader and getCachedReference methods will always return a new cache object that
 * only holds a weak reference to the CacheLoader or Supplier that has been passed in.  This is so we can support
 * plugin reloadability. Consumers of this style of cache should get the cache only once and retain the reference to the
 * cache during their complete lifecycle.
 *
 * The getCache methods called without a supplier will return a pre-existing cache of the same name if it already exists.
 * Plugin developers should be aware that this may contain data from a previous instance of their plugin (if it has been
 * reloaded) and should refresh the cache when they obtain it, if this is appropriate.
 *
 * @see Cache
 * @see CachedReference
 * @see CacheManager
 */
@PublicApi
public interface CacheFactory
{
    /**
     * Returns a Cached Reference, creating it if necessary.
     * @param name the name of the Cached Reference
     * @param supplier the supplier for value to be cached, called if the value needs to be generated
     * @return the Cached Reference
     */
    @Nonnull
    <V> CachedReference<V> getCachedReference(@Nonnull String name,
                                              @Nonnull Supplier<V> supplier);

    /**
     * Returns a Cached Reference, creating it if necessary.
     * @param name the name of the Cached Reference
     * @param supplier the supplier for value to be cached, called if the value needs to be generated
     * @param required specifies the required cache settings
     * @return the Cached Reference
     */
    @Nonnull
    <V> CachedReference<V> getCachedReference(@Nonnull String name,
                                              @Nonnull Supplier<V> supplier,
                                              @Nonnull CacheSettings required);

    /**
     * Returns a Cached Reference, creating it if necessary.
     * <p>
     * It is equivalent to calling {@link #getCachedReference(String, Supplier, CacheSettings)}
     *
     * @param owningClass the owning class
     * @param name the name of the cache spaced within the <tt>owningClass</tt>
     * @param supplier the supplier for value to be cached, called if the value needs to be generated
     * @return the Cached Reference
     */
    @Nonnull
    <V> CachedReference<V> getCachedReference(@Nonnull Class<?> owningClass,
                                              @Nonnull String name,
                                              @Nonnull Supplier<V> supplier);

    /**
     * Returns a Cached Reference, creating it if necessary.
     * <p>
     * It is equivalent to calling {@link #getCachedReference(String, Supplier, CacheSettings)}
     *
     * @param owningClass the owning class
     * @param name the name of the cache spaced within the <tt>owningClass</tt>
     * @param supplier the supplier for value to be cached, called if the value needs to be generated
     * @param required specifies the required cache settings
     * @return the Cached Reference
     */
    @Nonnull
    <V> CachedReference<V> getCachedReference(@Nonnull Class<?> owningClass,
                                              @Nonnull String name,
                                              @Nonnull Supplier<V> supplier,
                                              @Nonnull CacheSettings required);

    /**
     * Returns the cache with the given name, creates it if necessary.
     * This is equivalent to calling {@link #getCache(String, CacheLoader)}
     * with <tt>name</tt> and <tt>null</tt>.
     *
     * @param name the name of the cache
     * @return a Cache
     */
    @Nonnull
    <K, V> Cache<K, V> getCache(@Nonnull String name);

    /**
     * Returns the cache with the given name, creates it if necessary.
     * <p>
     * It is equivalent to calling {@link #getCache(String)} with the computed name.
     *
     * @param owningClass the owning class
     * @param name the name of the cache spaced within the <tt>owningClass</tt>
     * @return a Cache
     */
    @Nonnull
    <K, V> Cache<K, V> getCache(@Nonnull Class<?> owningClass, @Nonnull String name);

    /**
     * Returns the cache with the given name and loader, creates it if necessary.
     * This is equivalent to calling {@link #getCache(String, CacheLoader, CacheSettings)}
     * with <tt>name</tt>, <tt>null</tt> and <tt>new CacheSettingsBuilder().build()</tt>.
     *
     * @param name the name of the cache
     * @param loader the loader that will be used to provide values for keys that will be added to the cache. <tt>null</tt> indicates no loader
     * @return a Cache
     */
    @Nonnull
    <K, V> Cache<K, V> getCache(@Nonnull String name, @Nullable CacheLoader<K, V> loader);

    /**
     * Returns the cache with the given name, loader and required cache settings, creates it if necessary.
     *
     * @param name the name of the cache
     * @param loader the loader that will be used to provide values for keys that will be added to the cache. <tt>null</tt> indicates no loader
     * @param required the cache settings that are required to be set if the cache is created.
     * @return a Cache
     */
    @Nonnull
    <K, V> Cache<K, V> getCache(@Nonnull String name, @Nullable CacheLoader<K, V> loader, @Nonnull CacheSettings required);

    /**
     * Returns the {@link ReadThroughCache} with the given name, creates it if necessary.
     *
     * @param name the name of the cache
     * @return a {@link ReadThroughCache}
     * @since 6.1
     */
    default <K, V> ReadThroughCache<K, V> getReadThroughCache(@Nonnull String name)
    {
        return getCache(name);
    }

    /**
     * Returns the {@link ReadThroughCache} with the given name, creates it if necessary.
     *
     * @param name the name of the cache
     * @return a {@link ReadThroughCache}
     * @since 6.1
     */
    default <K, V> ReadThroughCache<K, V> getReadThroughCache(@Nonnull String name, @Nonnull CacheSettings required)
    {
        return getCache(name, null, required);
    }

    /**
     * Returns a cache with the given name, and the types specified.  Creates it if necessary.  If two caches of the
     * same name are queried, with different types, the call with incorrect type arguments will throw a
     * ClassCastException. For example with:<p>
     * Cache&lt;String,String&gt; firstCache = cacheManager.getCache("myCache", String.class, String.class); <br>
     * Cache&lt;String,Long&gt; secondCache = cacheManager.getCache("myCache", String.class, Long.class);
     * <p>
     * the second call to getCache will result in a ClassCastException.
     *
     * @param name the name of the cache
     * @param keyType the type of keys in the cache.  Must extend Serializable
     * @param valueType the type of values in the cache. Must extend Serializable
     * @return a Cache
     * @deprecated since 2.0, use getCache(name) instead
     */
    @Deprecated
    @Nonnull
    <K, V> Cache<K, V> getCache(@Nonnull String name, @Nonnull Class<K> keyType, @Nonnull Class<V> valueType);
}
