package com.atlassian.buildeng.hallelujah;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientHandlerException;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.WebResource;
import org.apache.log4j.Logger;

public class DefaultClientTestCaseProvider implements ClientTestCaseProvider
{
    private static final long SERVER_DISAPPEARED_MILLIS   = 60 * 1000;  /* 60 seconds */
    private static final long DEFAULT_SERVER_RETRY_MILLIS = 5 * 1000;   /* 1 seconds */

    private static final int HTTP_OK = 200;

    private static final Logger log = Logger.getLogger(DefaultClientTestCaseProvider.class);

    private long lastSendMillis;
    private long lastReceiveMillis;

    private boolean initialised;
    private final String serverBaseUri; // "http://example.com/base"
    private WebResource webResource;

    private boolean lastServerConnectionWasSucessful = false;
    private long serverRetryMillis = DEFAULT_SERVER_RETRY_MILLIS;

    public DefaultClientTestCaseProvider(String serverBaseUri)
    {
        this.serverBaseUri = serverBaseUri;
        initialised = false;
    }

    private void ensureInitialised ()
    {
        if (!initialised)
        {
            lastSendMillis = System.currentTimeMillis();
            lastReceiveMillis = System.currentTimeMillis();

            Client client = Client.create();
            webResource = client.resource(serverBaseUri);

            initialised = true;
        }
    }

    /**
     * This method will block
     *
     * @return
     */
    @Override
    public String getNextTestName()
    {
        ensureInitialised();

        String nextTestName = null;

        boolean waitingForTest = true;
        while (waitingForTest)
        {
            long nowMillis = System.currentTimeMillis();

            if (Math.abs(nowMillis - lastReceiveMillis) > SERVER_DISAPPEARED_MILLIS)
            {
                log.info("Haven't received a message from the server in a long time...");
                waitingForTest = false;
            }
            else if (Math.abs(nowMillis - lastSendMillis) > serverRetryMillis)
            {
                log.info("Trying to receive a message from the server");

                lastServerConnectionWasSucessful = false;

                ClientResponse response = null;

                try
                {
                    response = webResource.path("/test/next").accept("text/plain").get(ClientResponse.class);
                }
                catch (ClientHandlerException e)
                {
                    log.info("Error connecting to server: " + e);
                    /* swallow */
                }

                if (response != null)
                {
                    int responseStatus = response.getStatus();
                    if (responseStatus == HTTP_OK)
                    {
                        lastServerConnectionWasSucessful = true;
                        String textEntity = response.getEntity(String.class);
                        nextTestName = textEntity;
                        log.info("Next test name: " + nextTestName);
                        lastReceiveMillis = System.currentTimeMillis();
                        waitingForTest = false;
                    }
                    else
                    {
                        log.info("Client test case provider got a server response but had status: " + responseStatus);
                    }
                }

                lastSendMillis = System.currentTimeMillis();

                updateServerRetryMillis();
            }
            else
            {
                log.info("Client test case provider sleeping for " + serverRetryMillis + " milliseconds");
                sleep(serverRetryMillis);
            }
        }

        return nextTestName;
    }

    private void sleep(long millis)
    {
        try
        {
            Thread.sleep(millis);
        }
        catch (InterruptedException e)
        {
            /* swallow */
        }
    }

    private long updateServerRetryMillis ()
    {
        if (lastServerConnectionWasSucessful)
        {
            // hit the server faster and faster as long as it is responding
            serverRetryMillis = serverRetryMillis / 2;
        }
        else
        {
            serverRetryMillis = DEFAULT_SERVER_RETRY_MILLIS;
        }

        return serverRetryMillis;
    }
}
