package com.atlassian.buildeng.hallelujah.jms;

import com.atlassian.buildeng.hallelujah.api.HallelujahSystemProperties;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.log4j.Logger;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import static com.atlassian.buildeng.hallelujah.api.HallelujahSystemProperties.BROKER_URL_KEY;
import static com.atlassian.buildeng.hallelujah.api.HallelujahSystemProperties.CLIENT_KEY;
import static com.atlassian.buildeng.hallelujah.api.HallelujahSystemProperties.DEFAULT_BROKER_URL;
import static com.atlassian.buildeng.hallelujah.api.HallelujahSystemProperties.DEFAULT_CLIENT_KEY_PREFIX;
import static com.atlassian.buildeng.hallelujah.api.HallelujahSystemProperties.DEFAULT_QUEUE_KEY;
import static com.atlassian.buildeng.hallelujah.api.HallelujahSystemProperties.QUEUE_KEY;
import static com.atlassian.buildeng.hallelujah.api.HallelujahSystemProperties.RANDOM_CLIENT_ID_LENGTH;

/**
 * Reads necessary configuration for JMS operation, in particular the broker URL and test name/result queues
 */
public class JMSConfiguration
{
    private static final Logger log = Logger.getLogger(JMSConfiguration.class);
    
    private static final String CONFIG_DIR = "/tmp";
    private static final String CONFIG_FILE = "jmsconfig";
    private static final int ID_LENGTH = 12;

    private AtomicBoolean preFetching = new AtomicBoolean(false);
    
    private final String brokerUrl;
    private final String testNameQueue;
    private final String testResultQueue;
    private final String id;
    private final String controlTopic;


    private JMSConfiguration(final String brokerUrl, final String queuePrefix, final String id)
    {
        this.brokerUrl = brokerUrl;
        this.testNameQueue = queuePrefix + "-test-name";
        this.testResultQueue = queuePrefix + "-test-result";
        this.controlTopic = queuePrefix + "-control";
        this.id = id;
    }

    private JMSConfiguration(final String brokerUrl, final String testNameQueue, final String testResultQueue, final String id)
    {
        this.brokerUrl = brokerUrl;
        this.testNameQueue = testNameQueue;
        this.testResultQueue = testResultQueue;
        this.controlTopic = testNameQueue + "-control";
        this.id = id;
    }

    private JMSConfiguration(final File jmsConfig)
    {
        final List<String> config;
        try
        {
            config = FileUtils.readLines(jmsConfig);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
        brokerUrl = config.get(0);
        testNameQueue = config.get(1);
        testResultQueue = config.get(2);
        if (config.size() > 3)
        {
            id = config.get(3);
        }
        else
        {
            id = RandomStringUtils.randomAlphanumeric(ID_LENGTH);
        }
        controlTopic = testNameQueue + "-control";
            
    }

    public static JMSConfiguration fromFile(final File jmsConfig)
    {
        return new JMSConfiguration(jmsConfig);
    }

    public static JMSConfiguration fromSystemPropertiesRespectingDefaultFile()
    {
        final JMSConfiguration jmsConfigurationFromFile = JMSConfiguration.fromDefaultFile();
        final JMSConfiguration jmsConfiguration;
        if (jmsConfigurationFromFile == null)
        {
            jmsConfiguration = JMSConfiguration.fromSystemProperties();
        }
        else
        {
            final String brokerUrl = System.getProperty(HallelujahSystemProperties.BROKER_URL_KEY, jmsConfigurationFromFile.getBrokerUrl());
            final String queueId = System.getProperty(HallelujahSystemProperties.DEFAULT_QUEUE_KEY, jmsConfigurationFromFile.getTestNameQueue().replace("-name-queue", ""));
            final String clientId = System.getProperty(HallelujahSystemProperties.CLIENT_KEY, jmsConfigurationFromFile.getId());
            jmsConfiguration = JMSConfiguration.fromParams(brokerUrl, queueId, clientId);
        }
        return jmsConfiguration;
    }

    public static JMSConfiguration fromDefaultFile()
    {
        final File defaultConfigFile = new File(CONFIG_DIR, CONFIG_FILE);
        if (defaultConfigFile.exists())
        {
            return new JMSConfiguration(defaultConfigFile);
        }
        else
        {
            return null;
        }
    }

    public static String getBrokerUrlFromDefaultFile()
    {
        return JMSConfiguration.fromDefaultFile().getBrokerUrl();
    }

    public static JMSConfiguration fromParams(final String brokerUrl, final String testNameQueue, final String testResultQueue, final String id)
    {
        return new JMSConfiguration(brokerUrl, testNameQueue, testResultQueue, id);
    }

    public static JMSConfiguration fromParams(final String brokerUrl, final String queueId, final String clientId)
    {
        return new JMSConfiguration(brokerUrl, queueId, clientId);
    }

    public static JMSConfiguration fromSystemProperties()
    {
        final String brokerUrl = System.getProperty(BROKER_URL_KEY, DEFAULT_BROKER_URL);
        final String queueId = System.getProperty(QUEUE_KEY, DEFAULT_QUEUE_KEY);
        final String clientId = System.getProperty(CLIENT_KEY, DEFAULT_CLIENT_KEY_PREFIX + RandomStringUtils.randomAlphanumeric(RANDOM_CLIENT_ID_LENGTH));

        return JMSConfiguration.fromParams(brokerUrl, queueId, clientId);
    }

    public String getBrokerUrl()
    {
        return brokerUrl;
    }

    public String getTestNameQueue()
    {
        return testNameQueue;
    }

    public String getTestResultQueue()
    {
        return testResultQueue;
    }

    public String getControlTopic()
    {
        return controlTopic;
    }

    public String getId()
    {
        return id;
    }

    @Override
    public String toString() {
        return "JMSConfiguration{" +
                "preFetching=" + preFetching +
                ", brokerUrl='" + brokerUrl + '\'' +
                ", testNameQueue='" + testNameQueue + '\'' +
                ", testResultQueue='" + testResultQueue + '\'' +
                ", id='" + id + '\'' +
                '}';
    }
}
