define('bitbucket/internal/feature/pull-request/table/pull-request-table', ['module', 'exports', 'aui', 'aui/flag', 'jquery', 'lodash', 'bitbucket/internal/model/page-state', 'bitbucket/internal/model/pull-request', 'bitbucket/internal/util/events', 'bitbucket/internal/widget/paged-table/paged-table'], function (module, exports, _aui, _flag, _jquery, _lodash, _pageState, _pullRequest, _events, _pagedTable) {
    'use strict';

    Object.defineProperty(exports, "__esModule", {
        value: true
    });

    var _aui2 = babelHelpers.interopRequireDefault(_aui);

    var _flag2 = babelHelpers.interopRequireDefault(_flag);

    var _jquery2 = babelHelpers.interopRequireDefault(_jquery);

    var _lodash2 = babelHelpers.interopRequireDefault(_lodash);

    var _pageState2 = babelHelpers.interopRequireDefault(_pageState);

    var _pullRequest2 = babelHelpers.interopRequireDefault(_pullRequest);

    var _events2 = babelHelpers.interopRequireDefault(_events);

    var _pagedTable2 = babelHelpers.interopRequireDefault(_pagedTable);

    PullRequestTable.defaults = {
        allFetchedMessageHtml: '<p class="no-more-results">' + _aui2.default.I18n.getText('bitbucket.web.pullrequest.allfetched') + '</p>',
        alwaysDisplayRepositories: false,
        bufferPixels: 150,
        hideAuthorName: false,
        scope: 'repository',
        showStatus: false,
        target: '#pull-requests-table',
        container: '#pull-requests-table-container',
        tableMessageClass: 'pull-request-table-message',
        paginationContext: 'pull-request-table',
        pageSize: 25
    };

    /**
     * Create a paged table of pull requests filled (after the first page) based on the supplied parameters
     * @param {String} prState only show pull requests in this state (defaults to open)
     * @param {String} prOrder the order to show the pull requests in either 'newest' or 'oldest' first
     * @param {Function} getPullRequestsUrlBuilder the function to get the base URL to find pull requests in
     * @param {Object} options additional options including:
     *                  - alwaysDisplayRepositories When true, ref lozenges will always include repository information.
     *                    When false, the source lozenge will show repository information if it doesn't match the target repository.
     *                  - direction the direction of the pull requests you want to display ('outgoing' or 'incoming'), used with source
     *                  - hideAuthorName toggles the rendering of the author name in addition to their avatar (defaults to true)
     *                  - scope options: repository (show id) [default]; global (repo)
     *                  - showStatus toggles the rendering of the status column, the status column will contain OPEN, MERGED or DECLINED lozenges
     *                  - source the branch to show pull requests for
     *                  - initialData JSON object of initial table data
     *                  - PagedTable options
     */
    function PullRequestTable(prState, prOrder, getPullRequestsUrlBuilder, options) {
        this.prState = prState && prState.toUpperCase();
        this.prOrder = prOrder ? prOrder : this.prState === _pullRequest2.default.state.OPEN ? 'oldest' : 'newest';
        this.getPullRequestsUrlBuilder = getPullRequestsUrlBuilder;
        this.prDirection = options.direction;
        this.prSource = options.source;

        var dynamicDefaults = {
            noneFoundMessageHtml: bitbucket.internal.feature.pullRequest.pullRequestTableEmpty({
                state: this.prState
            })
        };
        options = _jquery2.default.extend(PullRequestTable.defaults, dynamicDefaults, options);

        this.alwaysDisplayRepositories = options.alwaysDisplayRepositories;
        this.hideAuthorName = options.hideAuthorName;
        this.scope = options.scope;
        this.showStatus = options.showStatus;

        // if we've been passed initialData then render the initial table
        // client side
        if (options.initialData) {
            (0, _jquery2.default)(options.container).replaceWith(bitbucket.internal.feature.pullRequest.pullRequestTable({
                pullRequestPage: options.initialData,
                alwaysDisplayRepositories: options.alwaysDisplayRepositories,
                currentUser: _pageState2.default.getCurrentUser().toJSON()
            }));
        }

        _pagedTable2.default.call(this, options);

        var toolTipArgs = {
            hoverable: false,
            offset: 5,
            delayIn: 0,
            gravity: function gravity() {
                // Always position on screen
                return _jquery2.default.fn.tipsy.autoNS.call(this) + _jquery2.default.fn.tipsy.autoWE.call(this);
            },
            live: true
        };

        (0, _jquery2.default)(this.options.target).find('.author > div, .author .aui-avatar-inner > img').tooltip(toolTipArgs);
    }

    _jquery2.default.extend(PullRequestTable.prototype, _pagedTable2.default.prototype);

    /**
     * Returns the URL used to retrieve pull requests to fill the table based on the criteria supplied at construction
     * @param {number} index to start the page of pull requests at.
     * @param {number} number of pull requests to retrieve in this page.
     */
    PullRequestTable.prototype.buildUrl = function (start, limit) {
        var self = this;
        var builder = self.getPullRequestsUrlBuilder().withParams({
            start: start,
            limit: limit,
            avatarSize: bitbucket.internal.widget.avatarSizeInPx({
                size: 'medium'
            }),
            withAttributes: true
        });

        if (self.prDirection) {
            builder = builder.withParams({
                direction: self.prDirection
            });
        }
        if (self.prSource) {
            builder = builder.withParams({
                at: self.prSource
            });
        }
        if (self.prState) {
            builder = builder.withParams({
                state: self.prState
            });
        }
        if (self.prOrder) {
            builder = builder.withParams({
                order: self.prOrder
            });
        }

        return builder.build();
    };

    PullRequestTable.prototype.focusInitialRow = function () {
        this.$table.find('tbody tr.pull-request-row:first').addClass('focused');
    };

    PullRequestTable.prototype.attachNewContent = function (data, attachmentMethod) {
        _pagedTable2.default.prototype.attachNewContent.call(this, data, attachmentMethod);

        _events2.default.trigger('bitbucket.internal.feature.pullRequestsTable.contentAdded', this, data);
    };

    PullRequestTable.prototype.handleNewRows = function (data, attachmentMethod) {
        var self = this;
        var rows = (0, _jquery2.default)(_lodash2.default.map(data.values, function (pr) {
            return bitbucket.internal.feature.pullRequest.pullRequestRow({
                alwaysDisplayRepositories: self.alwaysDisplayRepositories,
                currentUser: _pageState2.default.getCurrentUser().toJSON(),
                hideAuthorName: self.hideAuthorName,
                pullRequest: pr,
                scope: self.scope,
                showStatus: self.showStatus
            });
        }).join(''));
        this.$table.show().children('tbody')[attachmentMethod !== 'html' ? attachmentMethod : 'append'](rows);
    };

    PullRequestTable.prototype.initShortcuts = function () {
        var self = this;
        var sel = this.$table.selector;
        var openItemDisabled = false;
        var options = {
            focusedClass: 'focused',
            wrapAround: false,
            escToCancel: false
        };
        var focusedRowSelector = sel + ' .pull-request-row.' + options.focusedClass;
        var rowSelector = focusedRowSelector + ', ' + //Always include the currently selected element, even if it's a filtered row
        sel + ':not(.filter-current-user, .filter-current-user-unactioned) .pull-request-row, ' + //When not filtering pullrequests, include every row
        sel + '.filter-current-user .pull-request-row.current-user, ' + //When filtering pullrequests by user, don't include non-reviewer rows
        sel + '.filter-current-user-unactioned .pull-request-row.current-user:not(.current-user-actioned)'; //When filtering pullrequests by user unactioned, don't include rows user has approved

        _events2.default.on('bitbucket.internal.keyboard.shortcuts.disableOpenItemHandler', function () {
            openItemDisabled = true;
        });
        _events2.default.on('bitbucket.internal.keyboard.shortcuts.enableOpenItemHandler', function () {
            openItemDisabled = false;
        });

        this.bindMoveToNextHandler = function (keys) {
            (this.moveToNextItem ? this : _aui2.default.whenIType(keys)).moveToNextItem(rowSelector, options).execute(function () {
                if ((0, _jquery2.default)(rowSelector).last().hasClass(options.focusedClass)) {
                    window.scrollTo(0, document.documentElement.scrollHeight);
                }
            });
        };

        this.bindMoveToPreviousHandler = function (keys) {
            (this.moveToPrevItem ? this : _aui2.default.whenIType(keys)).moveToPrevItem(rowSelector, options);
        };

        this.bindOpenItemHandler = function (keys) {
            (this.execute ? this : _aui2.default.whenIType(keys)).execute(function () {
                if (!openItemDisabled) {
                    var $focusedItem = jQuery(focusedRowSelector);
                    if ($focusedItem.length) {
                        window.location.href = $focusedItem.find('td.id a').attr('href');
                    }
                }
            });
        };

        this.bindHighlightAssignedHandler = function (keys) {
            (this.execute ? this : _aui2.default.whenIType(keys)).execute(function () {
                rotateCurrentFilter(self.$table);
            });
        };

        var flag;
        function rotateCurrentFilter($el) {
            var filters = [{
                name: 'filter-current-user',
                description: _aui2.default.I18n.getText('bitbucket.web.pullrequest.filter-current-user')
            }, {
                name: 'filter-current-user-unactioned',
                description: _aui2.default.I18n.getText('bitbucket.web.pullrequest.filter-current-user-unactioned')
            }, {
                name: '',
                description: _aui2.default.I18n.getText('bitbucket.web.pullrequest.filter-all')
            }];
            var lastFilterIndex = filters.length - 1;

            _lodash2.default.some(filters, function (filter, index) {
                if ($el.hasClass(filter.name) || index === lastFilterIndex) {
                    var newFilter = index < lastFilterIndex ? filters[index + 1] : filters[0];

                    $el.removeClass(filter.name).addClass(newFilter.name);
                    if (flag) {
                        flag.close();
                    }
                    flag = (0, _flag2.default)({
                        type: 'info',
                        title: newFilter.description,
                        close: 'auto'
                    });
                    return true;
                }
                return false;
            });
        }

        _pagedTable2.default.prototype.initShortcuts.call(this);
    };

    exports.default = PullRequestTable;
    module.exports = exports['default'];
});