/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.stash.internal.auth;

import com.atlassian.bitbucket.auth.RememberMeMode;
import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.security.random.SecureTokenGenerator;
import com.atlassian.stash.internal.InternalConverter;
import com.atlassian.stash.internal.auth.InternalRememberMeToken;
import com.atlassian.stash.internal.auth.RememberMeService;
import com.atlassian.stash.internal.auth.RememberMeTokenDao;
import com.atlassian.stash.internal.spring.SpringTransactionUtils;
import com.atlassian.stash.internal.web.HttpRequestInfoHelper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.primitives.Ints;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.HibernateException;
import org.joda.time.DateTimeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.dao.DataAccessException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

@Service(value="rememberMeService")
@Transactional
public class DefaultRememberMeService
implements RememberMeService {
    private static final Logger log = LoggerFactory.getLogger(DefaultRememberMeService.class);
    static final int MAX_ATTEMPTS = 5;
    static final String PARAMETER_REMEMBER_ME = "_atl_remember_me";
    private final String cookieName;
    private final RememberMeTokenDao dao;
    private final long expirySeconds;
    private final long gracePeriodSeconds;
    private final RememberMeMode rememberMeMode;
    private final HttpRequestInfoHelper requestInfoHelper;
    private final SecureTokenGenerator tokenGenerator;
    private final TransactionTemplate transactionTemplate;

    @Autowired
    public DefaultRememberMeService(@Value(value="${auth.remember-me.cookie.name}") String cookieName, RememberMeTokenDao dao, @Value(value="${auth.remember-me.token.expiry}") int expiryDays, @Value(value="${auth.remember-me.token.grace.period}") long gracePeriodSeconds, @Value(value="${auth.remember-me.enabled}") String rememberMeMode, HttpRequestInfoHelper requestInfoHelper, SecureTokenGenerator tokenGenerator, PlatformTransactionManager transactionManager) {
        this.cookieName = cookieName;
        this.dao = dao;
        this.expirySeconds = TimeUnit.DAYS.toSeconds(expiryDays);
        this.gracePeriodSeconds = gracePeriodSeconds;
        this.rememberMeMode = RememberMeMode.fromId((String)rememberMeMode);
        this.requestInfoHelper = requestInfoHelper;
        this.tokenGenerator = tokenGenerator;
        this.transactionTemplate = new TransactionTemplate(transactionManager, SpringTransactionUtils.REQUIRES_NEW);
    }

    @Nullable
    public ApplicationUser authenticate(@Nonnull HttpServletRequest request, @Nonnull HttpServletResponse response) {
        Cookie cookie = this.getCookie(request);
        if (cookie == null) {
            log.trace("Remember-me cookie not found, skipping remember-me authentication");
            return null;
        }
        InternalRememberMeToken token = this.toToken(cookie);
        if (token == null) {
            log.info("Invalid remember-me cookie detected (wrong format: '{}') - canceling the cookie", (Object)cookie.getValue());
            this.setCookie(request, response, null);
            return null;
        }
        if (this.rememberMeMode == RememberMeMode.NEVER) {
            log.trace("Remember-me has been disabled, skipping remember-me authentication");
            return null;
        }
        InternalRememberMeToken validToken = this.claimToken(token, this.getRemoteAddress(request));
        if (validToken == null) {
            log.info("Invalid remember-me cookie detected (expired) - canceling the cookie");
            this.setCookie(request, response, null);
            return null;
        }
        if (!validToken.getToken().equals(token.getToken())) {
            this.setCookie(request, response, validToken);
        }
        log.trace("Remember-me authentication succeeded for series '{}' and user '{}'", (Object)validToken.getSeries(), (Object)this.getUsername(validToken));
        return validToken.getUser();
    }

    @Transactional(propagation=Propagation.SUPPORTS)
    public void cancelCookie(@Nonnull HttpServletRequest request, @Nonnull HttpServletResponse response) {
        Cookie cookie = this.getCookie(request);
        if (cookie != null) {
            if (log.isTraceEnabled()) {
                InternalRememberMeToken token = this.toToken(cookie);
                if (token == null) {
                    log.trace("Canceling invalid remember-me cookie");
                } else {
                    log.trace("Canceling remember-me cookie for series '{}'", (Object)token.getSeries());
                }
            }
            this.setCookie(request, response, null);
        }
    }

    public void createCookie(@Nonnull ApplicationUser user, @Nonnull HttpServletRequest request, @Nonnull HttpServletResponse response) {
        Cookie cookie = this.getCookie(request);
        if (cookie != null) {
            this.cancelCookie(request, response);
            InternalRememberMeToken token = this.toToken(cookie);
            throw new IllegalStateException("A remember-me cookie for series '+" + (token != null ? token.getSeries() : "invalid") + "' is already present. Cannot provide a remember-me cookie for a new series. " + "Canceling the existing cookie");
        }
        InternalRememberMeToken token = (InternalRememberMeToken)this.dao.create((Object)new InternalRememberMeToken.Builder().series(this.tokenGenerator.generateToken()).token(this.tokenGenerator.generateToken()).user(InternalConverter.convertToInternalUser((ApplicationUser)user)).expiresAfter(this.expirySeconds, TimeUnit.SECONDS).build());
        this.setCookie(request, response, token);
        log.debug("Created new remember-me series '{}' for user '{}'", (Object)token.getSeries(), (Object)user.getName());
    }

    public void deleteExpiredTokens(long maxExpiredTime, @Nonnull TimeUnit timeUnit) {
        Preconditions.checkArgument((maxExpiredTime >= 0L ? 1 : 0) != 0, (Object)"Negative expiry times are not allowed");
        int count = this.dao.deleteExpiredTokens(maxExpiredTime, timeUnit);
        log.debug("Cleaned up {} expired remember-me tokens", (Object)count);
    }

    public void deleteForUser(@Nonnull ApplicationUser user) {
        this.dao.deleteForUser(((ApplicationUser)Preconditions.checkNotNull((Object)user, (Object)"user")).getId());
    }

    @Transactional(propagation=Propagation.SUPPORTS)
    public boolean isCookieRequested(@Nonnull HttpServletRequest request) {
        String uri = StringUtils.substringBefore((String)request.getRequestURI(), (String)";");
        if (!uri.endsWith("j_atl_security_check")) {
            return false;
        }
        Cookie cookie = this.getCookie(request);
        if (cookie != null) {
            log.debug("Not providing a remember-me cookie because a cookie was already present on the request");
            return false;
        }
        switch (this.rememberMeMode) {
            case ALWAYS: {
                return true;
            }
            case NEVER: {
                return false;
            }
        }
        String rememberMe = request.getParameter(PARAMETER_REMEMBER_ME);
        return "on".equals(rememberMe) || "checked".equals(rememberMe) || "1".equals(rememberMe) || "true".equalsIgnoreCase(rememberMe);
    }

    public void logout(@Nonnull HttpServletRequest request, @Nonnull HttpServletResponse response) {
        InternalRememberMeToken token;
        Cookie cookie = this.getCookie(request);
        if (cookie != null && (token = this.toToken(cookie)) != null) {
            log.trace("Canceling remember-me series '{}' on logout", (Object)token.getSeries());
            this.dao.deleteForSeries(token.getSeries());
            this.cancelCookie(request, response);
        }
    }

    @VisibleForTesting
    protected String[] decodeCookieValue(String cookieValue) {
        String base64 = cookieValue + StringUtils.repeat((char)'=', (int)(cookieValue.length() % 4));
        if (!Base64.isBase64((String)base64)) {
            return null;
        }
        String decoded = new String(Base64.decodeBase64((byte[])base64.getBytes()));
        return StringUtils.split((String)decoded, (char)':');
    }

    @VisibleForTesting
    protected String encodeCookie(String ... cookieTokens) {
        String joined = StringUtils.join((Object[])cookieTokens, (String)":");
        String encoded = new String(Base64.encodeBase64((byte[])joined.getBytes()));
        return StringUtils.stripEnd((String)encoded, (String)"=");
    }

    private InternalRememberMeToken claimToken(InternalRememberMeToken token, String remoteAddress) {
        int i = 0;
        while (i < 5) {
            try {
                return (InternalRememberMeToken)this.transactionTemplate.execute(status -> this.doClaimToken(token, remoteAddress));
            }
            catch (HibernateException | DataAccessException e) {
                if (++i < 5) {
                    log.debug("Could not claim token for series {}: '{}'. Retrying", (Object)token.getSeries(), (Object)e.getMessage());
                    continue;
                }
                log.error("Could not claim token for series {} after {} attempts", new Object[]{token.getSeries(), i, e});
            }
        }
        return null;
    }

    private InternalRememberMeToken doClaimToken(InternalRememberMeToken token, String remoteAddress) {
        InternalRememberMeToken current = this.dao.findBySeriesAndToken(token.getSeries(), token.getToken());
        if (current == null) {
            return null;
        }
        if (current.isClaimed()) {
            return this.handleClaimedToken(current, remoteAddress);
        }
        String username = this.getUsername(current);
        if (current.isExpired()) {
            log.debug("Provided token for remember-me series '{}' for user '{}' has expired. Ignoring the token", (Object)token.getSeries(), (Object)username);
            return null;
        }
        if (this.dao.claim(new InternalRememberMeToken.Builder(current).claimed().expiresAfter(this.gracePeriodSeconds, TimeUnit.SECONDS).remoteAddress(remoteAddress).build())) {
            log.debug("Refreshed token for remember-me series '{}' for user '{}'", (Object)token.getSeries(), (Object)username);
            return (InternalRememberMeToken)this.dao.create((Object)((InternalRememberMeToken.Builder)new InternalRememberMeToken.Builder(current).id(0L)).token(this.tokenGenerator.generateToken()).expiresAfter(this.expirySeconds, TimeUnit.SECONDS).build());
        }
        this.dao.refresh((Object)current);
        if (current.isClaimed()) {
            return this.handleClaimedToken(current, remoteAddress);
        }
        return null;
    }

    private Cookie getCookie(HttpServletRequest request) {
        Cookie[] cookies = request.getCookies();
        if (cookies == null) {
            return null;
        }
        for (Cookie cookie : cookies) {
            if (!this.cookieName.equals(cookie.getName())) continue;
            return cookie;
        }
        return null;
    }

    private String getRemoteAddress(HttpServletRequest request) {
        String remoteAddress = this.requestInfoHelper.getRemoteAddress(request);
        return remoteAddress.length() < 256 ? remoteAddress : remoteAddress.substring(0, 255);
    }

    private String getUsername(InternalRememberMeToken token) {
        return StringUtils.defaultString((String)(token.getUser() == null ? null : token.getUser().getName()), (String)"<unknown>");
    }

    private InternalRememberMeToken handleClaimedToken(InternalRememberMeToken token, String remoteAddress) {
        boolean remoteAddressMatch = Objects.equals(remoteAddress, token.getRemoteAddress());
        if (remoteAddressMatch && !token.isExpired()) {
            log.trace("Accepting remember-me token for series '{}' for user '{}' that was already used for authentication from '{}'", new Object[]{token.getSeries(), this.getUsername(token), token.getRemoteAddress()});
            return token;
        }
        int deletedTokens = this.dao.deleteForSeries(token.getSeries());
        if (deletedTokens > 0) {
            if (remoteAddressMatch) {
                log.info("Expired remember-me token detected for series '{}' for user '{}' (used from '{}'). As a safety precaution, all ({}) tokens from that series have been canceled.", new Object[]{token.getSeries(), this.getUsername(token), token.getRemoteAddress(), deletedTokens});
            } else {
                log.info("Remember-me token detected for series '{}' for user '{}' (used from '{}'). Current request is from a different address ('{}')! As a safety precaution, all ({}) tokens from that series have been canceled.", new Object[]{token.getSeries(), this.getUsername(token), token.getRemoteAddress(), remoteAddress, deletedTokens});
            }
        }
        return null;
    }

    @VisibleForTesting
    void setCookie(HttpServletRequest request, HttpServletResponse response, InternalRememberMeToken token) {
        String cookieValue = null;
        if (token != null) {
            cookieValue = this.encodeCookie(token.getSeries(), token.getToken());
        }
        Cookie cookie = new Cookie(this.cookieName, cookieValue);
        cookie.setHttpOnly(true);
        cookie.setSecure(request.isSecure());
        String path = request.getContextPath();
        cookie.setPath(StringUtils.isNotBlank((CharSequence)path) ? path : "/");
        long maxAge = token == null ? 0L : TimeUnit.MILLISECONDS.toSeconds(token.getExpiryDate().getTime() - DateTimeUtils.currentTimeMillis());
        cookie.setMaxAge(Ints.saturatedCast((long)maxAge));
        if (maxAge < 1L) {
            cookie.setVersion(1);
        }
        response.addCookie(cookie);
    }

    private InternalRememberMeToken toToken(Cookie cookie) {
        String[] tokens = this.decodeCookieValue(cookie.getValue());
        if (tokens == null || tokens.length != 2) {
            return null;
        }
        return new InternalRememberMeToken.Builder().series(tokens[0]).token(tokens[1]).build();
    }
}

