package com.atlassian.bamboo.plugins.tomcat.upgrade;

import com.atlassian.bamboo.build.BuildDefinition;
import com.atlassian.bamboo.build.BuildDefinitionManager;
import com.atlassian.bamboo.build.Job;
import com.atlassian.bamboo.plan.PlanManager;
import com.atlassian.bamboo.plan.TopLevelPlan;
import com.atlassian.bamboo.plugins.tomcat.configuration.AbstractTomcatConfigurator;
import com.atlassian.bamboo.security.EncryptionService;
import com.atlassian.bamboo.task.TaskDefinition;
import com.atlassian.sal.api.message.Message;
import com.atlassian.sal.api.upgrade.PluginUpgradeTask;
import com.google.common.collect.Lists;
import org.apache.commons.lang.StringUtils;

import java.util.Collection;
import java.util.List;
import java.util.Map;

public class PasswordUpgradeTask implements PluginUpgradeTask
{
    private final PlanManager planManager;
    private final BuildDefinitionManager buildDefinitionManager;
    private final EncryptionService encryptionService;

    public static final String TOMCAT_PLUGIN = "com.atlassian.bamboo.plugins.tomcat.bamboo-tomcat-plugin";

    public PasswordUpgradeTask(PlanManager planManager,
                               BuildDefinitionManager buildDefinitionManager,
                               EncryptionService encryptionService)
    {
        this.planManager = planManager;
        this.buildDefinitionManager = buildDefinitionManager;
        this.encryptionService = encryptionService;
    }

    @Override
    public int getBuildNumber()
    {
        return 11;
    }

    @Override
    public String getShortDescription()
    {
        return "Encrypt stored passwords for Tomcat server configuration";
    }

    public Collection<Message> doUpgrade() throws Exception
    {
        Collection<Message> messages = Lists.newArrayList();
        // no authentication context available when upgrade task is executed, so unrestricted call
        List<TopLevelPlan> plans = planManager.getAllPlansUnrestricted();
        for (TopLevelPlan plan : plans)
        {
            for (Job job : plan.getAllJobs())
            {
                BuildDefinition buildDefinition = job.getBuildDefinition();
                for (TaskDefinition taskDefinition : buildDefinition.getTaskDefinitions())
                {
                    encryptPassword(taskDefinition, "deployAppTask", messages);
                    encryptPassword(taskDefinition, "startAppTask", messages);
                    encryptPassword(taskDefinition, "stopAppTask", messages);
                    encryptPassword(taskDefinition, "reloadAppTask", messages);
                    encryptPassword(taskDefinition, "undeployAppTask", messages);
                }
                buildDefinitionManager.savePlanAndDefinition(job, buildDefinition);
            }
        }
        return messages;
    }

    private void encryptPassword(final TaskDefinition taskDefinition, final String taskKey, final Collection<Message> messages)
    {
        Map<String, String> config = taskDefinition.getConfiguration();
        if (taskDefinition.getPluginKey().equals(TOMCAT_PLUGIN + ":" + taskKey))
        {
            String password = config.get(AbstractTomcatConfigurator.TOMCAT_PLAIN_PASSWORD);
            String encryptedPassword = config.get(AbstractTomcatConfigurator.TOMCAT_PASSWORD);

            if (StringUtils.isNotEmpty(password) && StringUtils.isEmpty(encryptedPassword))
            {
                String newPassword = encryptionService.encrypt(password);
                config.put(AbstractTomcatConfigurator.TOMCAT_PASSWORD, newPassword);
                config.remove(AbstractTomcatConfigurator.TOMCAT_PLAIN_PASSWORD);
                taskDefinition.setConfiguration(config);
            }
        }
    }

    @Override
    public String getPluginKey()
    {
        return TOMCAT_PLUGIN;
    }
}
